/*:
 * @target MZ
 * @plugindesc v1.31 ダメージ計算の最後に自由式を確実適用（回復除外・敵蘇生防止・ログ修正・タイミング未指定は1回だけ） 
 * @author You
 * @help DamageAppendFormulaMZ.js
 *
 * 【概要】
 * - 2段階の挿入点:
 *   1) AfterFormula … 公式ダメージ式の直後（属性/分散/防御等の計算済みの値）
 *   2) AfterAll     … すべての処理の本当の最後（executeDamage直前）
 * - スキル/アイテムのメモ欄:
 *     <DamageAppend: 式>
 *     <DamageAppendTiming: AfterFormula>
 *     <DamageAppendTiming: AfterAll>
 * - グローバル式（プラグインパラメータ）と併用可
 * - 追加式の結果は必ず「四捨五入した整数」に丸められます。
 * - HPダメージが正のときだけ追加式を適用します（回復・0ダメージにはかからない）。
 * - 敵は戦闘不能状態での回復ダメージ（負の値）では復活しません。
 *
 * 【式で使える識別子】
 * - damage : 現在のダメージ値（HP→正: ダメージ, 負: 回復）
 * - a      : 行動者（バトラー）
 * - b      : 対象（バトラー）
 * - v[n]   : 変数 n 番
 * - item   : 現在のスキル/アイテム
 *
 * 【タイミング仕様】
 * - メモ欄に <DamageAppend: ...> だけ書いた場合（Timing 指定なし）:
 *     → AfterFormula のときだけ適用（1回だけ）
 * - メモ欄に <DamageAppendTiming: AfterFormula> を書いた場合:
 *     → AfterFormula のときだけ適用
 * - メモ欄に <DamageAppendTiming: AfterAll> を書いた場合:
 *     → AfterAll のときだけ適用
 *
 * 【例】
 *  すべての攻撃ダメージに LUK 差分を追加したい場合:
 *    グローバル式 (AfterAll) に
 *      damage + (a.luk * 4 - b.luk * 2) * 1.3
 *    と書く。
 *
 * @param GlobalFormulaAfterFormula
 * @text グローバル式 (AfterFormula)
 * @desc 全スキル共通で「ダメージ計算直後」に適用する式。空欄なら無効。
 * @type string
 * @default
 *
 * @param GlobalFormulaAfterAll
 * @text グローバル式 (AfterAll)
 * @desc 全スキル共通で「ダメージ適用直前」に適用する式。空欄なら無効。
 * @type string
 * @default
 *
 * @param ClampMin
 * @text 最小値 (Clamp)
 * @desc 最終ダメージの最小値（例: 0にすればHPが増える系を防止）。空欄で無制限。
 * @type number
 * @min -999999
 * @default
 *
 * @param ClampMax
 * @text 最大値 (Clamp)
 * @desc 最終ダメージの最大値（例: 9999）。空欄で無制限。
 * @type number
 * @min -999999
 * @default
 *
 * @param DebugLog
 * @text デバッグログ出力
 * @desc F8コンソールに計算経過を表示します。
 * @type boolean
 * @default false
 */

(() => {
  "use strict";

  //-----------------------------------------------------------------------------
  // プラグインパラメータの取得
  //-----------------------------------------------------------------------------
  const PLUGIN_NAME = "DamageAppendFormulaMZ";
  const parameters = PluginManager.parameters(PLUGIN_NAME);

  const GLOBAL_FORMULA_AFTER_FORMULA = String(parameters["GlobalFormulaAfterFormula"] || "");
  const GLOBAL_FORMULA_AFTER_ALL = String(parameters["GlobalFormulaAfterAll"] || "");

  const CLAMP_MIN = parameters["ClampMin"] !== "" ? Number(parameters["ClampMin"]) : null;
  const CLAMP_MAX = parameters["ClampMax"] !== "" ? Number(parameters["ClampMax"]) : null;

  const DEBUG_LOG = String(parameters["DebugLog"] || "false") === "true";

  //-----------------------------------------------------------------------------
  // ヘルパー: メモ欄からタグを取得
  //-----------------------------------------------------------------------------
  function getMetaTag(item, tagName) {
    if (!item || !item.meta) return undefined;
    return item.meta[tagName];
  }

  //-----------------------------------------------------------------------------
  // ヘルパー: v[n] proxy
  //-----------------------------------------------------------------------------
  const v = new Proxy(
    {},
    {
      get(_target, prop) {
        const n = Number(prop);
        if (Number.isNaN(n)) return 0;
        return $gameVariables.value(n);
      },
      set(_target, prop, value) {
        const n = Number(prop);
        if (!Number.isNaN(n)) {
          $gameVariables.setValue(n, value);
        }
        return true;
      },
    }
  );

  //-----------------------------------------------------------------------------
  // 追加式の評価本体
  //-----------------------------------------------------------------------------
  function evalDamageAppendFormula(baseDamage, action, target, timing) {
    const item = action.item();
    const a = action.subject();
    const b = target;

    // ダメージが正のときだけ（回復や0には一切触れない）
    if (baseDamage <= 0) {
      return baseDamage;
    }

    // スキル/アイテムのメモ欄優先
    let formula = null;
    if (item) {
      const timingTag = getMetaTag(item, "DamageAppendTiming");
      const appendTag = getMetaTag(item, "DamageAppend");

      if (appendTag) {
        if (timingTag) {
          // タイミングタグがある場合：指定タイミングと一致したときのみ適用
          if (timingTag === timing) {
            formula = String(appendTag);
          }
        } else {
          // タイミングタグが無い場合：デフォルトで AfterFormula のみ適用（1回だけ）
          if (timing === "AfterFormula") {
            formula = String(appendTag);
          }
        }
      }
    }

    // メモ欄で指定がなければグローバル式を使用
    if (!formula) {
      if (timing === "AfterFormula") {
        formula = GLOBAL_FORMULA_AFTER_FORMULA;
      } else if (timing === "AfterAll") {
        formula = GLOBAL_FORMULA_AFTER_ALL;
      }
    }

    if (!formula || !formula.trim()) {
      return baseDamage;
    }

    const damage = baseDamage;
    const trimmed = formula.trim();

    try {
      /* eslint no-eval: 0 */
      const result = eval(trimmed);
      if (typeof result === "number" && Number.isFinite(result)) {
        return result;
      } else {
        if (DEBUG_LOG) {
          console.warn(
            "[DamageAppend] 式の結果が数値ではありません:",
            trimmed,
            "->",
            result
          );
        }
        return baseDamage;
      }
    } catch (e) {
      console.error("[DamageAppend] 式の評価に失敗しました:", trimmed);
      console.error(e);
      return baseDamage;
    }
  }

  //-----------------------------------------------------------------------------
  // 適用ラッパー
  //-----------------------------------------------------------------------------
  function applyAppend(baseDamage, action, target, timing) {
    // 念のためここでも回復・0を除外
    if (baseDamage <= 0) return baseDamage;
    return evalDamageAppendFormula(baseDamage, action, target, timing);
  }

  //-----------------------------------------------------------------------------
  // Hook: makeDamageValue (AfterFormula)
  //-----------------------------------------------------------------------------
  const _makeDamageValue = Game_Action.prototype.makeDamageValue;
  Game_Action.prototype.makeDamageValue = function (target, critical) {
    // 公式計算（属性、分散、防御含む）
    let value = _makeDamageValue.call(this, target, critical);
    const before = value;

    // AfterFormula 適用
    value = applyAppend(value, this, target, "AfterFormula");

    if (!Number.isFinite(value)) value = 0;
    value = Math.round(value);

    if (DEBUG_LOG && before !== value) {
      const sName = this.subject() ? this.subject().name() : "???";
      console.log(
        `[DamageAppend] (AfterFormula) ${sName} → ${target.name()} : ${before} → ${value}`
      );
    }

    return value;
  };

  //-----------------------------------------------------------------------------
  // Hook: executeDamage (AfterAll & 蘇生防止)
  //-----------------------------------------------------------------------------
  const _executeDamage = Game_Action.prototype.executeDamage;
  Game_Action.prototype.executeDamage = function (target, value) {
    const subject = this.subject();
    let adjusted = value;

    // 1. AfterAll 適用
    const before = adjusted;
    adjusted = applyAppend(adjusted, this, target, "AfterAll");

    // 2. Clamp
    if (CLAMP_MIN !== null && adjusted < CLAMP_MIN) adjusted = CLAMP_MIN;
    if (CLAMP_MAX !== null && adjusted > CLAMP_MAX) adjusted = CLAMP_MAX;

    // 3. 整数化
    if (!Number.isFinite(adjusted)) adjusted = 0;
    adjusted = Math.round(adjusted);

    // 4. 敵蘇生防止ロジック
    // 「敵」であり、かつ「現在戦闘不能」であり、かつ「今回のダメージが回復(負)」である場合
    // 回復量を0にして蘇生を防ぐ
    if (target.isEnemy() && target.isDead() && adjusted < 0) {
      if (DEBUG_LOG) {
        console.log(
          `[DamageAppend] Enemy revival prevented for ${target.name()}`
        );
      }
      adjusted = 0;
    }

    if (DEBUG_LOG && before !== adjusted) {
      console.log(
        `[DamageAppend] (AfterAll) ${subject.name()} → ${target.name()} : ${before} → ${adjusted}`
      );
    }

    // 5. 元の処理を実行（ここで実際にHPが変動する）
    _executeDamage.call(this, target, adjusted);
  };
})();
