// DrawGaugeMaxValue_smart.js
// Safer variant to avoid UI overlap in compact windows (e.g., menu).
// Based on Awaya_Matana's DrawGaugeMaxValue.js (MIT).
// Changes:
//  - Do NOT change gaugeX/gaugeHeight/label font (keep engine defaults).
//  - Only draw "current/max" when it fits. Optionally shrink to minFontSize.
//  - Scene filter: enable in Status and Battle by default; disabled in Menu.
//
// MIT License
/*:
 * @target MZ
 * @plugindesc [安全版] ゲージに「現在/最大」を表示（はみ出し防止・シーン別ON/OFF）
 * @author あわやまたな + ChatGPT
 * @url https://awaya3ji.seesaa.net/
 *
 * @help
 * コンパクトなウィンドウ（メニューなど）で他UIに重なる問題を回避します。
 *  - エンジン既定のゲージ寸法やラベルフォントは変更しません。
 *  - 文字幅がゲージに収まらない場合は、描画をやめるか最小フォントまで縮小します。
 *  - デフォルトでは「ステータス画面と戦闘のみ」で有効、メニューでは無効です。
 *
 * @param digits
 * @text 桁（最大の想定桁数）
 * @type number
 * @default 8
 * @desc 最大値の想定桁数。確保幅の見積もりに使用します。
 *
 * @param minWidth
 * @text 表示に必要な最小幅
 * @type number
 * @default 120
 * @desc ゲージ幅がこの値未満なら「/最大」を描きません。
 *
 * @param shrinkToFit
 * @text はみ出すとき縮小
 * @type boolean
 * @default true
 * @desc 文字が収まらないとき、最小フォントまで段階的に縮小します。
 *
 * @param minFontSize
 * @text 最小フォントサイズ
 * @type number
 * @default 14
 * @desc 縮小するときの下限フォントサイズ。
 *
 * @param enableInStatus
 * @text ステータス画面で有効
 * @type boolean
 * @default true
 *
 * @param enableInBattle
 * @text 戦闘中に有効
 * @type boolean
 * @default true
 *
 * @param enableInMenu
 * @text メニューで有効（推奨:OFF）
 * @type boolean
 * @default false
 *
 * @param types
 * @text 対象タイプ
 * @type string[]
 * @default ["hp","mp","tp"]
 * @desc 「hp」「mp」「tp」など。ここに含まれるタイプのみ適用。
 */
(() => {
  "use strict";
  const pluginName = document.currentScript.src.match(/^.*\/(.*).js$/)[1];
  const P = PluginManager.parameters(pluginName);
  const DIGITS = Math.max(1, Number(P.digits || 8));
  const MIN_WIDTH = Number(P.minWidth || 120);
  const SHRINK = String(P.shrinkToFit||"true")==="true";
  const MIN_FONT = Number(P.minFontSize || 14);
  const EN_STATUS = String(P.enableInStatus||"true")==="true";
  const EN_BATTLE = String(P.enableInBattle||"true")==="true";
  const EN_MENU   = String(P.enableInMenu||"false")==="true";
  const TYPES = new Set(JSON.parse(P.types || "[]"));

  const mayApplyHere = function(sprite) {
    const scene = SceneManager._scene;
    if (!scene) return true;
    if (scene instanceof Scene_Battle) return EN_BATTLE;
    if (scene instanceof Scene_Status) return EN_STATUS;
    if (scene instanceof Scene_Menu) return EN_MENU;
    // その他のシーンは安全のためOFF
    return false;
  };

  const padZero = "".padStart(DIGITS, "0");

  // safe measure utility
  const measure = (bmp, text, size) => {
    const keep = bmp.fontSize;
    bmp.fontSize = size;
    const w = bmp.measureTextWidth(text);
    bmp.fontSize = keep;
    return w;
  };

  const _drawValue = Sprite_Gauge.prototype.drawValue;
  Sprite_Gauge.prototype.drawValue = function() {
    if (!TYPES.has(this._statusType) || !mayApplyHere(this)) {
      return _drawValue.call(this);
    }
    const w = this.bitmapWidth();
    if (w < MIN_WIDTH) return _drawValue.call(this);

    const cur = String(this.currentValue());
    const max = String(this.currentMaxValue());
    const slash = "/";

    // base font (engine default for value)
    this.setupValueFont();
    const base = this.bitmap.fontSize;
    let size = base;

    // required width estimate using reserved max digits
    let need = measure(this.bitmap, cur, size)
             + measure(this.bitmap, slash, size)
             + measure(this.bitmap, padZero, size);

    if (need > w) {
      if (!SHRINK) return _drawValue.call(this);
      while (need > w && size > MIN_FONT) {
        size -= 1;
        need = measure(this.bitmap, cur, size)
             + measure(this.bitmap, slash, size)
             + measure(this.bitmap, padZero, size);
      }
      if (need > w) return _drawValue.call(this); // まだ無理なら諦める
    }

    // draw right-aligned without touching label/gauge dimensions
    const h = this.textHeight();
    const x = Math.max(0, w - need);
    const curW = measure(this.bitmap, cur, size);
    const slashW = measure(this.bitmap, slash, size);
    const maxW = Math.min(w, need - curW - slashW);

    const b = this.bitmap;
    b.fontSize = size;
    b.drawText(cur,   x,                 0, curW,   h, "left");
    b.drawText(slash, x + curW,          0, slashW, h, "left");
    b.drawText(max,   x + curW + slashW, 0, maxW,   h, "left");
  };

  // 注意：この安全版は gaugeX/gaugeHeight/labelFontSize を変更しません。
})();