//=============================================================================
// EndRoll_Kumoten.js
//=============================================================================
/*:
 * @target MZ
 * @plugindesc v1.3 エンドロール専用の縦スクロールクレジット（ボタンで早送り）を表示します。
 * @author 曇天 + ChatGPT
 *
 * @help
 * ■ 概要
 * プラグインコマンドから専用シーンに切り替え、
 * 画面全体に縦スクロールのエンドロールを表示します。
 * Window_ScrollText は一切使用しません。
 *
 * ■ 使い方
 * 1. プラグインパラメータ「デフォルトテキスト」にエンドロール本文を
 *    そのまま貼り付けます（複数行OK）。
 *
 * 2. エンドロールを流したいイベントで
 *    「プラグインコマンド → EndRoll_Kumoten →
 *      エンドロール開始（デフォルト）」を実行します。
 *
 * 3. 任意テキストをその場で使いたい場合は、
 *    「エンドロール開始（任意テキスト）」コマンドを使い、
 *    Text 引数に本文を記入してください。
 *
 * ・テキストの改行は、そのまま1行として扱われます。
 * ・\C[1] 等の制御文字も通常のメッセージ同様に使用可能です。
 * ・「ボタンでスキップを許可」が ON の場合、
 *   OK / キャンセル / クリックを押している間だけ早送りになります。
 *   （※押した瞬間に終了はしません）
 *
 * ■ 終了
 * 最後まで流れたあと、少し待ってから元のシーン（マップ等）に戻ります。
 *
 * @param DefaultText
 * @text デフォルトテキスト
 * @type multiline_string
 * @default
 * @desc 「エンドロール開始（デフォルト）」で表示されるテキスト。
 *
 * @param ScrollSpeed
 * @text スクロール速度
 * @type number
 * @min 1
 * @max 20
 * @default 2
 * @desc 1でゆっくり、数値を上げるほど速く上に流れます。
 *
 * @param WaitTop
 * @text 開始前の待ちフレーム
 * @type number
 * @default 30
 * @desc 画面切替後、スクロールを開始するまでの待ち時間（フレーム）。
 *
 * @param WaitBottom
 * @text 終了前の待ちフレーム
 * @type number
 * @default 60
 * @desc 一番下までスクロール後、シーンを閉じるまでの待ち時間（フレーム）。
 *
 * @param AllowSkip
 * @text ボタンで早送りを許可
 * @type boolean
 * @default true
 * @desc OK / キャンセル / クリックで早送りを許可するかどうか。
 *
 * @param FastRate
 * @text 早送り倍率
 * @type number
 * @min 2
 * @max 20
 * @default 4
 * @desc 早送り時の速度倍率（通常速度 × この値）。
 *
 * @param TextAlign
 * @text テキスト揃え
 * @type select
 * @option 左揃え
 * @value left
 * @option 中央揃え
 * @value center
 * @option 右揃え
 * @value right
 * @default left
 * @desc エンドロールの本文を左／中央／右のどれで揃えるか。
 *
 * @command StartFromParam
 * @text エンドロール開始（デフォルト）
 * @desc パラメータ「デフォルトテキスト」でエンドロールを開始します。
 *
 * @command StartWithText
 * @text エンドロール開始（任意テキスト）
 * @desc 任意指定したテキストでエンドロールを開始します。
 *
 * @arg Text
 * @text テキスト
 * @type multiline_string
 * @default
 * @desc ここに書いたテキストがそのままエンドロールに表示されます。
 *
 * @arg ScrollSpeed
 * @text スクロール速度
 * @type number
 * @min 0
 * @max 20
 * @default 0
 * @desc 0 の場合はパラメータ「スクロール速度」を使用します。
 */

(() => {
  "use strict";

  const pluginName = "EndRoll_Kumoten";
  const params = PluginManager.parameters(pluginName);

  const paramDefaultText = String(params["DefaultText"] || "");
  const paramScrollSpeed = Number(params["ScrollSpeed"] || 2);
  const paramWaitTop     = Number(params["WaitTop"] || 30);
  const paramWaitBottom  = Number(params["WaitBottom"] || 60);
  const paramAllowSkip   = String(params["AllowSkip"] || "true") === "true";
  const paramFastRate    = Math.max(2, Number(params["FastRate"] || 4));

  // テキスト揃え設定（left / center / right）
  const paramTextAlignRaw = String(params["TextAlign"] || "left");
  const paramTextAlign =
    paramTextAlignRaw === "center" || paramTextAlignRaw === "right"
      ? paramTextAlignRaw
      : "left";

  //-------------------------------------------------------------------------
  // 共通セットアップ
  //-------------------------------------------------------------------------

  function setupEndRoll(text, speed, waitTop, waitBottom, allowSkip) {
    const normalizedText = String(text || "").replace(/\r\n/g, "\n");
    const data = {
      text: normalizedText,
      speed: Math.max(1, Number(speed || 1)),
      waitTop: Math.max(0, Number(waitTop || 0)),
      waitBottom: Math.max(0, Number(waitBottom || 0)),
      allowSkip: !!allowSkip
    };
    $gameTemp._endRollKumotenData = data;
    SceneManager.push(Scene_EndRollKumoten);
  }

  //-------------------------------------------------------------------------
  // Plugin Commands
  //-------------------------------------------------------------------------

  PluginManager.registerCommand(pluginName, "StartFromParam", args => {
    const text = paramDefaultText;
    const speed = paramScrollSpeed;
    setupEndRoll(text, speed, paramWaitTop, paramWaitBottom, paramAllowSkip);
  });

  PluginManager.registerCommand(pluginName, "StartWithText", args => {
    const text = String(args.Text || "");
    const speedArg = Number(args.ScrollSpeed || 0);
    const speed = speedArg > 0 ? speedArg : paramScrollSpeed;
    setupEndRoll(text, speed, paramWaitTop, paramWaitBottom, paramAllowSkip);
  });

  //-------------------------------------------------------------------------
  // Scene_EndRollKumoten
  //-------------------------------------------------------------------------

  function Scene_EndRollKumoten() {
    this.initialize(...arguments);
  }

  Scene_EndRollKumoten.prototype = Object.create(Scene_Base.prototype);
  Scene_EndRollKumoten.prototype.constructor = Scene_EndRollKumoten;

  Scene_EndRollKumoten.prototype.initialize = function() {
    Scene_Base.prototype.initialize.call(this);
    const data = $gameTemp._endRollKumotenData || {};
    this._endRollData = {
      text: String(data.text || ""),
      speed: Math.max(1, Number(data.speed || 1)),
      waitTop: Math.max(0, Number(data.waitTop || 0)),
      waitBottom: Math.max(0, Number(data.waitBottom || 0)),
      allowSkip: !!data.allowSkip
    };
  };

  Scene_EndRollKumoten.prototype.create = function() {
    Scene_Base.prototype.create.call(this);
    this.createBackground();
    this.createWindowLayer();
    this.createEndRollWindow();
  };

  Scene_EndRollKumoten.prototype.start = function() {
    Scene_Base.prototype.start.call(this);
    this.startFadeIn(12, false);
  };

  Scene_EndRollKumoten.prototype.createBackground = function() {
    this._backgroundSprite = new Sprite();
    this._backgroundSprite.bitmap = new Bitmap(Graphics.boxWidth, Graphics.boxHeight);
    this._backgroundSprite.bitmap.fillRect(
      0,
      0,
      Graphics.boxWidth,
      Graphics.boxHeight,
      "#000000"
    );
    this.addChild(this._backgroundSprite);
  };

  Scene_EndRollKumoten.prototype.createEndRollWindow = function() {
    const rect = new Rectangle(0, 0, Graphics.boxWidth, Graphics.boxHeight);
    const text = this._endRollData.text;
    const lines = text.length > 0 ? text.split("\n") : [];
    this._endRollWindow = new Window_EndRollKumoten(
      rect,
      lines,
      this._endRollData.speed,
      this._endRollData.waitTop,
      this._endRollData.waitBottom,
      this._endRollData.allowSkip,
      paramFastRate,
      paramTextAlign
    );
    this.addWindow(this._endRollWindow);
  };

  Scene_EndRollKumoten.prototype.update = function() {
    Scene_Base.prototype.update.call(this);
    if (!this._endRollWindow) return;

    // 早送り入力の検出（押している間だけ早送り）
    const fast =
      this._endRollWindow.isSkippable() && this.isFastForwardPressed();
    this._endRollWindow.setFast(fast);

    if (this._endRollWindow.isFinished()) {
      SceneManager.pop();
    }
  };

  Scene_EndRollKumoten.prototype.isFastForwardPressed = function() {
    return (
      Input.isPressed("ok") ||
      Input.isPressed("cancel") ||
      TouchInput.isPressed()
    );
  };

  window.Scene_EndRollKumoten = Scene_EndRollKumoten;

  //-------------------------------------------------------------------------
  // Window_EndRollKumoten
  //-------------------------------------------------------------------------

  function Window_EndRollKumoten() {
    this.initialize(...arguments);
  }

  Window_EndRollKumoten.prototype = Object.create(Window_Base.prototype);
  Window_EndRollKumoten.prototype.constructor = Window_EndRollKumoten;

  Window_EndRollKumoten.prototype.initialize = function(
    rect,
    lines,
    speed,
    waitTop,
    waitBottom,
    allowSkip,
    fastRate,
    textAlign
  ) {
    Window_Base.prototype.initialize.call(this, rect);
    this._lines = lines || [];
    this._scrollSpeed = Math.max(1, Number(speed || 1));
    this._waitTop = Math.max(0, Number(waitTop || 0));
    this._waitBottom = Math.max(0, Number(waitBottom || 0));
    this._allowSkip = !!allowSkip;
    this._fastRate = Math.max(2, Number(fastRate || 4));
    this._fast = false;

    // 揃え方向（left / center / right）
    if (textAlign === "center" || textAlign === "right") {
      this._textAlign = textAlign;
    } else {
      this._textAlign = "left";
    }

    this.opacity = 0; // 枠非表示
    this._state = "waitTop"; // waitTop -> scroll -> waitBottom -> finished
    this._waitCount = this._waitTop;
    this._scrollY = 0;
    this._finished = false;

    this.setupContents();
  };

  Window_EndRollKumoten.prototype.isSkippable = function() {
    return this._allowSkip;
  };

  Window_EndRollKumoten.prototype.setFast = function(fast) {
    this._fast = !!fast;
  };

  Window_EndRollKumoten.prototype.setupContents = function() {
    const lh = this.lineHeight();
    const margin = lh; // 上下余白
    const lineCount = this._lines.length;
    this._contentHeight = lineCount * lh + margin * 2;

    if (this._contentHeight < this.innerHeight) {
      this._contentHeight = this.innerHeight;
    }

    this.createContents();
    this.refresh();

    // スタート位置：画面下の外側から上がってくる
    this._scrollY = -this.innerHeight;
    this.origin.y = this._scrollY;
  };

  Window_EndRollKumoten.prototype.contentsHeight = function() {
    return this._contentHeight || Window_Base.prototype.contentsHeight.call(this);
  };

  Window_EndRollKumoten.prototype.refresh = function() {
    if (!this.contents) return;
    this.contents.clear();
    this.resetFontSettings();
    this.resetTextColor();

    const lh = this.lineHeight();
    let y = lh; // 少し上に余白
    const align = this._textAlign || "left";

    for (let i = 0; i < this._lines.length; i++) {
      const line = this._lines[i];

      // 行の描画幅を計測（制御文字対応）
      const textWidth = this.textSizeEx(line).width;
      let x = 0;

      if (align === "center") {
        x = (this.innerWidth - textWidth) / 2;
      } else if (align === "right") {
        x = this.innerWidth - textWidth;
      } else {
        x = 0; // left
      }

      this.drawTextEx(line, x, y);
      y += lh;
    }
  };

  Window_EndRollKumoten.prototype.update = function() {
    Window_Base.prototype.update.call(this);
    this.updateScroll();
  };

  Window_EndRollKumoten.prototype.updateScroll = function() {
    if (this._finished) return;

    if (this._state === "waitTop") {
      if (this._waitCount > 0) {
        this._waitCount--;
      } else {
        this._state = "scroll";
      }
    } else if (this._state === "scroll") {
      const base = this._scrollSpeed;
      const speed = this._fast ? base * this._fastRate : base;
      this._scrollY += speed;
      this.origin.y = this._scrollY;
      if (this._scrollY >= this._contentHeight) {
        this._state = "waitBottom";
        this._waitCount = this._waitBottom;
      }
    } else if (this._state === "waitBottom") {
      if (this._waitCount > 0) {
        this._waitCount--;
      } else {
        this._finished = true;
      }
    }
  };

  Window_EndRollKumoten.prototype.isFinished = function() {
    return this._finished;
  };

  Window_EndRollKumoten.prototype.forceFinish = function() {
    this._finished = true;
  };
})();
