/*:
 * @target MZ
 * @plugindesc メニューのLv表示が極端に小さくなる問題を修正（カンマ付き/自動フィット）
 * @author ChatGPT
 * @help FixMenuLevelCommaMZ.js
 * 
 * メニューのアクター一覧（Window_MenuStatus）で、Lv数値が小さく潰れるケースを修正します。
 * - フォントサイズを一旦メインフォントに戻して描画
 * - 役者Lvをカンマ区切り（1,234,567）で表示
 * - 幅が足りない場合のみ最小サイズまで段階的に縮小（既定14px）
 * - 可能なら数値領域を少し広げる（左へシフト）ことで縮小を避けます
 * 
 * @param minFontSize
 * @text 最小フォントサイズ
 * @type number
 * @default 14
 * 
 * @param extraWidth
 * @text 追加で確保する横幅(px)
 * @type number
 * @default 24
 * @desc 既定幅が足りない場合に左側へ確保する余白。大きすぎると他要素に被るので注意。
 */
(() => {
  "use strict";
  const PN = document.currentScript.src.match(/([^\/]+)\.js$/)[1];
  const P = PluginManager.parameters(PN);
  const MIN_FONT = Number(P.minFontSize || 14);
  const EXTRA_W  = Number(P.extraWidth || 24);

  // カンマ整形（整数のみ）
  const comma = (n) => {
    try { return Number(n).toLocaleString(); }
    catch { return String(n); }
  };

  // 幅計測ユーティリティ
  const measure = (bmp, text, size) => {
    const keep = bmp.fontSize;
    bmp.fontSize = size;
    const w = bmp.measureTextWidth(text);
    bmp.fontSize = keep;
    return w;
  };

  // Window_MenuStatus の Lv描画をパッチ
  const _drawActorLevel = Window_MenuStatus.prototype.drawActorLevel;
  Window_MenuStatus.prototype.drawActorLevel = function(actor, x, y) {
    // 既定は x, y, 幅は 48 程度。カンマで長くなるため、左へ少し拡張。
    const baseFont = $gameSystem.mainFontSize();
    const label = TextManager.levelA; // "Lv"
    const valueStrRaw = actor.level;
    const valueStr = comma(valueStrRaw);

    // ラベルは既定どおり
    this.changeTextColor(ColorManager.systemColor());
    this.drawText(label, x, y, 36);
    this.resetTextColor();

    // 数値の描画領域
    let vx = x + 36 - EXTRA_W;  // 左に広げる
    let vw = 48 + EXTRA_W;      // 幅を足す
    if (vx < 0) { vw += vx; vx = 0; }

    // まず標準フォントで収まるか確認
    let size = baseFont;
    let need = measure(this.contents, valueStr, size);

    if (need > vw) {
      // 最小フォントまで縮小
      while (need > vw && size > MIN_FONT) {
        size -= 1;
        need = measure(this.contents, valueStr, size);
      }
    }

    // 描画
    const keep = this.contents.fontSize;
    this.contents.fontSize = size;
    this.drawText(valueStr, vx, y, vw, "right");
    this.contents.fontSize = keep;
  };
})();