/*:
 * @target MZ
 * @plugindesc v1.0.1 KT_ActorTamaKyoudaPicture: 味方がタマ強打（指定ステート）を受けた時に img/pictures の画像を表示
 * @author ChatGPT
 *
 * @help
 * ■概要
 * - 戦闘中、味方（アクター）が「タマ強打」ステートを受けたタイミングで、
 *   指定した画像（img/pictures）を画面中央付近に表示します。
 * - 既存の KT_ActorTamaKyoudaReact 系プラグインと同じ判定（新規付与 / タマ強打中にタマ強打付与効果を含む技が命中）に合わせて動作します。
 *
 * ■使い方
 * 1) img/pictures/ に表示したい画像（PNG等）を入れます。
 * 2) 本プラグインの「アクター別画像設定」で、アクターIDと画像を紐付けます。
 *    - 画像はファイル選択（一覧）で選べます。
 * 3) 画像IDは PictureIdBase から順に使用します（他用途と被らない範囲にしてください）。
 *
 * ■注意
 * - 本プラグインは「戦闘中のみ」動作します。
 * - 使用するピクチャIDの範囲は、他の演出（イベント等）と衝突しないようにしてください。
 *
 * @param TamaStateId
 * @text タマ強打ステートID
 * @type state
 * @default 33
 *
 * @param EnableReapply
 * @text タマ強打中の再命中でも表示
 * @type boolean
 * @on 表示する
 * @off 表示しない
 * @default true
 *
 * @param ActorPictureList
 * @text アクター別画像設定
 * @type struct<ActorPicture>[]
 * @default []
 *
 * @param PictureIdBase
 * @text 使用ピクチャID先頭
 * @type number
 * @min 1
 * @max 100
 * @default 80
 *
 * @param DelayFrames
 * @text 表示ディレイ(フレーム)
 * @type number
 * @min 0
 * @default 0
 *
 * @param DurationFrames
 * @text 表示時間(フレーム)
 * @type number
 * @min 1
 * @default 45
 *
 * @param RandomX
 * @text 位置ランダムX(±px)
 * @type number
 * @min 0
 * @default 24
 *
 * @param RandomY
 * @text 位置ランダムY(±px)
 * @type number
 * @min 0
 * @default 24
 *
 * @param Origin
 * @text 原点
 * @type select
 * @option 左上(0)
 * @value 0
 * @option 中央(1)
 * @value 1
 * @default 1
 *
 * @param ScaleX
 * @text 拡大率X(%)
 * @type number
 * @min 1
 * @default 100
 *
 * @param ScaleY
 * @text 拡大率Y(%)
 * @type number
 * @min 1
 * @default 100
 *
 * @param Opacity
 * @text 不透明度(0-255)
 * @type number
 * @min 0
 * @max 255
 * @default 255
 *
 * @param BlendMode
 * @text 合成方法
 * @type select
 * @option 通常(0)
 * @value 0
 * @option 加算(1)
 * @value 1
 * @option 乗算(2)
 * @value 2
 * @option スクリーン(3)
 * @value 3
 * @default 0
 */

/*~struct~ActorPicture:
 * @param ActorId
 * @text アクターID
 * @type actor
 * @default 1
 *
 * @param PictureName
 * @text 画像（img/pictures）
 * @type file
 * @dir img/pictures
 * @default
 */

(() => {
  "use strict";

  const PLUGIN_NAME = "KT_ActorTamaKyoudaPicture";

  // ----------------------------
  // Params
  // ----------------------------
  const params = PluginManager.parameters(PLUGIN_NAME);

  const jsonParse = (v, def) => {
    try { return JSON.parse(v); } catch { return def; }
  };

  const parseStructArray = (v) => {
    const arr = jsonParse(v, []);
    if (!Array.isArray(arr)) return [];
    return arr.map(s => jsonParse(s, {}));
  };

  const normalizePictureName = (name) => {
    let s = String(name || "").trim();
    if (!s) return "";
    // In some environments file-type may return a path; normalize just in case.
    s = s.replace(/^img\/pictures\//i, "");
    s = s.replace(/^\//, "");
    // $gameScreen.showPicture expects no extension
    s = s.replace(/\.(png|jpg|jpeg|webp)$/i, "");
    return s.trim();
  };

  const CFG = {
    stateId: Number(params.TamaStateId || 33),
    enableReapply: String(params.EnableReapply || "true") === "true",

    pictureIdBase: Number(params.PictureIdBase || 80),
    delay: Number(params.DelayFrames || 0),
    duration: Number(params.DurationFrames || 45),

    randX: Number(params.RandomX || 0),
    randY: Number(params.RandomY || 0),

    origin: Number(params.Origin || 1),
    scaleX: Number(params.ScaleX || 100),
    scaleY: Number(params.ScaleY || 100),
    opacity: Number(params.Opacity || 255),
    blendMode: Number(params.BlendMode || 0),

    actorPictures: parseStructArray(params.ActorPictureList).map((o) => ({
      actorId: Number(o.ActorId || 0),
      pictureName: normalizePictureName(o.PictureName),
    })),
  };

  const RULES = CFG.actorPictures
    .filter(r => r.actorId > 0 && r.pictureName)
    .map((r, idx) => ({ ...r, idx }));

  const RULE_BY_ACTOR = new Map();
  for (const r of RULES) RULE_BY_ACTOR.set(r.actorId, r);

  const U = {
    inBattle() { return $gameParty && $gameParty.inBattle && $gameParty.inBattle(); },
    isActor(battler) { return battler && battler.isActor && battler.isActor(); },
    frame() { return (typeof Graphics !== "undefined" && Graphics.frameCount != null) ? Graphics.frameCount : 0; },
    centerX() { return (typeof Graphics !== "undefined") ? Math.floor(Graphics.boxWidth / 2) : 0; },
    centerY() { return (typeof Graphics !== "undefined") ? Math.floor(Graphics.boxHeight / 2) : 0; },
    randRange(range) {
      if (!range || range <= 0) return 0;
      return Math.round((Math.random() * 2 - 1) * range);
    },
  };

  // ----------------------------
  // Runtime (in $gameTemp)
  // ----------------------------
  function rtEnsure() {
    if (!$gameTemp) return;
    if (!$gameTemp._kt_atkp_picTasks) $gameTemp._kt_atkp_picTasks = [];
    if (!$gameTemp._kt_atkp_picUsed) $gameTemp._kt_atkp_picUsed = {};
    if (!$gameTemp._kt_atkp_picGuard) $gameTemp._kt_atkp_picGuard = { frame: -1, seen: {} };
  }

  function rtResetBattle() {
    if (!$gameTemp) return;
    $gameTemp._kt_atkp_picTasks = [];
    $gameTemp._kt_atkp_picUsed = {};
    $gameTemp._kt_atkp_picGuard = { frame: -1, seen: {} };
  }

  function passGuard(actorId, key) {
    rtEnsure();
    if (!$gameTemp) return true;
    const f = U.frame();
    const g = $gameTemp._kt_atkp_picGuard;
    if (!g || g.frame !== f) {
      $gameTemp._kt_atkp_picGuard = { frame: f, seen: {} };
    }
    const seen = $gameTemp._kt_atkp_picGuard.seen;
    const k = `${actorId}:${key}`;
    if (seen[k]) return false;
    seen[k] = true;
    return true;
  }

  function picIdForRule(rule) {
    let id = CFG.pictureIdBase + rule.idx;
    if (id < 1) id = 1;
    if (id > 100) id = 100;
    return id;
  }

  function enqueuePicture(actor, kind) {
    if (!U.inBattle()) return;
    if (!U.isActor(actor)) return;

    const actorId = actor.actorId();
    const rule = RULE_BY_ACTOR.get(actorId);
    if (!rule) return;

    // kind: "new" or "reapply"
    if (!passGuard(actorId, `tama:${kind}`)) return;

    rtEnsure();
    if (!$gameTemp) return;

    const now = U.frame();
    const showAt = now + Math.max(0, CFG.delay);
    const eraseAt = showAt + Math.max(1, CFG.duration);

    const task = {
      actorId,
      pictureId: picIdForRule(rule),
      name: rule.pictureName,
      x: U.centerX() + U.randRange(CFG.randX),
      y: U.centerY() + U.randRange(CFG.randY),
      showAt,
      eraseAt,
      shown: false,
      erased: false,
    };

    $gameTemp._kt_atkp_picTasks.push(task);
  }

  function showTask(task) {
    if (!$gameScreen) return;
    if (!task.name) return;
    $gameScreen.showPicture(
      task.pictureId,
      task.name,
      CFG.origin,
      task.x,
      task.y,
      CFG.scaleX,
      CFG.scaleY,
      CFG.opacity,
      CFG.blendMode
    );
    rtEnsure();
    if ($gameTemp) $gameTemp._kt_atkp_picUsed[task.pictureId] = true;
  }

  function eraseTask(task) {
    if (!$gameScreen) return;
    $gameScreen.erasePicture(task.pictureId);
  }

  function updateTasks() {
    if (!$gameTemp || !$gameTemp._kt_atkp_picTasks) return;
    const f = U.frame();
    const tasks = $gameTemp._kt_atkp_picTasks;

    for (const t of tasks) {
      if (!t.shown && f >= t.showAt) {
        showTask(t);
        t.shown = true;
      }
      if (t.shown && !t.erased && f >= t.eraseAt) {
        eraseTask(t);
        t.erased = true;
      }
    }

    // remove finished
    $gameTemp._kt_atkp_picTasks = tasks.filter(t => !t.erased);
  }

  function eraseAllUsedPictures() {
    if (!$gameTemp || !$gameTemp._kt_atkp_picUsed) return;
    const used = $gameTemp._kt_atkp_picUsed;
    for (const idStr of Object.keys(used)) {
      const id = Number(idStr);
      if (Number.isFinite(id) && id > 0) {
        try { $gameScreen.erasePicture(id); } catch {}
      }
    }
  }

  // ----------------------------
  // Detection helpers (align with KT_ActorTamaKyoudaReact style)
  // ----------------------------
  function actionMayAddState(action, stateId) {
    try {
      const item = action.item && action.item();
      if (item && Array.isArray(item.effects)) {
        for (const e of item.effects) {
          if (e && e.code === Game_Action.EFFECT_ADD_STATE && e.dataId === stateId) return true;
        }
      }
    } catch {}
    try {
      if (action.isAttack && action.isAttack()) {
        const subject = action.subject && action.subject();
        if (subject && subject.attackStates && Array.isArray(subject.attackStates())) {
          return subject.attackStates().includes(stateId);
        }
      }
    } catch {}
    return false;
  }

  function resultIsHit(target) {
    try {
      const r = target.result && target.result();
      if (r) {
        if (typeof r.isHit === "function") return r.isHit();
        // fallback
        if ("missed" in r || "evaded" in r) return !r.missed && !r.evaded;
      }
    } catch {}
    return false;
  }

  function resultAddedState(target, stateId) {
    try {
      const r = target.result && target.result();
      if (!r) return false;
      if (typeof r.isStateAdded === "function") return r.isStateAdded(stateId);
      if (Array.isArray(r.addedStates)) return r.addedStates.includes(stateId);
    } catch {}
    return false;
  }

  // ----------------------------
  // Hooks
  // ----------------------------
  const _BattleManager_startBattle = BattleManager.startBattle;
  BattleManager.startBattle = function() {
    rtResetBattle();
    _BattleManager_startBattle.apply(this, arguments);
  };

  const _BattleManager_endBattle = BattleManager.endBattle;
  BattleManager.endBattle = function(result) {
    try { eraseAllUsedPictures(); } catch {}
    rtResetBattle();
    _BattleManager_endBattle.apply(this, arguments);
  };

  const _Scene_Battle_update = Scene_Battle.prototype.update;
  Scene_Battle.prototype.update = function() {
    _Scene_Battle_update.apply(this, arguments);
    if (U.inBattle()) updateTasks();
  };

  // (A) 新規付与（addNewState）
  const _Game_BattlerBase_addNewState = Game_BattlerBase.prototype.addNewState;
  Game_BattlerBase.prototype.addNewState = function(stateId) {
    _Game_BattlerBase_addNewState.apply(this, arguments);

    if (!U.inBattle()) return;
    if (!U.isActor(this)) return;
    if (stateId === CFG.stateId) {
      enqueuePicture(this, "new");
    }
  };

  // (A) 新規付与（applyの結果から救済） + (B) タマ強打中に命中
  const _Game_Action_apply = Game_Action.prototype.apply;
  Game_Action.prototype.apply = function(target) {
    const inBattle = U.inBattle();
    const isActorTarget = inBattle && target && U.isActor(target);

    let hadTamaBefore = false;
    let mayAddTama = false;

    if (isActorTarget) {
      try {
        hadTamaBefore = target.isStateAffected(CFG.stateId);
        mayAddTama = actionMayAddState(this, CFG.stateId);
      } catch {}
    }

    _Game_Action_apply.apply(this, arguments);

    if (!isActorTarget) return;

    const hit = resultIsHit(target);
    const added = resultAddedState(target, CFG.stateId);

    // (A) 新規付与（結果からの救済）
    if (added && !hadTamaBefore) {
      enqueuePicture(target, "new");
      return;
    }

    // (B) タマ強打中に、タマ強打付与効果を含む技が命中
    if (CFG.enableReapply && hadTamaBefore && mayAddTama && hit) {
      enqueuePicture(target, "reapply");
      return;
    }
  };
})();
