/*:
 * @target MZ
 * @plugindesc v1.1.2 KT_ActorTamaKyoudaReact: 味方がタマ強打ステートになったときトースト表示（顔/右半分ランダム/自動高さ/戦闘終了直前まとめ表示/死亡同時も拾う/追加リアクション対応）
 * @author ChatGPT
 *
 * @help
 * ■概要
 * - 戦闘中、味方（アクター）が指定ステート（タマ強打）になったタイミングでトースト表示します。
 * - 置換： {target} = 対象名 / {speaker} = 発言者名
 *
 * ■表示条件（要件）
 * (A) 新規付与：味方がタマ強打ステートになったとき
 * (B) タマ強打中：タマ強打中の味方に「タマ強打付与効果を含む技」が命中したとき
 *
 * ■致死時フォールバック（v1.1.2）
 * - タマ強打付与効果を含む技が命中し、その行動で対象が戦闘不能になった場合、
 *   ステート付与が成立しなくてもトーストを表示できます（設定でON/OFF）。
 *
 * ■戦闘不能同時
 * - (A)(B) の条件を満たしたうえで同じ行動内で戦闘不能になっても表示されます。
 * - ただし「タマ強打付与効果はあるが、タマ強打にならないまま戦闘不能になった」ケースは表示しません。
 *
 * ■追加リアクション（v1.1.0）
 * - 対象のトーストを出した後に、任意の発言者（別アクターなど）のトーストを続けて出せます。
 * - @param ExtraReactions で設定します。
 *
 * @param StateIds
 * @text タマ強打ステートID（複数）
 * @type state[]
 * @default ["33"]
 *
 * @param StateId
 * @text タマ強打ステートID
 * @type state
 * @default 2
 *
 * @param DeathStateId
 * @text 戦闘不能ステートID
 * @type state
 * @default 1
 *
 * @param EnableDeathSameFrame
 * @text 同フレーム戦闘不能でも表示
 * @type boolean
 * @default true
 *
 * @param ShowOnReapplyAction
 * @text タマ強打中に命中でも表示（味方）
 * @type boolean
 * @desc タマ強打中の味方に、タマ強打付与効果を含む行動が「命中した」とき表示
 * @default true
 *
 * @param FallbackOnDeathWhenTamaNotApplied
 * @text 致死でも表示（付与失敗フォールバック）
 * @type boolean
 * @desc タマ強打付与効果を含む技が命中し、その行動で戦闘不能になった場合、ステート付与が成立しなくても表示
 * @default true
 *
 * @param DebugLog
 * @text デバッグログ
 * @type boolean
 * @default false
 *
 * @param ActorDefaultMessages
 * @text 味方がタマ強打（共通）のメッセージ（複数）
 * @type string[]
 * @default ["{target}！大丈夫！？"]
 *
 * @param ActorMessages
 * @text 味方がタマ強打（アクター別）
 * @type struct<ActorMsg>[]
 * @default []
 *
 * @param ExtraReactions
 * @text 追加リアクション（複数）
 * @type struct<ExtraReact>[]
 * @desc 対象Aの後に、発言者Bなどのトーストを追加で出す設定
 * @default []
 *
 * @param ActorFaceOverrides
 * @text アクター顔指定（複数）
 * @type struct<ActorFaceEntry>[]
 * @desc アクターIDごとに顔（img/faces）を指定。未指定ならDB設定の顔を使用。
 * @default []
 *
 * @param ToastWidth
 * @text トースト幅
 * @type number
 * @min 240
 * @default 560
 *
 * @param ToastMinHeight
 * @text トースト最小高さ
 * @type number
 * @min 96
 * @default 160
 *
 * @param ToastMaxHeight
 * @text トースト最大高さ
 * @type number
 * @min 96
 * @default 360
 *
 * @param AutoFitHeight
 * @text 全文が見えるように自動高さ
 * @type boolean
 * @default true
 *
 * @param ToastFrames
 * @text 表示フレーム数
 * @type number
 * @min 1
 * @default 120
 *
 * @param ToastOpenCloseSpeed
 * @text 開閉速度（1-64）
 * @type number
 * @min 1
 * @max 64
 * @default 16
 *
 * @param RightHalfDefault
 * @text 右半分を基本にする
 * @type boolean
 * @desc trueなら、位置範囲未指定時に「画面右半分」内でランダム表示
 * @default true
 *
 * @param ToastXMin
 * @text X最小（範囲指定）
 * @type number
 * @min -99999
 * @max 99999
 * @default 0
 *
 * @param ToastXMax
 * @text X最大（範囲指定）
 * @type number
 * @min -99999
 * @max 99999
 * @default 0
 *
 * @param ToastYMin
 * @text Y最小（範囲指定）
 * @type number
 * @min -99999
 * @max 99999
 * @default 0
 *
 * @param ToastYMax
 * @text Y最大（範囲指定）
 * @type number
 * @min -99999
 * @max 99999
 * @default 0
 *
 * @param FlushToastOnEndBattle
 * @text 戦闘終了直前：未表示分をまとめてトーストに移す
 * @type boolean
 * @default true
 *
 * @param PreloadActorFaces
 * @text 戦闘開始時：顔を先読み
 * @type boolean
 * @default true
 */

/*~struct~ActorMsg:
 * @param ActorId
 * @text アクターID
 * @type actor
 * @default 1
 *
 * @param Messages
 * @text このアクター用メッセージ（複数）
 * @type string[]
 * @default ["{target}！？い、いまのは…！"]
 */

/*~struct~ExtraReact:
 * @param TargetActorId
 * @text 対象アクターID（0=全員）
 * @type number
 * @min 0
 * @default 0
 *
 * @param SpeakerActorId
 * @text 発言者アクターID（0=自動で別メンバー）
 * @type number
 * @min 0
 * @default 0
 *
 * @param RequireSpeakerInParty
 * @text 発言者がパーティにいる時のみ
 * @type boolean
 * @default true
 *
 * @param RequireSpeakerAlive
 * @text 発言者が生存時のみ
 * @type boolean
 * @default true
 *
 * @param Messages
 * @text 発言メッセージ（複数）
 * @type string[]
 * @default ["{target}！だ、大丈夫ですか！？"]
 */

/*~struct~ActorFaceEntry:
 * @param ActorId
 * @text アクターID
 * @type actor
 * @default 1
 *
 * @param FaceFile
 * @text 顔ファイル（img/faces）
 * @type file
 * @dir img/faces
 * @default
 *
 * @param FaceIndex
 * @text 顔インデックス（0-7）
 * @type number
 * @min 0
 * @max 7
 * @default 0
 */

(() => {
  "use strict";

  function detectPluginName() {
    try {
      const cs = document.currentScript;
      if (cs && cs.src) {
        const m = cs.src.match(/([^/\\]+)\.js(?:\?.*)?$/);
        if (m) return m[1];
      }
    } catch {}
    const KEY = "KT_ActorTamaKyoudaReact";
    try {
      if (Array.isArray(window.$plugins)) {
        const hit = window.$plugins.find(p => p && typeof p.description === "string" && p.description.includes(KEY));
        if (hit && hit.name) return String(hit.name);
      }
    } catch {}
    return "KT_ActorTamaKyoudaReact";
  }

  const PLUGIN_NAME = detectPluginName();
  const P = PluginManager.parameters(PLUGIN_NAME) || {};

  const U = {
    bool(v, def) { if (v === undefined || v === null || v === "") return !!def; return String(v) === "true"; },
    num(v, def) { const n = Number(v); return Number.isFinite(n) ? n : def; },
    clamp(n, a, b) { n = Number(n); if (!Number.isFinite(n)) return a; return Math.max(a, Math.min(b, n)); },
    strArray(v) {
      if (!v) return [];
      try {
        const arr = JSON.parse(v);
        if (!Array.isArray(arr)) return [];
        return arr.map(s => String(s)).map(s => s.trim()).filter(s => s.length > 0);
      } catch { return []; }
    },
    pickRandom(arr) {
      if (!arr || arr.length === 0) return "";
      const i = (Math.randomInt ? Math.randomInt(arr.length) : Math.floor(Math.random() * arr.length));
      return arr[i];
    },
    inBattle() { try { return $gameParty && $gameParty.inBattle(); } catch { return false; } },
    battlerName(b) { try { return b && b.name ? b.name() : ""; } catch { return ""; } },
    isActor(b) { try { return !!(b && b.isActor && b.isActor()); } catch { return false; } },
    actorId(b) { try { return b && b.actorId ? b.actorId() : 0; } catch { return 0; } },
    frameCount() {
      try { return (typeof Graphics !== "undefined" && Number.isFinite(Graphics.frameCount)) ? Graphics.frameCount : 0; }
      catch { return 0; }
    },
    randInt(min, max) {
      min = Math.floor(Number(min)); max = Math.floor(Number(max));
      if (!Number.isFinite(min) || !Number.isFinite(max)) return 0;
      if (min > max) [min, max] = [max, min];
      return min + Math.floor(Math.random() * (max - min + 1));
    },
  };

  const CFG = {
    stateIds: (function(){ try{ const a=JSON.parse(P.StateIds||'[]').map(Number).filter(n=>n>0); return a.length?a:[U.num(P.StateId,2)]; }catch(e){ return [U.num(P.StateId,2)]; } })(),
    stateId: U.num(P.StateId, 2),
    deathStateId: U.num(P.DeathStateId, 1),
    enableDeathSameFrame: U.bool(P.EnableDeathSameFrame, true),
    showOnReapplyAction: U.bool(P.ShowOnReapplyAction, true),
    fallbackOnDeathWhenTamaNotApplied: U.bool(P.FallbackOnDeathWhenTamaNotApplied, true),
    debug: U.bool(P.DebugLog, false),

    actorDefaultMessages: U.strArray(P.ActorDefaultMessages),
    actorMessageMap: Object.create(null),
    extraReactions: [],

    actorFaceOverrideMap: Object.create(null),

    toastWidth: Math.max(240, U.num(P.ToastWidth, 560)),
    toastMinHeight: Math.max(96, U.num(P.ToastMinHeight, 160)),
    toastMaxHeight: Math.max(96, U.num(P.ToastMaxHeight, 360)),
    autoFitHeight: U.bool(P.AutoFitHeight, true),
    toastFrames: Math.max(1, U.num(P.ToastFrames, 120)),
    toastSpeed: U.clamp(U.num(P.ToastOpenCloseSpeed, 16), 1, 64),

    rightHalfDefault: U.bool(P.RightHalfDefault, true),
    xMin: U.num(P.ToastXMin, 0),
    xMax: U.num(P.ToastXMax, 0),
    yMin: U.num(P.ToastYMin, 0),
    yMax: U.num(P.ToastYMax, 0),

    flushToastOnEndBattle: U.bool(P.FlushToastOnEndBattle, true),
    preloadActorFaces: U.bool(P.PreloadActorFaces, true),
  };

  function dlog(label, payload) {
    if (!CFG.debug) return;
    try { console.log("[" + PLUGIN_NAME + "] " + label, payload); } catch {}
  }

  function parseStructList(rawJson) {
    const raw = rawJson || "[]";
    let list = [];
    try { list = JSON.parse(raw); if (!Array.isArray(list)) list = []; } catch { list = []; }
    return list.map(s => { try { return JSON.parse(s); } catch { return {}; } });
  }

  function parseActorMsgList(rawJson, outMap) {
    const list = parseStructList(rawJson);
    for (const o of list) {
      const aid = U.num(o.ActorId, 0);
      const msgs = U.strArray(o.Messages);
      if (aid > 0 && msgs.length > 0) outMap[aid] = msgs;
    }
  }

  function parseActorFaceList(rawJson, outMap) {
    const list = parseStructList(rawJson);
    for (const o of list) {
      const aid = U.num(o.ActorId, 0);
      const faceName = String(o.FaceFile || "").trim();
      const faceIndex = U.clamp(U.num(o.FaceIndex, 0), 0, 7);
      if (aid > 0 && faceName) outMap[aid] = { faceName, faceIndex };
    }
  }

  function parseExtraReactions(rawJson, outList) {
    const list = parseStructList(rawJson);
    for (const o of list) {
      const targetActorId = U.num(o.TargetActorId, 0);
      const speakerActorId = U.num(o.SpeakerActorId, 0);
      const requireInParty = U.bool(o.RequireSpeakerInParty, true);
      const requireAlive = U.bool(o.RequireSpeakerAlive, true);
      const msgs = U.strArray(o.Messages);
      if (msgs.length <= 0) continue;
      outList.push({
        targetActorId: Math.max(0, targetActorId),
        speakerActorId: Math.max(0, speakerActorId),
        requireInParty,
        requireAlive,
        messages: msgs,
      });
    }
  }

  parseActorMsgList(P.ActorMessages, CFG.actorMessageMap);
  parseExtraReactions(P.ExtraReactions, CFG.extraReactions);
  parseActorFaceList(P.ActorFaceOverrides, CFG.actorFaceOverrideMap);

  const RT = {
    reset() {
      if (!$gameTemp) return;
      $gameTemp._kt_atk_guard = Object.create(null);
      $gameTemp._kt_atk_pendingToast = [];
      $gameTemp._kt_atk_tamaFrame = Object.create(null);
    },
    guardMap() {
      if (!$gameTemp) return Object.create(null);
      if (!$gameTemp._kt_atk_guard) $gameTemp._kt_atk_guard = Object.create(null);
      return $gameTemp._kt_atk_guard;
    },
    targetKey(target) {
      if (!target) return "";
      if (U.isActor(target)) return "A" + U.actorId(target);
      return "";
    },
    passGuardEvent(target, eventKey) {
      const key = this.targetKey(target);
      if (!key) return true;
      const f = U.frameCount();
      const k = key + ":" + eventKey + ":F" + f;
      const m = this.guardMap();
      if (m[k]) return false;
      m[k] = true;
      return true;
    },
    pendingToast() {
      if (!$gameTemp) return [];
      if (!$gameTemp._kt_atk_pendingToast) $gameTemp._kt_atk_pendingToast = [];
      return $gameTemp._kt_atk_pendingToast;
    },
    pushToast(entry) {
      if (!entry || !entry.text) return;
      this.pendingToast().push(entry);
    },
    popToast() {
      const q = this.pendingToast();
      return q.length > 0 ? q.shift() : null;
    },
    drainAllToArray() {
      const q = this.pendingToast();
      if (!q || q.length === 0) return [];
      const out = q.slice(0);
      q.length = 0;
      return out;
    },

    tamaFrameMap() {
      if (!$gameTemp) return Object.create(null);
      if (!$gameTemp._kt_atk_tamaFrame) $gameTemp._kt_atk_tamaFrame = Object.create(null);
      return $gameTemp._kt_atk_tamaFrame;
    },
    markTamaThisFrame(target) {
      const k = this.targetKey(target);
      if (!k) return;
      this.tamaFrameMap()[k] = U.frameCount();
    },
    wasTamaThisFrame(target) {
      const k = this.targetKey(target);
      if (!k) return false;
      return this.tamaFrameMap()[k] === U.frameCount();
    },
  };

  // ------------------------------------------------------------
  // Toast Window
  // ------------------------------------------------------------
  class Window_KT_ActorTamaToast extends Window_Base {
    initialize() {
      const rect = new Rectangle(0, 0, CFG.toastWidth, CFG.toastMinHeight);
      super.initialize(rect);
      this.openness = 0;

      this._speed = CFG.toastSpeed;
      this._remain = 0;
      this._entry = null;
      this._ktNeedClearOnClosed = false;

      this._queue = [];
    }

    setSpeed(n) { this._speed = U.clamp(n, 1, 64); }

    updateOpen() {
      if (this._opening) {
        this.openness = Math.min(255, this.openness + this._speed);
        if (this.openness >= 255) this._opening = false;
      }
    }
    updateClose() {
      if (this._closing) {
        this.openness = Math.max(0, this.openness - this._speed);
        if (this.openness <= 0) this._closing = false;
      }
    }

    isBusy() {
      return !!(this._entry || this._remain > 0 || this.openness > 0 || this._opening || this._closing);
    }

    enqueue(entry) {
      if (!entry || !entry.text) return;
      this._queue.push(entry);
    }

    enqueueMany(list) {
      if (!Array.isArray(list) || list.length === 0) return;
      for (const e of list) if (e && e.text) this._queue.push(e);
    }

    _dequeue() {
      return this._queue.length > 0 ? this._queue.shift() : null;
    }

    open() { this._opening = true; this._closing = false; }
    close() { this._closing = true; this._opening = false; this._ktNeedClearOnClosed = true; }

    update() {
      super.update();

      if (this._ktNeedClearOnClosed && this.openness === 0 && !this._closing && !this._opening) {
        this._ktNeedClearOnClosed = false;
        this._entry = null;
        this._remain = 0;
        try { this.contents.clear(); } catch {}
      }

      if (this._entry && !this._closing) {
        this._remain--;
        if (this._remain <= 0) this.close();
      }

      if (!this._entry && this.openness === 0 && !this._opening && !this._closing) {
        let next = this._dequeue();
        if (!next) next = RT.popToast();
        if (next) this._startEntry(next);
      }
    }

    showEntry(entry) {
      if (!entry || !entry.text) return;
      if (this.isBusy()) this.enqueue(entry);
      else this._startEntry(entry);
    }

    _startEntry(entry) {
      this._entry = entry;
      this._remain = Math.max(1, entry.frames || CFG.toastFrames);

      this._resizeForText(entry);
      this._placeRandom();

      this.refresh();
      this.open();
    }

    _resolveRange() {
      const bw = Graphics.boxWidth;
      const bh = Graphics.boxHeight;

      const userXSpecified = !(CFG.xMin === 0 && CFG.xMax === 0);
      const userYSpecified = !(CFG.yMin === 0 && CFG.yMax === 0);

      let xMin, xMax, yMin, yMax;

      if (userXSpecified) {
        xMin = CFG.xMin;
        xMax = CFG.xMax;
      } else if (CFG.rightHalfDefault) {
        xMin = Math.floor(bw / 2);
        xMax = Math.max(xMin, bw - this.width);
      } else {
        xMin = 0;
        xMax = Math.max(0, bw - this.width);
      }

      if (userYSpecified) {
        yMin = CFG.yMin;
        yMax = CFG.yMax;
      } else {
        yMin = 0;
        yMax = Math.max(0, bh - this.height);
      }

      return { xMin, xMax, yMin, yMax, bw, bh };
    }

    _placeRandom() {
      const r = this._resolveRange();
      let x = U.randInt(r.xMin, r.xMax);
      let y = U.randInt(r.yMin, r.yMax);

      x = U.clamp(x, 0, Math.max(0, r.bw - this.width));
      y = U.clamp(y, 0, Math.max(0, r.bh - this.height));

      this.move(x, y, this.width, this.height);
    }

    _resizeForText(entry) {
      if (!CFG.autoFitHeight) {
        if (this.width !== CFG.toastWidth || this.height !== CFG.toastMinHeight) {
          this.move(this.x, this.y, CFG.toastWidth, CFG.toastMinHeight);
          this.createContents();
        }
        return;
      }

      const faceName = entry.faceName || "";
      const text = this.convertEscapeCharacters(String(entry.text));

      const newW = CFG.toastWidth;
      const pad = this.padding;
      const innerW = newW - pad * 2;

      const faceW = (faceName ? ImageManager.faceWidth : 0);
      const textW = Math.max(48, innerW - (faceName ? (faceW + this.itemPadding()) : 0));

      const lines = this._wrapLinesForMeasure(text, textW);
      const lineH = this.lineHeight();
      const textH = Math.max(1, lines.length) * lineH;

      const faceH = (faceName ? ImageManager.faceHeight : 0);
      const innerHNeed = Math.max(faceH, textH);

      let newH = innerHNeed + pad * 2;
      newH = U.clamp(newH, CFG.toastMinHeight, CFG.toastMaxHeight);

      if (this.width !== newW || this.height !== newH) {
        this.move(this.x, this.y, newW, newH);
        this.createContents();
      }
    }

    refresh() {
      this.contents.clear();
      if (!this._entry) return;

      const text = this.convertEscapeCharacters(String(this._entry.text));
      const faceName = this._entry.faceName || "";
      const faceIndex = U.clamp(U.num(this._entry.faceIndex, 0), 0, 7);

      const pad = this.itemPadding();
      const innerW = this.innerWidth;
      const innerH = this.innerHeight;

      let textX = 0;
      if (faceName) {
        try { ImageManager.loadFace(faceName); } catch {}
        this.drawFace(faceName, faceIndex, 0, 0, ImageManager.faceWidth, ImageManager.faceHeight);
        textX = ImageManager.faceWidth + pad;
      }

      const textW = innerW - textX;
      this._drawWrappedText(text, textX, 0, textW, innerH);
    }

    _drawWrappedText(text, x, y, w, h) {
      const lineH = this.lineHeight();
      const maxLines = Math.max(1, Math.floor(h / lineH));
      const lines = this._wrapLinesForDraw(text, w, maxLines);
      for (let i = 0; i < lines.length; i++) this.drawTextEx(lines[i], x, y + i * lineH, w);
    }

    _wrapLinesForMeasure(text, w) {
      const tokens = this._tokenizeForWrap(String(text));
      const lines = [];
      let cur = "";
      for (const token of tokens) {
        const next = cur ? (cur + token) : token;
        if (this._measureWidth(next) <= w) cur = next;
        else { if (cur) lines.push(cur); cur = token; }
      }
      if (cur) lines.push(cur);
      return lines.length > 0 ? lines : [String(text)];
    }

    _wrapLinesForDraw(text, w, maxLines) {
      const tokens = this._tokenizeForWrap(String(text));
      const lines = [];
      let cur = "";
      for (const t of tokens) {
        const next = cur ? (cur + t) : t;
        if (this._measureWidth(next) <= w) cur = next;
        else {
          if (cur) lines.push(cur);
          cur = t;
          if (lines.length >= maxLines) break;
        }
      }
      if (lines.length < maxLines && cur) lines.push(cur);
      return lines;
    }

    _measureWidth(s) {
      const str = String(s);
      let w = 0;
      const icons = str.match(/\x1bI\[\d+\]/gi);
      if (icons && icons.length) w += icons.length * (ImageManager.iconWidth || 32);
      w += this.textWidth(this._stripEscape(str));
      return w;
    }

    _stripEscape(s) {
      return String(s)
        .replace(/\x1bC\[\d+\]/gi, "")
        .replace(/\x1bI\[\d+\]/gi, "")
        .replace(/\x1bV\[\d+\]/gi, "")
        .replace(/\x1bN\[\d+\]/gi, "")
        .replace(/\x1bG/gi, "")
        .replace(/\x1b\{/g, "")
        .replace(/\x1b\}/g, "")
        .replace(/\x1b/g, "");
    }

    _tokenizeForWrap(text) {
      const s = String(text);
      const re = /\x1b[A-Z](?:\[\d+\])?|\x1bG|\x1b\{|\x1b\}|\s+|./g;
      return s.match(re) || [];
    }
  }

  // ------------------------------------------------------------
  // Scene & Battle hooks
  // ------------------------------------------------------------
  const _BattleManager_startBattle = BattleManager.startBattle;
  BattleManager.startBattle = function() {
    RT.reset();
    _BattleManager_startBattle.apply(this, arguments);

    if (CFG.preloadActorFaces) {
      try {
        const ms = $gameParty && $gameParty.members ? $gameParty.members() : [];
        for (const a of ms) if (a && a.faceName && a.faceName()) ImageManager.loadFace(a.faceName());
      } catch {}
      try {
        for (const k of Object.keys(CFG.actorFaceOverrideMap)) {
          const ov = CFG.actorFaceOverrideMap[k];
          if (ov && ov.faceName) ImageManager.loadFace(ov.faceName);
        }
      } catch {}
    }
  };

  const _BattleManager_endBattle = BattleManager.endBattle;
  BattleManager.endBattle = function(result) {
    if (CFG.flushToastOnEndBattle) {
      try {
        const sc = SceneManager._scene;
        const toastW = sc && sc._ktActorTamaToastWindow;
        if (toastW) {
          const all = RT.drainAllToArray();
          if (all.length > 0) toastW.enqueueMany(all);
          if (!toastW.isBusy()) {
            const e = toastW._dequeue ? toastW._dequeue() : null;
            if (e) toastW.showEntry(e);
          }
        }
      } catch (e) {
        dlog("endBattle flush error", e);
      }
    }
    return _BattleManager_endBattle.apply(this, arguments);
  };

  if (typeof Scene_Battle !== "undefined") {
    const _Scene_Battle_createAllWindows = Scene_Battle.prototype.createAllWindows;
    Scene_Battle.prototype.createAllWindows = function() {
      _Scene_Battle_createAllWindows.apply(this, arguments);

      this._ktActorTamaToastWindow = new Window_KT_ActorTamaToast();
      this._ktActorTamaToastWindow.setSpeed(CFG.toastSpeed);
      this.addWindow(this._ktActorTamaToastWindow);

      try {
        const all = RT.drainAllToArray();
        if (all.length > 0) this._ktActorTamaToastWindow.enqueueMany(all);
      } catch {}

      const first = this._ktActorTamaToastWindow._dequeue();
      if (first) this._ktActorTamaToastWindow.showEntry(first);
    };

    const _Scene_Battle_terminate = Scene_Battle.prototype.terminate;
    Scene_Battle.prototype.terminate = function() {
      _Scene_Battle_terminate.apply(this, arguments);
      try { RT.reset(); } catch {}
    };
  }

  // ------------------------------------------------------------
  // Output
  // ------------------------------------------------------------
  function speakerNameById(actorId) {
    const aid = Number(actorId || 0);
    if (aid <= 0) return "";
    try {
      const a = $gameActors && $gameActors.actor(aid);
      return a ? a.name() : "";
    } catch {}
    return "";
  }

  function formatText(raw, target, speakerActorId) {
    const tname = U.battlerName(target);
    const sname = speakerActorId ? speakerNameById(speakerActorId) : tname;
    return String(raw || "")
      .replace(/{target}/g, tname)
      .replace(/{speaker}/g, sname);
  }

  function resolveActorFaceById(actorId) {
    const aid = Number(actorId || 0);
    if (aid <= 0) return { faceName: "", faceIndex: 0 };

    const ov = CFG.actorFaceOverrideMap[aid];
    if (ov && ov.faceName) return { faceName: ov.faceName, faceIndex: U.clamp(ov.faceIndex, 0, 7) };

    try {
      const a = $gameActors && $gameActors.actor(aid);
      if (a && a.faceName && a.faceName()) {
        return { faceName: a.faceName(), faceIndex: U.clamp(a.faceIndex(), 0, 7) };
      }
    } catch {}
    return { faceName: "", faceIndex: 0 };
  }

  function enqueueToast(entry) {
    try {
      const sc = SceneManager._scene;
      if (sc && sc._ktActorTamaToastWindow) sc._ktActorTamaToastWindow.showEntry(entry);
      else RT.pushToast(entry);
    } catch {
      RT.pushToast(entry);
    }
  }

  function pushToast(text, face) {
    if (!text) return;
    const entry = {
      text: String(text),
      faceName: face && face.faceName ? String(face.faceName) : "",
      faceIndex: face && Number.isFinite(face.faceIndex) ? face.faceIndex : 0,
      frames: CFG.toastFrames,
    };
    enqueueToast(entry);
  }

  // ------------------------------------------------------------
  // Message selection
  // ------------------------------------------------------------
  function pickActorMessage(target) {
    const aid = U.actorId(target);
    const list = CFG.actorMessageMap[aid] || CFG.actorDefaultMessages;
    if (!list || list.length === 0) return "";
    return formatText(U.pickRandom(list), target, aid);
  }

  function actorInParty(actorId) {
    try {
      const ms = $gameParty && $gameParty.members ? $gameParty.members() : [];
      return ms.some(a => a && a.actorId && a.actorId() === actorId);
    } catch {}
    return false;
  }

  function actorIsAlive(actorId) {
    try {
      const a = $gameActors && $gameActors.actor(actorId);
      return a ? a.isAlive() : false;
    } catch {}
    return false;
  }

  function autoPickSpeakerId(excludeActorId, requireAlive) {
    try {
      const ms = $gameParty && $gameParty.members ? $gameParty.members() : [];
      for (const a of ms) {
        if (!a || !a.actorId) continue;
        const aid = a.actorId();
        if (aid === excludeActorId) continue;
        if (requireAlive && !a.isAlive()) continue;
        return aid;
      }
    } catch {}
    return 0;
  }

  function showTamaToast(target, reason) {
    if (!U.inBattle()) return;
    if (!target || !U.isActor(target)) return;

    // 1発火につき1回だけガード（この中で複数トーストを積む）
    if (!RT.passGuardEvent(target, "TAMA_TOAST")) {
      dlog("Guarded", { reason, target: U.battlerName(target) });
      return;
    }

    const targetId = U.actorId(target);

    // (1) 対象本人のリアクション
    const msg = pickActorMessage(target);
    const face = resolveActorFaceById(targetId);
    if (msg) pushToast(msg, face);

    // (2) 追加リアクション（別アクター等）
    if (Array.isArray(CFG.extraReactions) && CFG.extraReactions.length > 0) {
      for (const r of CFG.extraReactions) {
        if (!r) continue;
        if (r.targetActorId > 0 && r.targetActorId !== targetId) continue;

        let speakerId = Number(r.speakerActorId || 0);
        if (speakerId <= 0) speakerId = autoPickSpeakerId(targetId, !!r.requireAlive);
        if (speakerId <= 0) continue;

        if (r.requireInParty && !actorInParty(speakerId)) continue;
        if (r.requireAlive && !actorIsAlive(speakerId)) continue;

        const line = U.pickRandom(r.messages);
        const lineText = formatText(line, target, speakerId);
        const speakerFace = resolveActorFaceById(speakerId);
        if (lineText) pushToast(lineText, speakerFace);
      }
    }

    dlog("Shown", { reason, target: U.battlerName(target), msg, extraCount: (CFG.extraReactions || []).length });
  }

  // ------------------------------------------------------------
  // Hooks（state）
  // ------------------------------------------------------------
  const _Game_BattlerBase_addNewState = Game_BattlerBase.prototype.addNewState;
  Game_BattlerBase.prototype.addNewState = function(stateId) {
    _Game_BattlerBase_addNewState.apply(this, arguments);

    if (!U.inBattle()) return;
    if (!U.isActor(this)) return;

    if (CFG.stateIds.includes(stateId)) {
      if (CFG.enableDeathSameFrame) RT.markTamaThisFrame(this);
      showTamaToast(this, "tamaApplied(addNewState)");
    }

    if (CFG.enableDeathSameFrame && stateId === CFG.deathStateId) {
      if (RT.wasTamaThisFrame(this)) {
        showTamaToast(this, "deathSameFrame(addNewState)");
      }
    }
  };

  const _Game_Battler_addState = Game_Battler.prototype.addState;
  Game_Battler.prototype.addState = function(stateId) {
    const had = this.isStateAffected(stateId);
    _Game_Battler_addState.apply(this, arguments);

    if (!U.inBattle()) return;
    if (!U.isActor(this)) return;

    // 新規付与のみ
    if (had) return;
    if (!this.isStateAffected(stateId)) return;

    if (CFG.stateIds.includes(stateId)) {
      if (CFG.enableDeathSameFrame) RT.markTamaThisFrame(this);
      showTamaToast(this, "tamaApplied(addState)");
    }

    if (CFG.enableDeathSameFrame && stateId === CFG.deathStateId) {
      if (RT.wasTamaThisFrame(this)) {
        showTamaToast(this, "deathSameFrame(addState)");
      }
    }
  };

  function actionHasAddStateEffect(action, stateId) {
    try {
      const item = action && action.item ? action.item() : null;
      const effects = item && item.effects ? item.effects : null;
      if (!effects) return false;
      for (let i = 0; i < effects.length; i++) {
        const ef = effects[i];
        if (ef && ef.code === Game_Action.EFFECT_ADD_STATE && ef.dataId === stateId) return true;
      }
    } catch {}
    return false;
  }

  // 通常攻撃の「攻撃時ステート」も（可能な範囲で）考慮
  function actionMayAddState(action, stateId) {
    if (actionHasAddStateEffect(action, stateId)) return true;
    try {
      if (action && action.isAttack && action.isAttack()) {
        const subject = action.subject && action.subject();
        if (subject && subject.attackStates && Array.isArray(subject.attackStates())) {
          return subject.attackStates().includes(stateId);
        }
      }
    } catch {}
    return false;
  }

  // ------------------------------------------------------------
  // Hooks（apply/result）
  // ------------------------------------------------------------
  const _Game_Action_apply = Game_Action.prototype.apply;
  Game_Action.prototype.apply = function(target) {
    const inBattle = U.inBattle();
    const isActorTarget = inBattle && target && U.isActor(target);

    let hadTamaBefore = false;
    let mayAddTama = false;

    let wasDeadBefore = false;

    if (isActorTarget) {
      try {
        hadTamaBefore = target.isStateAffected(CFG.stateId);
        mayAddTama = actionMayAddState(this, CFG.stateId);
        wasDeadBefore = target.isDead ? target.isDead() : false;
      } catch {}
    }

    _Game_Action_apply.apply(this, arguments);

    if (!isActorTarget) return;

    // 命中判定
    let hit = false;
    try {
      const r = target.result && target.result();
      if (r) {
        if (typeof r.isHit === "function") hit = r.isHit();
        else hit = !r.missed && !r.evaded;
      }
    } catch {}

    // タマ強打が「追加された」判定（新規付与用）
    let added = false;
    try {
      const r = target.result && target.result();
      if (r) {
        if (typeof r.isStateAdded === "function") {
          added = r.isStateAdded(CFG.stateId);
        } else if (Array.isArray(r.addedStates)) {
          added = r.addedStates.includes(CFG.stateId);
        }
      }
    } catch {}

    const diedNow = (!wasDeadBefore && (target.isDead ? target.isDead() : false));

    // (A) 新規付与（結果からの救済）
    if (added && !hadTamaBefore) {
      if (CFG.enableDeathSameFrame) RT.markTamaThisFrame(target);
      showTamaToast(target, "tamaAddedByResult(apply)");
      return;
    }

    // (A-2) フォールバック：命中し、その行動で戦闘不能になったためステート付与が成立しなかった場合でも表示
    if (CFG.fallbackOnDeathWhenTamaNotApplied && !hadTamaBefore && mayAddTama && hit && diedNow) {
      let nowHas = false;
      try { nowHas = target.isStateAffected(CFG.stateId); } catch { nowHas = false; }
      if (!nowHas) {
        showTamaToast(target, "fallbackDeathWhileTamaAttempt(apply)");
      }
    }

    // (B) タマ強打中に、タマ強打付与効果を含む技が命中
    if (CFG.showOnReapplyAction && hadTamaBefore && mayAddTama && hit) {
      showTamaToast(target, "actorTamaHitWhileTama(apply)");
      return;
    }
  };

  RT.reset();
})();
