/*:
 * @target MZ
 * @plugindesc v0.4.4 KT_AutoBugProbe（safeStringify＋ログローテーション＋イベント/変数差分＋日本語msg＋アラート＋自動フォールバック＋コンソールミラー(バッチ)）
 * @author ChatGPT
 *
 * @param enabled
 * @text 有効化
 * @type boolean
 * @default true
 *
 * @param enabledSwitchId
 * @text 有効化スイッチID（0=常に有効）
 * @type switch
 * @default 0
 * @desc 指定スイッチがONの間だけログ出力等を行います。0なら常に有効です。
 *
 * @param maxRecent
 * @text 直近行動ログ保持数（メモリ）
 * @type number
 * @min 10
 * @max 2000
 * @default 160
 *
 * @param fileSubFolder
 * @text ログ保存サブフォルダ名
 * @type string
 * @default kt_buglogs
 *
 * @param singleFileName
 * @text 単一ログファイル名（固定）
 * @type string
 * @default buglog.jsonl
 *
 * @param fileMaxLines
 * @text ファイル最大行数（0で無制限）
 * @type number
 * @min 0
 * @max 200000
 * @default 5000
 * @desc 行数を超えたらローテーション。重い全文トリムはしません。
 *
 * @param fileMaxBytes
 * @text ファイル最大サイズ(byte)（0で無制限）
 * @type number
 * @min 0
 * @default 0
 * @desc サイズを超えたらローテーション（statは間引き実行）
 *
 * @param fileMaxFiles
 * @text 最大保持ファイル数（current含む）
 * @type number
 * @min 1
 * @max 200
 * @default 3
 * @desc 例:3なら current + archive2個 を保持（古い順に削除）
 *
 * @param fileTrimCheckInterval
 * @text ローテ判定間隔（書き込み回数）
 * @type number
 * @min 1
 * @max 10000
 * @default 200
 * @desc fileMaxBytes の判定や、メタ保存の間引きにも使います
 *
 * @param trySaveFolderFirst
 * @text saveフォルダ優先（NW.js時）
 * @type boolean
 * @default true
 *
 * @param useLocalStorageFallback
 * @text localStorageフォールバックを使う
 * @type boolean
 * @default true
 *
 * @param localStorageKey
 * @text localStorageキー名
 * @type string
 * @default KT_AutoBugProbe_Log
 *
 * @param localStorageMaxLines
 * @text localStorage保持行数（概算）
 * @type number
 * @min 50
 * @max 5000
 * @default 800
 *
 * @param interceptConsoleWarnError
 * @text console.warn/errorも記録
 * @type boolean
 * @default true
 *
 * @param dumpToConsoleWhenNoFile
 * @text ファイル不可時はconsoleにも出す
 * @type boolean
 * @default true
 *
 * @param alsoDumpImportantToConsole
 * @text 重要イベントは常にconsoleにも出す（オブジェクト）
 * @type boolean
 * @default false
 *
 * @param consoleMirrorMode
 * @text コンソールミラー（バッチ）
 * @type select
 * @option しない
 * @value none
 * @option 重要のみ（おすすめ）
 * @value important
 * @option すべて（msg行のみ）
 * @value msg
 * @option すべて（オブジェクト）
 * @value all
 * @default none
 *
 * @param consoleMirrorIntervalMs
 * @text コンソールミラー間隔(ms)
 * @type number
 * @min 0
 * @max 2000
 * @default 200
 *
 * @param consoleMirrorMaxPerFlush
 * @text 1回の最大出力件数
 * @type number
 * @min 1
 * @max 500
 * @default 50
 *
 * @param enableHumanMessage
 * @text 人間向け要約(msg)を付与
 * @type boolean
 * @default true
 *
 * @param humanMessageMaxChars
 * @text msg最大文字数
 * @type number
 * @min 80
 * @max 2000
 * @default 320
 *
 * @param applyLogMode
 * @text 戦闘行動ログ出力（apply）
 * @type select
 * @option なし
 * @value none
 * @option 結果のみ
 * @value end
 * @option 開始+結果
 * @value begin_end
 * @default end
 *
 * @param applyLogOnlyInBattle
 * @text 戦闘中のみ行動ログを出す
 * @type boolean
 * @default true
 *
 * @param logBattleStartPartySnapshot
 * @text 戦闘開始時：味方の装備/能力値を記録
 * @type boolean
 * @default true
 *
 * @param battleStartSnapshotDetail
 * @text 戦闘開始時：詳細レベル
 * @type select
 * @option 基本（装備＋最終能力値）
 * @value basic
 * @option 詳細（基礎/加算/バフも）
 * @value detailed
 * @default basic
 *
 * @param includeStateIdsInApplyLog
 * @text 行動ログにステートID一覧を含める
 * @type boolean
 * @default true
 *
 * @param maxStateIds
 * @text ステートID最大数
 * @type number
 * @min 0
 * @max 200
 * @default 40
 *
 * @param includeDamageBreakdown
 * @text ダメージ内訳を記録（makeDamageValue）
 * @type boolean
 * @default true
 *
 * @param breakdownLevel
 * @text 内訳レベル
 * @type select
 * @option 基本（内訳＋倍率）
 * @value basic
 * @option 詳細（基本＋a/b主要パラメータ）
 * @value params
 * @default basic
 *
 * @param logInterpreterEvents
 * @text マップ/イベントログを有効化
 * @type boolean
 * @default true
 *
 * @param logEventBeginEnd
 * @text イベント開始/終了を記録
 * @type boolean
 * @default true
 *
 * @param logEventCommands
 * @text 実行コマンドの要約を記録
 * @type select
 * @option なし
 * @value none
 * @option 重要コマンドのみ（※条件分岐は除外）
 * @value important
 * @option 全コマンド
 * @value all
 * @default important
 *
 * @param commandParamMaxChars
 * @text コマンド引数の最大文字数（要約）
 * @type number
 * @min 50
 * @max 2000
 * @default 240
 *
 * @param logVariables
 * @text 変数変更を記録
 * @type boolean
 * @default true
 *
 * @param variableLogSource
 * @text 変数ログ取得元
 * @type select
 * @option command122差分（推奨）
 * @value command122
 * @option setValueフック（互換）
 * @value setValue
 * @option 両方（重複回避あり）
 * @value both
 * @default command122
 *
 * @param watchVariableIds
 * @text 監視する変数ID（空=全部）
 * @type string
 * @default
 * @desc 例: 1,2,10-20
 *
 * @param includeUnchangedInVarSet
 * @text 監視IDの変化なしも出す（command122）
 * @type boolean
 * @default false
 *
 * @param includeUnchangedMaxIds
 * @text 変化なしも出す上限ID数（安全）
 * @type number
 * @min 1
 * @max 200
 * @default 30
 *
 * @param logSwitches
 * @text スイッチ変更を記録
 * @type boolean
 * @default true
 *
 * @param switchLogSource
 * @text スイッチログ取得元
 * @type select
 * @option command121差分（推奨）
 * @value command121
 * @option setValueフック（互換）
 * @value setValue
 * @option 両方（重複回避あり）
 * @value both
 * @default command121
 *
 * @param watchSwitchIds
 * @text 監視するスイッチID（空=全部）
 * @type string
 * @default
 * @desc 例: 1,2,10-20
 *
 * @param logSelfSwitches
 * @text セルフスイッチ変更を記録
 * @type boolean
 * @default true
 *
 * @param selfSwitchLogSource
 * @text セルフスイッチログ取得元
 * @type select
 * @option command123差分（推奨）
 * @value command123
 * @option setValueフック（互換）
 * @value setValue
 * @option 両方（重複回避あり）
 * @value both
 * @default command123
 *
 * @param valuePreviewMaxChars
 * @text 値プレビュー最大文字数（変数など）
 * @type number
 * @min 50
 * @max 4000
 * @default 400
 *
 * @param alertKind
 * @text アラート種別名(kind)
 * @type string
 * @default var_alert
 *
 * @param alertAlsoConsole
 * @text アラートは常にconsoleにも出す
 * @type boolean
 * @default true
 *
 * @param varAlerts
 * @text 変数アラート設定
 * @type struct<VarAlert>[]
 * @default []
 *
 * @command ShowLogInfo
 * @text ログ情報を表示
 * @desc 現在のログ出力先（mode/file/key）をコンソールに表示します。
 *
 * @command ClearLocalStorageLogs
 * @text localStorageログ削除
 * @desc localStorageフォールバックログを削除します。
 *
 * @command ClearFileLogs
 * @text ファイルログを空にする
 * @desc ファイルモード時、ログファイルを空にして先頭にboot行を追加します（アーカイブは消しません）。
 *
 * @help
 * ■主な用途
 * - バグ再現が難しい箇所（変数、イベント、戦闘ダメージ等）を「いつ・どこで・何が起きたか」で追えるログにします。
 * - JSONL（1行=1JSON）で保存しつつ、人間が読める日本語要約 msg を付与します。
 *
 * ■v0.4.4 変更点
 * - 戦闘開始時に、参加した味方の装備品と能力値をスナップショットとして記録。
 *
 * ■v0.4.2 変更点
 * - event_begin の場所情報が「マップ?」になりやすい問題を軽減（setup後に記録）。
 * - 「重要コマンドのみ」モードでは条件分岐(111)を除外し、ログの埋まりを抑制。
 * - 変数操作(122)の「ランダム」表示を改善（例: ランダム 0～12）。
 *
 * ■注意（互換性）
 * - makeDamageValue を改造する他プラグインがある場合、順序により内訳ログが欠けることがあります。
 */
/*~struct~VarAlert:
 * @param varId
 * @text 変数ID
 * @type number
 * @min 1
 * @default 1
 *
 * @param op
 * @text 条件
 * @type select
 * @option 等しい（==）
 * @value eq
 * @option 以上（>=）
 * @value gte
 * @option 以下（<=）
 * @value lte
 * @option 変化した（old!=new）
 * @value changed
 * @default eq
 *
 * @param value
 * @text 比較値（changed以外）
 * @type string
 * @default 0
 *
 * @param label
 * @text ラベル（msgに表示）
 * @type string
 * @default
 *
 * @param oncePerSession
 * @text セッション中1回だけ
 * @type boolean
 * @default false
 */

(() => {
  "use strict";

  const PLUGIN_NAME = "KT_AutoBugProbe";
  const p = PluginManager.parameters(PLUGIN_NAME);

  const ENABLED = String(p.enabled) === "true";
  if (!ENABLED) return;


  const ENABLE_SWITCH_ID = Number(p.enabledSwitchId || 0);

  function isRuntimeEnabled() {
    if (!ENABLE_SWITCH_ID || ENABLE_SWITCH_ID <= 0) return true;
    try {
      if (typeof $gameSwitches === "undefined" || !$gameSwitches) return false;
      return !!$gameSwitches.value(ENABLE_SWITCH_ID);
    } catch (_) {
      return false;
    }
  }
  const MAX_RECENT = Number(p.maxRecent || 160);

  const FILE_SUB = String(p.fileSubFolder || "kt_buglogs");
  const SINGLE_FILE_NAME = String(p.singleFileName || "buglog.jsonl");
  const FILE_MAX_LINES = Number(p.fileMaxLines ?? 5000);
  const FILE_MAX_BYTES = Number(p.fileMaxBytes ?? 0);
  const FILE_MAX_FILES = Math.max(1, Number(p.fileMaxFiles ?? 3));
  const FILE_CHECK_INTERVAL = Math.max(1, Number(p.fileTrimCheckInterval || 200)); // 互換名（interval）

  const TRY_SAVE_FIRST = String(p.trySaveFolderFirst) === "true";

  const USE_LS = String(p.useLocalStorageFallback) === "true";
  const LS_KEY = String(p.localStorageKey || "KT_AutoBugProbe_Log");
  const LS_MAX_LINES = Number(p.localStorageMaxLines || 800);

  const INTERCEPT_CONSOLE = String(p.interceptConsoleWarnError) === "true";
  const DUMP_CONSOLE_NOFILE = String(p.dumpToConsoleWhenNoFile) === "true";
  const ALSO_IMPORTANT_CONSOLE = String(p.alsoDumpImportantToConsole) === "true";

  const CONSOLE_MIRROR_MODE = String(p.consoleMirrorMode || "none"); // none | important | msg | all
  const CONSOLE_MIRROR_INTERVAL = Math.max(0, Number(p.consoleMirrorIntervalMs || 200));
  const CONSOLE_MIRROR_MAX = Math.max(1, Number(p.consoleMirrorMaxPerFlush || 50));

  const ENABLE_HUMAN_MSG = String(p.enableHumanMessage ?? "true") === "true";
  const HUMAN_MSG_MAX = Number(p.humanMessageMaxChars || 320);

  const APPLY_LOG_MODE = String(p.applyLogMode || "end"); // none | end | begin_end
  const APPLY_ONLY_BATTLE = String(p.applyLogOnlyInBattle) === "true";
  const LOG_BATTLE_START_SNAPSHOT = String(p.logBattleStartPartySnapshot ?? "true") === "true";
  const BATTLE_START_SNAPSHOT_DETAIL = String(p.battleStartSnapshotDetail || "basic"); // basic | detailed

  const INCLUDE_STATE_IDS = String(p.includeStateIdsInApplyLog) === "true";
  const MAX_STATE_IDS = Number(p.maxStateIds || 40);

  const INCLUDE_BREAKDOWN = String(p.includeDamageBreakdown) === "true";
  const BREAKDOWN_LEVEL = String(p.breakdownLevel || "basic"); // basic | params

  const LOG_INTERPRETER = String(p.logInterpreterEvents) === "true";
  const LOG_EVENT_BEGIN_END = String(p.logEventBeginEnd) === "true";
  const LOG_EVENT_COMMANDS = String(p.logEventCommands || "important"); // none | important | all
  const CMD_PARAM_MAX = Number(p.commandParamMaxChars || 240);

  const LOG_VARS = String(p.logVariables) === "true";
  const VAR_SOURCE = String(p.variableLogSource || "command122"); // command122 | setValue | both
  const WATCH_VAR_RAW = String(p.watchVariableIds || "").trim();
  const INCLUDE_UNCHANGED_VAR = String(p.includeUnchangedInVarSet) === "true";
  const INCLUDE_UNCHANGED_MAX_IDS = Number(p.includeUnchangedMaxIds || 30);

  const LOG_SWITCHES = String(p.logSwitches) === "true";
  const SW_SOURCE = String(p.switchLogSource || "command121"); // command121 | setValue | both
  const WATCH_SWITCH_RAW = String(p.watchSwitchIds || "").trim();

  const LOG_SELF_SWITCHES = String(p.logSelfSwitches) === "true";
  const SS_SOURCE = String(p.selfSwitchLogSource || "command123"); // command123 | setValue | both

  const VALUE_PREVIEW_MAX = Number(p.valuePreviewMaxChars || 400);

  const ALERT_KIND = String(p.alertKind || "var_alert");
  const ALERT_ALSO_CONSOLE = String(p.alertAlsoConsole) === "true";

  const IMPORTANT_KINDS = new Set([
    "window_error",
    "unhandledrejection",
    "apply_exception",
    "suspicious_zero_damage",
    "console_error",
    ALERT_KIND,
  ]);

  // ----------------------------
  // helpers
  // ----------------------------
  function nowIso() { return new Date().toISOString(); }

  function makeSessionId() {
    return Date.now().toString(36) + "_" + Math.floor(Math.random() * 1e9).toString(36);
  }
  const SESSION = makeSessionId();

  function safeString(x) {
    try {
      if (x == null) return null;
      if (typeof x === "string") return x;
      if (x && x.stack) return String(x.stack);
      return String(x);
    } catch (_) {
      return "[unstringifiable]";
    }
  }

  function safeStringify(obj) {
    const seen = new WeakSet();
    const maxKeyLen = 200;

    function replacer(key, value) {
      try {
        if (typeof key === "string" && key.length > maxKeyLen) key = key.slice(0, maxKeyLen) + "...";

        if (typeof value === "bigint") return value.toString() + "n";
        if (typeof value === "function") return `[Function ${value.name || "anonymous"}]`;
        if (value === undefined) return "[undefined]";

        if (value instanceof Error) {
          return { name: value.name, message: value.message, stack: value.stack };
        }

        if (value && typeof value === "object") {
          if (seen.has(value)) return "[Circular]";
          seen.add(value);
        }
        return value;
      } catch (e) {
        return `[unstringifiable:${safeString(e)}]`;
      }
    }

    try {
      return JSON.stringify(obj, replacer);
    } catch (e) {
      try {
        return JSON.stringify({ kind: "stringify_failed", at: nowIso(), session: SESSION, error: safeString(e) });
      } catch (_) {
        return `{"kind":"stringify_failed","session":"${SESSION}"}`;
      }
    }
  }

  function sceneName() {
    const s = SceneManager && SceneManager._scene;
    if (!s || !s.constructor) return null;
    return s.constructor.name || null;
  }

  function isInBattle() {
    const s = SceneManager && SceneManager._scene;
    return !!(s && s.constructor && s.constructor.name === "Scene_Battle");
  }

  function battleMeta() {
    try {
      return {
        troopTurn: ($gameTroop && typeof $gameTroop.turnCount === "function") ? $gameTroop.turnCount() : null,
        partySteps: ($gameParty && typeof $gameParty.steps === "function") ? $gameParty.steps() : null,
      };
    } catch (_) {
      return {};
    }
  }

  function truncateStr(s, max) {
    if (s == null) return s;
    s = String(s);
    if (s.length <= max) return s;
    return s.slice(0, max) + `...(${s.length})`;
  }

  function valuePreview(v) {
    try {
      if (v == null) return null;
      const t = typeof v;
      if (t === "number" || t === "boolean") return v;
      if (t === "bigint") return v.toString() + "n";
      if (t === "string") return truncateStr(v, VALUE_PREVIEW_MAX);
      const js = safeStringify(v);
      return truncateStr(js, VALUE_PREVIEW_MAX);
    } catch (_) {
      return "[unpreviewable]";
    }
  }

  function parseIdSet(spec) {
    const s = String(spec || "").trim();
    if (!s) return null; // null means "all"
    const out = new Set();
    const parts = s.split(",").map(x => x.trim()).filter(Boolean);
    for (const part of parts) {
      const m = part.match(/^(\d+)\s*-\s*(\d+)$/);
      if (m) {
        const a = Number(m[1]), b = Number(m[2]);
        const lo = Math.min(a, b), hi = Math.max(a, b);
        for (let i = lo; i <= hi; i++) out.add(i);
      } else if (/^\d+$/.test(part)) {
        out.add(Number(part));
      }
    }
    return out.size ? out : null;
  }
  const WATCH_VAR_SET = parseIdSet(WATCH_VAR_RAW);
  const WATCH_SWITCH_SET = parseIdSet(WATCH_SWITCH_RAW);

  function shouldWatchId(setOrNull, id) {
    if (!setOrNull) return true;
    return setOrNull.has(id);
  }

  function mapNameById(mapId) {
    try {
      if (typeof $dataMapInfos !== "undefined" && $dataMapInfos && $dataMapInfos[mapId]) {
        return $dataMapInfos[mapId].name ?? null;
      }
    } catch (_) {}
    return null;
  }

  function mapDisplayName() {
    try {
      if ($gameMap && typeof $gameMap.displayName === "function") {
        return $gameMap.displayName() || null;
      }
    } catch (_) {}
    return null;
  }

  function eventNameOnCurrentMap(eventId) {
    try {
      if (typeof $dataMap !== "undefined" && $dataMap && $dataMap.events && $dataMap.events[eventId]) {
        return $dataMap.events[eventId].name ?? null;
      }
    } catch (_) {}
    return null;
  }

  function commonEventNameById(commonEventId) {
    try {
      if (typeof $dataCommonEvents !== "undefined" && $dataCommonEvents && $dataCommonEvents[commonEventId]) {
        return $dataCommonEvents[commonEventId].name ?? null;
      }
    } catch (_) {}
    return null;
  }

  function summarizeParams(params) {
    try {
      const js = safeStringify(params);
      return truncateStr(js, CMD_PARAM_MAX);
    } catch (_) {
      return truncateStr(String(params), CMD_PARAM_MAX);
    }
  }

  // ----------------------------
  // Human message (日本語)
  // ----------------------------
  function jpOnOff(v) { return v ? "ON" : "OFF"; }
  function oneLineJp(x, max=240) {
    const s = (x == null) ? "" : String(x);
    return truncateStr(s.replace(/\s+/g, " ").trim(), max);
  }

  function humanCtxJp(ctx) {
    if (!ctx) return "場所不明";
    const mapLabel = ctx.mapDisplayName || ctx.mapName || "";
    const evLabel  = ctx.eventName || "";
    const ceLabel  = ctx.commonEventName || "";

    const map = (ctx.mapId != null)
      ? `マップ${ctx.mapId}${mapLabel ? `「${mapLabel}」` : ""}`
      : "マップ?";
    const ev  = (ctx.eventId != null)
      ? `イベント${ctx.eventId}${evLabel ? `「${evLabel}」` : ""}`
      : "イベント?";
    const ce  = (ctx.commonEventId != null)
      ? `コモン${ctx.commonEventId}${ceLabel ? `「${ceLabel}」` : ""}`
      : "";
    const idx = (ctx.index != null) ? `コマンド#${ctx.index}` : "";

    return [map, ev, ce, idx].filter(Boolean).join(" / ");
  }

  function codeNameJp(code) {
    const m = {
      101:"文章の表示", 102:"選択肢の表示", 103:"数値入力", 104:"アイテム選択", 105:"文章スクロール",
      111:"条件分岐", 112:"ループ", 113:"ループの中断", 115:"イベント処理の中断",
      117:"コモンイベント", 121:"スイッチの操作", 122:"変数の操作", 123:"セルフスイッチの操作",
      201:"場所移動", 202:"乗り物位置設定", 203:"イベント位置設定", 205:"移動ルートの設定",
      231:"ピクチャの表示", 232:"ピクチャの移動", 233:"ピクチャの回転", 234:"ピクチャの色調変更",
      241:"BGM演奏", 242:"BGM停止", 243:"BGS演奏", 244:"BGS停止", 245:"ME演奏", 246:"SE演奏",
      301:"戦闘の処理", 302:"ショップの処理", 303:"名前入力", 331:"アクターのHP増減", 332:"アクターのMP増減",
      355:"スクリプト", 655:"スクリプト（続き）",
      356:"プラグインコマンド(MV互換)", 357:"プラグインコマンド(MZ)",
    };
    return m[code] || `コード${code}`;
  }

  // RPG Maker MZ: command122 params are typically [start, end, opType, operandType, operand1, operand2]
  // operandType: 0=定数,1=変数,2=ランダム,3=ゲームデータ,4=スクリプト
  function op122ToJp(params) {
    if (!Array.isArray(params)) return "";
    const start = Number(params[0]), end = Number(params[1]);
    const opType = Number(params[2]), rhsType = Number(params[3]);

    const opNames = ["代入", "加算", "減算", "乗算", "除算", "剰余"];
    const rhsNames = ["定数", "変数", "ランダム", "ゲームデータ", "スクリプト"];

    const range = (Number.isFinite(start) && Number.isFinite(end))
      ? (start === end ? `V${start}` : `V${Math.min(start,end)}～V${Math.max(start,end)}`)
      : "V?";

    const op = opNames[opType] || `操作${opType}`;
    const rhsName = rhsNames[rhsType] || `右辺${rhsType}`;

    let rhsText = "";
    if (rhsType === 0) {
      rhsText = String(params[4]);
    } else if (rhsType === 1) {
      rhsText = `V${String(params[4])}`;
    } else if (rhsType === 2) {
      const a = params[4], b = params[5];
      rhsText = `${String(a)}～${String(b)}`;
    } else if (rhsType === 3) {
      rhsText = truncateStr(String(params[4]), 80);
    } else if (rhsType === 4) {
      rhsText = truncateStr(String(params[4]), 80);
    } else {
      rhsText = truncateStr(String(params[4]), 80);
    }

    return `${range} を ${op}（${rhsName} ${rhsText}）`;
  }

  function op121ToJp(params) {
    if (!Array.isArray(params)) return "";
    const start = Number(params[0]), end = Number(params[1]);
    const v = !!params[2];
    const range = (Number.isFinite(start) && Number.isFinite(end))
      ? (start === end ? `S${start}` : `S${Math.min(start,end)}～S${Math.max(start,end)}`)
      : "S?";
    return `${range} を ${jpOnOff(v)}`;
  }

  function op123ToJp(params) {
    if (!Array.isArray(params)) return "";
    const letter = params[0];
    const v = !!params[1];
    return `セルフ${letter} を ${jpOnOff(v)}`;
  }

  function alertRuleJp(rule) {
    const op = rule.op;
    const v = (op === "changed") ? "" : ` ${rule.valueRaw}`;
    const label = rule.label ? `（${rule.label}）` : "";
    const opJp =
      op === "eq" ? "=="
      : op === "gte" ? ">="
      : op === "lte" ? "<="
      : op === "changed" ? "変化"
      : op;
    return `V${rule.varId} ${opJp}${v}${label}`;
  }

  function makeHumanMessage(kind, payload, entry) {
    const scene = entry?.scene || sceneName() || "";
    const b = entry?.battle || {};
    const turn = (b && b.troopTurn != null) ? `${b.troopTurn}ターン目` : "";

    if (kind === "event_begin") return `【イベント開始】${humanCtxJp(payload?.ctx)}（シーン:${scene}）`;
    if (kind === "event_end") return `【イベント終了】${humanCtxJp(payload?.ctx)}（シーン:${scene}）`;

    if (kind === "event_command") {
      const ctx = payload?.ctx;
      const code = payload?.cmd?.code;
      const name = codeNameJp(code);
      const raw = payload?.cmd?.parametersRaw;
      let detail = "";
      if (code === 122) detail = op122ToJp(raw);
      else if (code === 121) detail = op121ToJp(raw);
      else if (code === 123) detail = op123ToJp(raw);
      else if (code === 355 || code === 655) detail = `内容:${oneLineJp(payload?.cmd?.parameters, 120)}`;
      else if (code === 357) detail = `引数:${oneLineJp(payload?.cmd?.parameters, 160)}`;
      return `【実行】${humanCtxJp(ctx)} / ${name}${detail ? ` / ${detail}` : ""}`;
    }

    if (kind === "var_set") {
      const ctx = payload?.ctx;
      const changes = payload?.changes || [];
      const chText = changes.slice(0, 8).map(c => {
        const same = !c.changed;
        return `V${c.varId}:${oneLineJp(c.old,60)}→${oneLineJp(c.new,60)}${same ? "（変化なし）" : ""}`;
      }).join(" / ");
      const opText = Array.isArray(payload?.opRaw) ? op122ToJp(payload.opRaw) : "";
      return `【変数】${humanCtxJp(ctx)} / ${chText}${changes.length>8 ? ` …(+${changes.length-8})` : ""}${opText ? ` / 操作:${opText}` : ""}`;
    }

    if (kind === "switch_set") {
      const ctx = payload?.ctx;
      const changes = payload?.changes || [];
      const chText = changes.slice(0, 12).map(c => `S${c.switchId}:${jpOnOff(c.old)}→${jpOnOff(c.new)}`).join(" / ");
      const opText = Array.isArray(payload?.opRaw) ? op121ToJp(payload.opRaw) : "";
      return `【スイッチ】${humanCtxJp(ctx)} / ${chText}${changes.length>12 ? ` …(+${changes.length-12})` : ""}${opText ? ` / 操作:${opText}` : ""}`;
    }

    if (kind === "selfswitch_set") {
      const ctx = payload?.ctx;
      const key = payload?.key;
      const k = Array.isArray(key) ? `(${key.join(",")})` : oneLineJp(key, 60);
      const opText = Array.isArray(payload?.opRaw) ? op123ToJp(payload.opRaw) : "";
      return `【セルフS】${humanCtxJp(ctx)} / key=${k} / ${jpOnOff(payload?.old)}→${jpOnOff(payload?.new)}${opText ? ` / 操作:${opText}` : ""}`;
    }

    if (kind === "apply_end") {
      const subj = payload?.subject?.name || "？";
      const item = payload?.item?.name || "？";
      const tgt  = payload?.targetAfter?.name || payload?.targetBefore?.name || "？";
      const dmg  = payload?.result?.hpDamage;
      const crit = payload?.result?.critical ? "あり" : "なし";
      return `【戦闘】${turn} / ${subj}が「${item}」→ ${tgt} / HPダメ:${dmg} / クリティカル:${crit}`;
    }

    if (kind === "suspicious_zero_damage") {
      const subj = payload?.subject?.name || "？";
      const item = payload?.item?.name || "？";
      const tgt  = payload?.targetAfter?.name || payload?.targetBefore?.name || "？";
      return `【注意】命中したのにHPダメージ0 / ${subj}「${item}」→ ${tgt}`;
    }

    if (kind === "battle_start_snapshot") {
      const troop = payload?.troop?.name || (payload?.troop?.id != null ? `トループ${payload.troop.id}` : "トループ?");
      const actors = (payload?.party?.actors || []).map(a => {
        const nm = a?.name || "？";
        const lv = (a?.level != null) ? `Lv${a.level}` : "";
        return `${nm}${lv ? " " + lv : ""}`;
      }).join(" / ");
      const enemies = (payload?.enemies || []).map(e => e?.name || "？").join(" / ");
      return `【戦闘開始】${troop} / 味方:${actors || "なし"} / 敵:${enemies || "なし"}`;
    }

    if (kind === ALERT_KIND) {
      const ctx = payload?.ctx;
      const rule = payload?.rule;
      const oldV = payload?.old;
      const newV = payload?.new;
      return `【アラート】${humanCtxJp(ctx)} / ${alertRuleJp(rule)} / ${oneLineJp(oldV,80)}→${oneLineJp(newV,80)}`;
    }

    if (kind === "window_error") return `【エラー】${oneLineJp(payload?.message, 180)}（${payload?.filename}:${payload?.lineno}）`;
    if (kind === "unhandledrejection") return `【エラー】Promise例外: ${oneLineJp(payload?.reason, 200)}`;
    if (kind === "console_error") return `【エラー】console.error: ${oneLineJp((payload?.args||[]).join(" / "), 220)}`;

    return `【${kind}】シーン:${scene}`;
  }

  function attachHumanMsg(entry) {
    if (!ENABLE_HUMAN_MSG) return entry;
    try {
      const msg = makeHumanMessage(entry.kind, entry.payload, entry);
      entry.msg = truncateStr(msg, HUMAN_MSG_MAX);
    } catch (_) {}
    return entry;
  }

  // ----------------------------
  // Recent buffer
  // ----------------------------
  const recent = [];
  function pushRecent(obj) {
    recent.push({ t: Date.now(), ...obj });
    if (recent.length > MAX_RECENT) recent.splice(0, recent.length - MAX_RECENT);
  }

  // ----------------------------
  // Console Mirror (batched)
  // ----------------------------
  const ConsoleMirror = (() => {
    const mode = CONSOLE_MIRROR_MODE;
    if (mode === "none") return { enqueue: () => {} };

    const interval = CONSOLE_MIRROR_INTERVAL;
    const maxPer = CONSOLE_MIRROR_MAX;

    const queue = [];
    let timer = null;
    let dropped = 0;

    const QUEUE_CAP = Math.max(200, maxPer * 30);

    function mkLine(o) {
      const at = o.at || "";
      const kind = o.kind || "";
      const msg = o.msg ? ` ${o.msg}` : "";
      return `${at} ${kind}${msg}`;
    }

    function flush() {
      timer = null;
      if (!queue.length) return;

      const batch = queue.splice(0, maxPer);

      if (mode === "all") {
        for (const o of batch) console.log("[KT_AutoBugProbe]", o);
      } else {
        const lines = batch.map(mkLine);
        if (dropped > 0) {
          lines.unshift(`(consoleMirror: 直近で ${dropped} 件を出力待ち上限で省略)`);
          dropped = 0;
        }
        console.log("[KT_AutoBugProbe]\n" + lines.join("\n"));
      }

      if (queue.length) timer = setTimeout(flush, interval);
    }

    function enqueue(entry, important) {
      if (mode === "important" && !important) return;

      const store = (mode === "all") ? entry : { at: entry.at, kind: entry.kind, msg: entry.msg, session: entry.session };

      queue.push(store);
      if (queue.length > QUEUE_CAP) {
        const over = queue.length - QUEUE_CAP;
        queue.splice(0, over);
        dropped += over;
      }

      if (!timer) timer = setTimeout(flush, interval);
    }

    return { enqueue };
  })();

  // ----------------------------
  // Writer with auto fallback + rotation
  // ----------------------------
  const Writer = (() => {
    const isNw = Utils.isNwjs && Utils.isNwjs();
    let mode = "console"; // file_save | file_datapath | localStorage | console
    let filePath = null;
    let outDir = null;

    let fs = null;
    let path = null;

    let fileWriteCount = 0;

    // rotation meta
    let metaPath = null;
    let rotateIndex = 0;
    let currentLines = 0;
    let metaDirty = false;

    const META_SAVE_INTERVAL = Math.max(5, Math.min(FILE_CHECK_INTERVAL, 500));

    function consoleEmit(obj) {
      if (!DUMP_CONSOLE_NOFILE && mode !== "console") return;
      console.log("[KT_AutoBugProbe]", obj);
    }

    function tryInitNode() {
      if (!isNw) return false;
      try {
        fs = require("fs");
        path = require("path");
        return !!(fs && path);
      } catch (_) {
        fs = null;
        path = null;
        return false;
      }
    }

    function getSaveBaseDir() {
      try {
        if (StorageManager && typeof StorageManager.fileDirectoryPath === "function") {
          return StorageManager.fileDirectoryPath();
        }
        if (StorageManager && typeof StorageManager.localFileDirectoryPath === "function") {
          return StorageManager.localFileDirectoryPath();
        }
        if (StorageManager && typeof StorageManager.filePath === "function" && path) {
          const full = StorageManager.filePath(1);
          if (full) return path.dirname(full);
        }
      } catch (_) {}
      return null;
    }

    function getDataPathBaseDir() {
      try {
        return (nw && nw.App && nw.App.dataPath) ? nw.App.dataPath : null;
      } catch (_) {
        return null;
      }
    }

    function ensureDir(dir) {
      if (!fs.existsSync(dir)) fs.mkdirSync(dir, { recursive: true });
    }

    function parseFileBase() {
      const pp = path.parse(SINGLE_FILE_NAME);
      return { base: pp.name, ext: pp.ext || ".jsonl" };
    }

    function loadMeta() {
      if (!metaPath) return;
      try {
        if (!fs.existsSync(metaPath)) return;
        const txt = fs.readFileSync(metaPath, "utf8");
        const m = JSON.parse(txt);
        if (m && typeof m === "object") {
          rotateIndex = Number(m.rotateIndex || 0);
          currentLines = Number(m.currentLines || 0);
        }
      } catch (_) {}
    }

    function saveMeta(force = false) {
      if (!metaPath) return;
      if (!metaDirty && !force) return;
      try {
        const m = {
          version: "0.4.3",
          metaVersion: 1,
          rotateIndex,
          currentLines,
          lastWriteAt: nowIso(),
          file: SINGLE_FILE_NAME,
          session: SESSION,
        };
        fs.writeFileSync(metaPath, safeStringify(m), "utf8");
        metaDirty = false;
      } catch (_) {}
    }

    function cleanupArchives(dir) {
      const keepArchives = Math.max(0, FILE_MAX_FILES - 1);
      try {
        const { base, ext } = parseFileBase();
        const re = new RegExp(`^${escapeRegExp(base)}__(\\d+)${escapeRegExp(ext)}$`);
        const files = fs.readdirSync(dir);
        const archives = [];
        for (const f of files) {
          const m = f.match(re);
          if (m) archives.push({ name: f, idx: Number(m[1]) || 0 });
        }
        archives.sort((a, b) => a.idx - b.idx);

        const over = archives.length - keepArchives;
        if (over > 0) {
          for (let i = 0; i < over; i++) {
            try { fs.unlinkSync(path.join(dir, archives[i].name)); } catch (_) {}
          }
        }
      } catch (_) {}
    }

    function escapeRegExp(s) {
      return String(s).replace(/[.*+?^${}()|[\]\\]/g, "\\$&");
    }

    function rotateIfNeededBeforeAppend() {
      if (!(mode === "file_save" || mode === "file_datapath")) return;

      if (FILE_MAX_LINES > 0 && (currentLines + 1) > FILE_MAX_LINES) {
        doRotate("lines");
        return;
      }

      if (FILE_MAX_BYTES > 0 && (fileWriteCount % FILE_CHECK_INTERVAL) === 0) {
        try {
          const st = fs.statSync(filePath);
          if (st && st.size >= FILE_MAX_BYTES) {
            doRotate("bytes");
            return;
          }
        } catch (_) {}
      }
    }

    function doRotate(reason) {
      try {
        const { base, ext } = parseFileBase();
        rotateIndex = (rotateIndex || 0) + 1;

        const archived = path.join(outDir, `${base}__${String(rotateIndex).padStart(6, "0")}${ext}`);
        try {
          if (fs.existsSync(filePath)) {
            fs.renameSync(filePath, archived);
          }
        } catch (e) {
          console.warn("[KT_AutoBugProbe] rotate rename failed:", e);
        }

        cleanupArchives(outDir);

        const boot = safeStringify({ kind: "boot", at: nowIso(), session: SESSION, writer: "file_rotated", reason, version: "0.4.3" });
        fs.writeFileSync(filePath, boot + "\n", "utf8");

        currentLines = 1;
        metaDirty = true;
        saveMeta(true);
      } catch (e) {
        console.warn("[KT_AutoBugProbe] rotate failed:", e);
      }
    }

    function testWriteFixedFile(baseDir, kindLabel) {
      const dir = path.join(baseDir, FILE_SUB);
      const file = path.join(dir, SINGLE_FILE_NAME);
      const mfile = path.join(dir, `${SINGLE_FILE_NAME}.meta.json`);

      try {
        ensureDir(dir);

        outDir = dir;
        filePath = file;
        metaPath = mfile;

        loadMeta();

        if ((!currentLines || currentLines < 0) && fs.existsSync(filePath)) {
          try {
            const st = fs.statSync(filePath);
            if (st && st.size <= 1024 * 1024) {
              const txt = fs.readFileSync(filePath, "utf8");
              currentLines = txt.split(/\r?\n/).filter(l => l.length > 0).length;
            } else {
              currentLines = 0;
            }
          } catch (_) {
            currentLines = 0;
          }
        }

        fs.appendFileSync(filePath, safeStringify({ kind: "boot", at: nowIso(), session: SESSION, writer: kindLabel, version: "0.4.3" }) + "\n", "utf8");
        currentLines = (currentLines || 0) + 1;
        metaDirty = true;
        saveMeta(true);

        return true;
      } catch (_) {
        return false;
      }
    }

    function appendLocalStorageLine(obj) {
      const line = safeStringify(obj);
      let text = localStorage.getItem(LS_KEY) || "";
      text = text ? (text + "\n" + line) : line;

      const lines = text.split("\n");
      if (lines.length > LS_MAX_LINES) {
        text = lines.slice(lines.length - LS_MAX_LINES).join("\n");
      }

      try {
        localStorage.setItem(LS_KEY, text);
      } catch (_) {
        try {
          const lines2 = lines.slice(Math.max(0, lines.length - Math.floor(LS_MAX_LINES / 2)));
          localStorage.setItem(LS_KEY, lines2.join("\n"));
        } catch (_) {}
      }
    }

    function init() {
      if (tryInitNode()) {
        const saveDir = getSaveBaseDir();
        const dataDir = getDataPathBaseDir();

        if (TRY_SAVE_FIRST && saveDir && testWriteFixedFile(saveDir, "file_save")) {
          mode = "file_save";
        } else if (dataDir && testWriteFixedFile(dataDir, "file_datapath")) {
          mode = "file_datapath";
        } else if (!TRY_SAVE_FIRST && saveDir && testWriteFixedFile(saveDir, "file_save")) {
          mode = "file_save";
        }
      }

      if (mode === "console" && USE_LS) {
        try {
          if (typeof localStorage !== "undefined") {
            appendLocalStorageLine({ kind: "boot", at: nowIso(), session: SESSION, writer: "localStorage", version: "0.4.3" });
            mode = "localStorage";
          }
        } catch (_) {}
      }

      console.log(
        `[KT_AutoBugProbe] mode=${mode}` +
          (filePath ? ` file=${filePath}` : "") +
          (mode === "localStorage" ? ` key=${LS_KEY}` : "") +
          (CONSOLE_MIRROR_MODE !== "none" ? ` mirror=${CONSOLE_MIRROR_MODE}` : "") +
          ((mode === "file_save" || mode === "file_datapath")
            ? ` rotate(lines=${FILE_MAX_LINES || 0}, bytes=${FILE_MAX_BYTES || 0}, maxFiles=${FILE_MAX_FILES})`
            : "")
      );
    }

    function write(obj, important = false) {
      const alsoConsole = important && ALSO_IMPORTANT_CONSOLE;

      if (mode === "file_save" || mode === "file_datapath") {
        try {
          rotateIfNeededBeforeAppend();
          const line = safeStringify(obj) + "\n";
          fs.appendFileSync(filePath, line, "utf8");
          fileWriteCount++;
          currentLines = (currentLines || 0) + 1;

          metaDirty = true;
          if ((fileWriteCount % META_SAVE_INTERVAL) === 0) saveMeta(false);

          if (alsoConsole) console.log("[KT_AutoBugProbe]", obj);

          ConsoleMirror.enqueue(obj, important);
          return;
        } catch (e) {
          mode = USE_LS ? "localStorage" : "console";
          console.warn("[KT_AutoBugProbe] file logging disabled (append failed).", e);
        }
      }

      if (mode === "localStorage") {
        try {
          appendLocalStorageLine(obj);
          if (alsoConsole) console.log("[KT_AutoBugProbe]", obj);

          ConsoleMirror.enqueue(obj, important);
          return;
        } catch (_) {
          mode = "console";
        }
      }

      if (DUMP_CONSOLE_NOFILE || mode === "console") consoleEmit(obj);
      ConsoleMirror.enqueue(obj, important);
    }

    function info() {
      return {
        mode,
        session: SESSION,
        filePath,
        outDir,
        localStorageKey: LS_KEY,
        consoleMirror: CONSOLE_MIRROR_MODE,
        rotate: {
          fileMaxLines: FILE_MAX_LINES,
          fileMaxBytes: FILE_MAX_BYTES,
          fileMaxFiles: FILE_MAX_FILES,
          rotateIndex,
          currentLines,
          metaPath
        }
      };
    }

    function clearLocalStorage() {
      try { localStorage.removeItem(LS_KEY); return true; } catch (_) { return false; }
    }

    function clearFile() {
      if (!(mode === "file_save" || mode === "file_datapath")) return { ok: false, reason: "not_file_mode" };
      if (!fs || !filePath) return { ok: false, reason: "no_fs_or_path" };
      try {
        fs.writeFileSync(filePath, "", "utf8");
        fs.appendFileSync(filePath, safeStringify({ kind: "boot", at: nowIso(), session: SESSION, writer: "file_truncated", version: "0.4.3" }) + "\n", "utf8");
        fileWriteCount = 0;
        currentLines = 1;
        metaDirty = true;
        saveMeta(true);
        return { ok: true };
      } catch (e) {
        return { ok: false, reason: safeString(e) };
      }
    }

    init();
    return { write, info, clearLocalStorage, clearFile };
  })();

  function dump(kind, payload) {
    if (!isRuntimeEnabled()) return;
    const important = IMPORTANT_KINDS.has(kind);
    const entry = {
      kind,
      at: nowIso(),
      session: SESSION,
      scene: sceneName(),
      battle: battleMeta(),
      payload,
      recent: important ? recent.slice() : undefined,
    };
    Writer.write(attachHumanMsg(entry), important);
  }

  function writeLog(kind, payload) {
    if (!isRuntimeEnabled()) return;
    const entry = {
      kind,
      at: nowIso(),
      session: SESSION,
      scene: sceneName(),
      battle: battleMeta(),
      payload,
    };
    Writer.write(attachHumanMsg(entry), false);
  }

  // ----------------------------
  // Interpreter context
  // ----------------------------
  const ProbeCtx = {
    inInterpreter: false,
    ctx: null,
    currentCmdCode: null,
  };

  function interpreterCtx(ip, cmd) {
    try {
      const mapId = (typeof ip._mapId === "number") ? ip._mapId : ($gameMap ? $gameMap.mapId() : 0);
      const eventId = (typeof ip._eventId === "number") ? ip._eventId : 0;
      const depth = (typeof ip._depth === "number") ? ip._depth : null;
      const index = (typeof ip._index === "number") ? ip._index : null;
      const code = cmd ? cmd.code : (ip.currentCommand ? (ip.currentCommand() ? ip.currentCommand().code : null) : null);

      let pageIndex = null;
      try {
        if ($gameMap && eventId > 0 && $gameMap.event) {
          const ev = $gameMap.event(eventId);
          if (ev && typeof ev._pageIndex === "number") pageIndex = ev._pageIndex;
        }
      } catch (_) {}

      const commonEventId = (typeof ip._ktCommonEventId === "number") ? ip._ktCommonEventId : null;

      return {
        mapId,
        mapName: mapNameById(mapId),
        mapDisplayName: mapDisplayName(),
        eventId,
        eventName: eventId > 0 ? eventNameOnCurrentMap(eventId) : null,
        pageIndex,
        commonEventId,
        commonEventName: commonEventId ? commonEventNameById(commonEventId) : null,
        depth,
        index,
        code,
      };
    } catch (_) {
      return { mapId: null, eventId: null };
    }
  }

  function runtimeCtxFallback() {
    try {
      const mapId = $gameMap ? $gameMap.mapId() : null;
      return {
        mapId,
        mapName: mapId ? mapNameById(mapId) : null,
        mapDisplayName: mapDisplayName(),
        eventId: 0,
        eventName: null,
        pageIndex: null,
        commonEventId: null,
        commonEventName: null,
        depth: null,
        index: null,
        code: null,
      };
    } catch (_) {
      return null;
    }
  }

  // 「important」モードでは、条件分岐(111)を除外する（ログ過多になりやすい）
  const IMPORTANT_EVENT_CODES = new Set([
    101,102,103,104,105,
    // 111: 条件分岐（除外）
    112,113,115,
    117,
    121,122,123,
    201,202,203,
    205,
    231,232,233,234,
    241,242,243,244,245,246,
    301,302,303,331,332,
    355,655,
    356,357,
  ]);

  function shouldLogCmd(code) {
    if (!LOG_INTERPRETER) return false;
    if (LOG_EVENT_COMMANDS === "none") return false;
    if (LOG_EVENT_COMMANDS === "all") return true;
    return IMPORTANT_EVENT_CODES.has(code);
  }

  if (LOG_INTERPRETER && LOG_EVENT_BEGIN_END) {
    const _setup = Game_Interpreter.prototype.setup;
    Game_Interpreter.prototype.setup = function(list, eventId) {
      // v0.4.2: setup後に ctx を取る（マップ?/イベント? を減らす）
      const r = _setup.call(this, list, eventId);
      try {
        writeLog("event_begin", {
          ctx: interpreterCtx(this, null),
          listLen: Array.isArray(list) ? list.length : null,
        });
      } catch (_) {}
      return r;
    };

    const _terminate = Game_Interpreter.prototype.terminate;
    Game_Interpreter.prototype.terminate = function() {
      try {
        writeLog("event_end", { ctx: interpreterCtx(this, null) });
      } catch (_) {}
      return _terminate.call(this);
    };

    const _command117 = Game_Interpreter.prototype.command117;
    if (typeof _command117 === "function") {
      Game_Interpreter.prototype.command117 = function(params) {
        try {
          const commonEventId = params && params[0] != null ? Number(params[0]) : null;
          this._ktLastCommonEventId = Number.isFinite(commonEventId) ? commonEventId : null;
        } catch (_) {}
        return _command117.call(this, params);
      };
    }

    const _setupChild = Game_Interpreter.prototype.setupChild;
    if (typeof _setupChild === "function") {
      Game_Interpreter.prototype.setupChild = function(list, eventId) {
        const r = _setupChild.call(this, list, eventId);
        try {
          if (this._childInterpreter && this._ktLastCommonEventId != null) {
            this._childInterpreter._ktCommonEventId = this._ktLastCommonEventId;
            this._ktLastCommonEventId = null;
          }
        } catch (_) {}
        return r;
      };
    }
  }

  if (LOG_INTERPRETER) {
    const _executeCommand = Game_Interpreter.prototype.executeCommand;
    Game_Interpreter.prototype.executeCommand = function() {
      const cmd = this.currentCommand ? this.currentCommand() : null;
      const ctx = interpreterCtx(this, cmd);

      ProbeCtx.inInterpreter = true;
      ProbeCtx.ctx = ctx;
      ProbeCtx.currentCmdCode = cmd ? cmd.code : null;

      try {
        if (cmd && shouldLogCmd(cmd.code)) {
          writeLog("event_command", {
            ctx,
            cmd: {
              code: cmd.code,
              indent: cmd.indent ?? null,
              parameters: cmd.parameters ? summarizeParams(cmd.parameters) : null,
              parametersRaw: cmd.parameters ?? null,
            },
          });
        }
        return _executeCommand.call(this);
      } finally {
        ProbeCtx.inInterpreter = false;
        ProbeCtx.ctx = null;
        ProbeCtx.currentCmdCode = null;
      }
    };
  }

  // ----------------------------
  // Alerts
  // ----------------------------
  function parseStructArray(param) {
    try {
      const arr = JSON.parse(param || "[]");
      if (!Array.isArray(arr)) return [];
      return arr.map(s => {
        try { return JSON.parse(s); } catch (_) { return null; }
      }).filter(Boolean);
    } catch (_) {
      return [];
    }
  }

  const VAR_ALERT_RULES = parseStructArray(p.varAlerts).map(r => ({
    varId: Number(r.varId || 0),
    op: String(r.op || "eq"),
    valueRaw: String(r.value ?? ""),
    label: String(r.label || ""),
    oncePerSession: String(r.oncePerSession) === "true",
  })).filter(r => Number.isFinite(r.varId) && r.varId > 0);

  const firedAlertKeys = new Set();
  function makeAlertKey(rule) {
    return `${SESSION}|${rule.varId}|${rule.op}|${rule.valueRaw}|${rule.label}|${rule.oncePerSession}`;
  }

  function toNumberMaybe(v) {
    const n = Number(v);
    return Number.isFinite(n) ? n : null;
  }

  function coerceTarget(rule, newV) {
    const raw = rule.valueRaw;
    if (typeof newV === "boolean") {
      if (raw === "true" || raw === "1" || raw === "ON") return true;
      if (raw === "false" || raw === "0" || raw === "OFF") return false;
      return !!raw;
    }
    const n = toNumberMaybe(raw);
    if (typeof newV === "number") return (n != null) ? n : raw;
    return raw;
  }

  function matchAlert(rule, oldV, newV) {
    const op = rule.op;
    if (op === "changed") return oldV !== newV;

    const target = coerceTarget(rule, newV);
    const nNew = toNumberMaybe(newV);
    const nT = toNumberMaybe(target);

    if (op === "eq") {
      if (nNew != null && nT != null) return nNew === nT;
      return String(newV) === String(target);
    }
    if (op === "gte") {
      if (nNew == null || nT == null) return false;
      return nNew >= nT;
    }
    if (op === "lte") {
      if (nNew == null || nT == null) return false;
      return nNew <= nT;
    }
    return false;
  }

  function maybeEmitVarAlert(varId, oldV, newV, ctx) {
    if (!VAR_ALERT_RULES.length) return;

    for (const rule of VAR_ALERT_RULES) {
      if (rule.varId !== varId) continue;

      const key = makeAlertKey(rule);
      if (rule.oncePerSession && firedAlertKeys.has(key)) continue;

      if (matchAlert(rule, oldV, newV)) {
        firedAlertKeys.add(key);

        const payload = { ctx, varId, old: valuePreview(oldV), new: valuePreview(newV), rule };
        dump(ALERT_KIND, payload);

        if (ALERT_ALSO_CONSOLE) console.log("[KT_AutoBugProbe][ALERT]", payload);
      }
    }
  }

  // ----------------------------
  // Variables / Switches / SelfSwitches
  // ----------------------------
  function varSourceHas(kind) { return VAR_SOURCE === kind || VAR_SOURCE === "both"; }
  function swSourceHas(kind) { return SW_SOURCE === kind || SW_SOURCE === "both"; }
  function ssSourceHas(kind) { return SS_SOURCE === kind || SS_SOURCE === "both"; }

  function allowIncludeUnchanged(countTouched) {
    if (!INCLUDE_UNCHANGED_VAR) return false;
    if (!WATCH_VAR_SET) return false;
    if (countTouched > INCLUDE_UNCHANGED_MAX_IDS) return false;
    return true;
  }

  if (LOG_VARS && varSourceHas("command122") && typeof Game_Interpreter.prototype.command122 === "function") {
    const _command122 = Game_Interpreter.prototype.command122;
    Game_Interpreter.prototype.command122 = function(params) {
      const ctx = LOG_INTERPRETER ? interpreterCtx(this, this.currentCommand ? this.currentCommand() : null) : (ProbeCtx.ctx || null);

      const startId = params ? Number(params[0]) : NaN;
      const endId = params ? Number(params[1]) : NaN;

      const oldMap = new Map();
      let touchedCount = 0;

      try {
        if (Number.isFinite(startId) && Number.isFinite(endId) && $gameVariables) {
          const a = Math.min(startId, endId);
          const b = Math.max(startId, endId);
          for (let i = a; i <= b; i++) {
            if (!shouldWatchId(WATCH_VAR_SET, i)) continue;
            oldMap.set(i, $gameVariables.value(i));
            touchedCount++;
          }
        }
      } catch (_) {}

      const r = _command122.call(this, params);

      try {
        if ($gameVariables && oldMap.size > 0) {
          const changes = [];
          const includeUnchanged = allowIncludeUnchanged(touchedCount);

          for (const [i, oldV] of oldMap.entries()) {
            const newV = $gameVariables.value(i);
            const changed = oldV !== newV;
            if (changed || includeUnchanged) {
              changes.push({ varId: i, old: valuePreview(oldV), new: valuePreview(newV), changed });
              maybeEmitVarAlert(i, oldV, newV, ctx);
            }
          }

          if (changes.length > 0) {
            writeLog("var_set", {
              ctx,
              source: "command122",
              op: params ? summarizeParams(params) : null,
              opRaw: params ?? null,
              includeUnchanged,
              changes,
            });
          }
        }
      } catch (_) {}

      return r;
    };
  }

  if (LOG_VARS && varSourceHas("setValue")) {
    const _setVar = Game_Variables.prototype.setValue;
    Game_Variables.prototype.setValue = function(variableId, value) {
      const id = Number(variableId);
      const oldV = this.value(id);
      const r = _setVar.call(this, id, value);
      const newV = this.value(id);

      try {
        if (ProbeCtx.inInterpreter && ProbeCtx.currentCmdCode === 122) return r;
        if (!shouldWatchId(WATCH_VAR_SET, id)) return r;

        const changed = oldV !== newV;
        if (changed) {
          const ctx = ProbeCtx.ctx || runtimeCtxFallback();
          maybeEmitVarAlert(id, oldV, newV, ctx);
          writeLog("var_set", {
            ctx,
            source: "setValue",
            changes: [{ varId: id, old: valuePreview(oldV), new: valuePreview(newV), changed: true }],
          });
        }
      } catch (_) {}
      return r;
    };
  }

  if (LOG_SWITCHES && swSourceHas("command121") && typeof Game_Interpreter.prototype.command121 === "function") {
    const _command121 = Game_Interpreter.prototype.command121;
    Game_Interpreter.prototype.command121 = function(params) {
      const ctx = LOG_INTERPRETER ? interpreterCtx(this, this.currentCommand ? this.currentCommand() : null) : (ProbeCtx.ctx || null);
      const startId = params ? Number(params[0]) : NaN;
      const endId = params ? Number(params[1]) : NaN;
      const oldMap = new Map();

      try {
        if (Number.isFinite(startId) && Number.isFinite(endId) && $gameSwitches) {
          const a = Math.min(startId, endId);
          const b = Math.max(startId, endId);
          for (let i = a; i <= b; i++) {
            if (!shouldWatchId(WATCH_SWITCH_SET, i)) continue;
            oldMap.set(i, !!$gameSwitches.value(i));
          }
        }
      } catch (_) {}

      const r = _command121.call(this, params);

      try {
        if ($gameSwitches && oldMap.size > 0) {
          const changes = [];
          for (const [i, oldV] of oldMap.entries()) {
            const newV = !!$gameSwitches.value(i);
            if (oldV !== newV) changes.push({ switchId: i, old: oldV, new: newV });
          }
          if (changes.length > 0) {
            writeLog("switch_set", {
              ctx,
              source: "command121",
              op: params ? summarizeParams(params) : null,
              opRaw: params ?? null,
              changes,
            });
          }
        }
      } catch (_) {}
      return r;
    };
  }

  if (LOG_SWITCHES && swSourceHas("setValue")) {
    const _setSw = Game_Switches.prototype.setValue;
    Game_Switches.prototype.setValue = function(switchId, value) {
      const id = Number(switchId);
      const oldV = this.value(id);
      const r = _setSw.call(this, id, value);
      const newV = this.value(id);

      try {
        if (ProbeCtx.inInterpreter && ProbeCtx.currentCmdCode === 121) return r;
        if (!shouldWatchId(WATCH_SWITCH_SET, id)) return r;
        if (!!oldV !== !!newV) {
          const ctx = ProbeCtx.ctx || runtimeCtxFallback();
          writeLog("switch_set", {
            ctx,
            source: "setValue",
            changes: [{ switchId: id, old: !!oldV, new: !!newV }],
          });
        }
      } catch (_) {}
      return r;
    };
  }

  if (LOG_SELF_SWITCHES && ssSourceHas("command123") && typeof Game_Interpreter.prototype.command123 === "function") {
    const _command123 = Game_Interpreter.prototype.command123;
    Game_Interpreter.prototype.command123 = function(params) {
      const ctx = LOG_INTERPRETER ? interpreterCtx(this, this.currentCommand ? this.currentCommand() : null) : (ProbeCtx.ctx || null);
      let key = null;
      let oldV = null;

      try {
        const mapId = (typeof this._mapId === "number") ? this._mapId : ($gameMap ? $gameMap.mapId() : 0);
        const eventId = (typeof this._eventId === "number") ? this._eventId : 0;
        const letter = params ? params[0] : null;
        key = [mapId, eventId, letter];
        oldV = $gameSelfSwitches ? !!$gameSelfSwitches.value(key) : null;
      } catch (_) {}

      const r = _command123.call(this, params);

      try {
        if (key && $gameSelfSwitches) {
          const newV = !!$gameSelfSwitches.value(key);
          if (oldV !== null && oldV !== newV) {
            writeLog("selfswitch_set", {
              ctx,
              source: "command123",
              op: params ? summarizeParams(params) : null,
              opRaw: params ?? null,
              key,
              old: oldV,
              new: newV,
            });
          }
        }
      } catch (_) {}
      return r;
    };
  }

  if (LOG_SELF_SWITCHES && ssSourceHas("setValue")) {
    const _setSS = Game_SelfSwitches.prototype.setValue;
    Game_SelfSwitches.prototype.setValue = function(key, value) {
      let oldV = null;
      try { oldV = this.value(key); } catch (_) { oldV = null; }
      const r = _setSS.call(this, key, value);
      let newV = null;
      try { newV = this.value(key); } catch (_) { newV = null; }

      try {
        if (ProbeCtx.inInterpreter && ProbeCtx.currentCmdCode === 123) return r;
        if (!!oldV !== !!newV) {
          const ctx = ProbeCtx.ctx || runtimeCtxFallback();
          writeLog("selfswitch_set", {
            ctx,
            source: "setValue",
            key: Array.isArray(key) ? key.slice() : key,
            old: !!oldV,
            new: !!newV,
          });
        }
      } catch (_) {}
      return r;
    };
  }

  // ----------------------------
  // Battle helpers
  // ----------------------------
  function stateIdsOf(battler) {
    if (!INCLUDE_STATE_IDS) return null;
    try {
      if (!battler || !battler.states) return null;
      const ids = battler.states().map(s => s.id);
      if (MAX_STATE_IDS > 0 && ids.length > MAX_STATE_IDS) return ids.slice(0, MAX_STATE_IDS);
      return ids;
    } catch (_) {
      return null;
    }
  }

  function battlerInfo(b) {
    if (!b) return null;
    const isActor = b.isActor ? !!b.isActor() : null;
    const id = isActor ? (b.actorId ? b.actorId() : null) : (b.enemyId ? b.enemyId() : null);
    return {
      name: b.name ? b.name() : null,
      isActor,
      id,
      hp: (typeof b.hp === "number") ? b.hp : null,
      mp: (typeof b.mp === "number") ? b.mp : null,
      tp: (typeof b.tp === "number") ? b.tp : null,
      states: stateIdsOf(b),
    };
  }

  function battlerParams(b) {
    if (!b) return null;
    const pick = (k) => { try { const v = b[k]; return (typeof v === "number") ? v : null; } catch (_) { return null; } };
    return {
      mhp: pick("mhp"), mmp: pick("mmp"),
      atk: pick("atk"), def: pick("def"),
      mat: pick("mat"), mdf: pick("mdf"),
      agi: pick("agi"), luk: pick("luk"),
      hit: pick("hit"), eva: pick("eva"),
      cri: pick("cri"), cev: pick("cev"),
      mev: pick("mev"), mrf: pick("mrf"),
      cnt: pick("cnt"), grd: pick("grd"),
      pdr: pick("pdr"), mdr: pick("mdr"), rec: pick("rec"),
    };
  }

// ----------------------------
// Battle start snapshot (party equips + params)
// ----------------------------
function roundFloat(v, digits = 6) {
  try {
    if (typeof v !== "number") return v;
    const p = Math.pow(10, digits);
    return Math.round(v * p) / p;
  } catch (_) { return v; }
}

function equipSlotName(etypeId) {
  try {
    if (!$dataSystem || !$dataSystem.equipTypes) return null;
    return $dataSystem.equipTypes[etypeId] || null;
  } catch (_) { return null; }
}

function equipItemInfoBasic(item) {
  if (!item) return null;
  return { id: item.id ?? null, name: item.name ?? null };
}

function equipItemInfoDetailed(item) {
  if (!item) return null;
  let kind = "equip";
  try {
    if (DataManager && DataManager.isWeapon && DataManager.isWeapon(item)) kind = "weapon";
    else if (DataManager && DataManager.isArmor && DataManager.isArmor(item)) kind = "armor";
  } catch (_) {}
  return {
    kind,
    id: item.id ?? null,
    name: item.name ?? null,
    etypeId: item.etypeId ?? null,
    wtypeId: item.wtypeId ?? null,
    atypeId: item.atypeId ?? null,
    params8: Array.isArray(item.params) ? item.params.slice(0, 8) : null,
    traitsCount: Array.isArray(item.traits) ? item.traits.length : null,
    metaKeys: (item.meta && typeof item.meta === "object") ? Object.keys(item.meta).slice(0, 40) : null,
  };
}

function actorParam8Final(actor) {
  try {
    return {
      mhp: actor.mhp, mmp: actor.mmp,
      atk: actor.atk, def: actor.def,
      mat: actor.mat, mdf: actor.mdf,
      agi: actor.agi, luk: actor.luk,
    };
  } catch (_) { return null; }
}

function actorParam8Detailed(actor) {
  const names = ["mhp","mmp","atk","def","mat","mdf","agi","luk"];
  const out = {};
  try {
    for (let i = 0; i < 8; i++) {
      const key = names[i];
      const finalV = (actor && actor.param) ? actor.param(i) : null;
      const baseV = (actor && actor.paramBase) ? actor.paramBase(i) : null;
      const plusV = (actor && actor.paramPlus) ? actor.paramPlus(i) : null;
      let buff = null;
      try { buff = (actor && actor._buffs && typeof actor._buffs[i] === "number") ? actor._buffs[i] : null; } catch (_) { buff = null; }
      let buffRate = null;
      try { buffRate = (actor && actor.paramBuffRate) ? actor.paramBuffRate(i) : null; } catch (_) { buffRate = null; }
      out[key] = {
        final: (typeof finalV === "number") ? finalV : null,
        base: (typeof baseV === "number") ? baseV : null,
        plus: (typeof plusV === "number") ? plusV : null,
        buff,
        buffRate: (typeof buffRate === "number") ? roundFloat(buffRate, 6) : buffRate,
      };
    }
    return out;
  } catch (_) { return null; }
}

function actorXParams(actor) {
  const names = ["hit","eva","cri","cev","mev","mrf","cnt","hrg","mrg","trg"];
  const out = {};
  try {
    if (!actor || !actor.xparam) return null;
    for (let i = 0; i < 10; i++) out[names[i]] = roundFloat(actor.xparam(i), 6);
    return out;
  } catch (_) { return null; }
}

function actorSParams(actor) {
  const names = ["tgr","grd","rec","pha","mcr","tcr","pdr","mdr","fdr","exr"];
  const out = {};
  try {
    if (!actor || !actor.sparam) return null;
    for (let i = 0; i < 10; i++) out[names[i]] = roundFloat(actor.sparam(i), 6);
    return out;
  } catch (_) { return null; }
}

function actorEquipSnapshot(actor, detailed) {
  try {
    if (!actor) return null;
    const slots = (actor.equipSlots && Array.isArray(actor.equipSlots())) ? actor.equipSlots() : [];
    const equips = (actor.equips && Array.isArray(actor.equips())) ? actor.equips() : [];
    const out = [];
    for (let i = 0; i < Math.max(slots.length, equips.length); i++) {
      const etypeId = (typeof slots[i] === "number") ? slots[i] : null;
      const item = equips[i] || null;
      out.push({
        slotIndex: i,
        etypeId,
        slotName: etypeId != null ? equipSlotName(etypeId) : null,
        item: detailed ? equipItemInfoDetailed(item) : equipItemInfoBasic(item),
      });
    }
    return out;
  } catch (_) { return null; }
}

function actorBattleStartSnapshot(actor, detailLevel) {
  if (!actor) return null;
  const detailed = (detailLevel === "detailed");
  const info = battlerInfo(actor);
  const params = detailed ? actorParam8Detailed(actor) : actorParam8Final(actor);

  let classId = null, className = null, level = null;
  try { classId = (typeof actor._classId === "number") ? actor._classId : null; } catch (_) {}
  try { className = (actor.currentClass && actor.currentClass()) ? actor.currentClass().name : null; } catch (_) { className = null; }
  try { level = (typeof actor.level === "number") ? actor.level : null; } catch (_) { level = null; }

  const snap = {
    ...info,
    actorId: info?.id ?? (actor.actorId ? actor.actorId() : null),
    level,
    classId,
    className,
    params,
    equips: actorEquipSnapshot(actor, detailed),
  };

  if (detailed) {
    snap.xparams = actorXParams(actor);
    snap.sparams = actorSParams(actor);
    // battlerParams は getter による派生値も含む（参考）
    snap.derived = battlerParams(actor);
  }

  return snap;
}

function logBattleStartSnapshot() {
  if (!LOG_BATTLE_START_SNAPSHOT) return;

  const detail = BATTLE_START_SNAPSHOT_DETAIL;
  let troopId = null;
  try { troopId = (BattleManager && typeof BattleManager._troopId === "number") ? BattleManager._troopId : null; } catch (_) { troopId = null; }

  let troopName = null;
  try { troopName = (troopId != null && $dataTroops && $dataTroops[troopId]) ? ($dataTroops[troopId].name ?? null) : null; } catch (_) { troopName = null; }

  const actors = [];
  try {
    const members = ($gameParty && $gameParty.battleMembers) ? $gameParty.battleMembers() : [];
    for (const a of members) actors.push(actorBattleStartSnapshot(a, detail));
  } catch (_) {}

  let enemies = null;
  try {
    enemies = ($gameTroop && $gameTroop.members) ? $gameTroop.members().map(battlerInfo) : null;
  } catch (_) { enemies = null; }

  writeLog("battle_start_snapshot", {
    troop: { id: troopId, name: troopName },
    party: { actors },
    enemies,
  });
}

  function itemInfo(item) {
    if (!item) return null;
    let isSkill = null, isItem = null;
    try {
      if (DataManager && DataManager.isSkill) isSkill = !!DataManager.isSkill(item);
      if (DataManager && DataManager.isItem) isItem = !!DataManager.isItem(item);
    } catch (_) {}
    return {
      name: item.name ?? null,
      id: item.id ?? null,
      isSkill,
      isItem,
      damageType: item.damage ? (item.damage.type ?? null) : null,
      elementId: item.damage ? (item.damage.elementId ?? null) : null,
      variance: item.damage ? (item.damage.variance ?? null) : null,
      formula: item.damage ? (item.damage.formula ?? null) : null,
    };
  }

  // ----------------------------
  // Damage breakdown capture (wrap makeDamageValue)
  // ----------------------------
  const _makeDamageValue = Game_Action.prototype.makeDamageValue;
  Game_Action.prototype.makeDamageValue = function(target, critical) {
    if (!INCLUDE_BREAKDOWN) return _makeDamageValue.call(this, target, critical);

    const action = this;
    const item = action.item ? action.item() : null;

    if (!action._ktProbeDmgMap) action._ktProbeDmgMap = new WeakMap();

    const cap = {
      critical: !!critical,
      isPhysical: (action.isPhysical ? !!action.isPhysical() : null),
      isMagical: (action.isMagical ? !!action.isMagical() : null),
      item: itemInfo(item),
      baseValue: null,
      elementRate: null,
      varianceAmp: null,
      varianceDelta: null,
      guardDivObserved: null,
      afterElement: null,
      afterPdrMdr: null,
      afterRec: null,
      afterCritical: null,
      afterVariance: null,
      afterGuard: null,
      rounded: null,
      targetRates: null,
      guardDiv: null,
      finalValue: null,
      finalMinusRounded: null,
    };

    if (BREAKDOWN_LEVEL === "params") {
      cap.aParams = battlerParams(action.subject ? action.subject() : null);
      cap.bParams = battlerParams(target);
    }

    const orig = {
      evalDamageFormula: action.evalDamageFormula,
      calcElementRate: action.calcElementRate,
      applyCritical: action.applyCritical,
      applyVariance: action.applyVariance,
      applyGuard: action.applyGuard,
    };

    try {
      if (typeof orig.evalDamageFormula === "function") {
        action.evalDamageFormula = function(t) {
          const v = orig.evalDamageFormula.call(action, t);
          cap.baseValue = (typeof v === "number") ? v : v;
          return v;
        };
      }
      if (typeof orig.calcElementRate === "function") {
        action.calcElementRate = function(t) {
          const r = orig.calcElementRate.call(action, t);
          cap.elementRate = (typeof r === "number") ? r : r;
          return r;
        };
      }
      if (typeof orig.applyVariance === "function") {
        action.applyVariance = function(v, variance) {
          const before = v;
          const out = orig.applyVariance.call(action, v, variance);
          try {
            const amp = Math.floor(Math.max(Math.abs(before) * (Number(variance) || 0) / 100, 0));
            cap.varianceAmp = (typeof amp === "number") ? amp : null;
          } catch (_) { cap.varianceAmp = null; }
          if (typeof before === "number" && typeof out === "number") cap.varianceDelta = out - before;
          cap.afterVariance = out;
          return out;
        };
      }
      if (typeof orig.applyGuard === "function") {
        action.applyGuard = function(v, t) {
          const before = v;
          const out = orig.applyGuard.call(action, v, t);
          if (typeof before === "number" && typeof out === "number" && out !== 0) cap.guardDivObserved = before / out;
          cap.afterGuard = out;
          return out;
        };
      }

      const finalValue = _makeDamageValue.call(action, target, critical);
      cap.finalValue = finalValue;

      try {
        const guarding = (target && typeof target.isGuard === "function") ? !!target.isGuard() : null;
        const grd = (target && typeof target.grd === "number") ? target.grd : null;
        cap.targetRates = {
          guarding,
          grd,
          pdr: (target && typeof target.pdr === "number") ? target.pdr : null,
          mdr: (target && typeof target.mdr === "number") ? target.mdr : null,
          rec: (target && typeof target.rec === "number") ? target.rec : null,
        };
        if (guarding != null && grd != null) cap.guardDiv = guarding ? (2 * grd) : 1;
      } catch (_) { cap.targetRates = null; }

      try {
        let v = cap.baseValue;
        if (typeof v === "number") {
          const er = (typeof cap.elementRate === "number") ? cap.elementRate : 1;
          cap.afterElement = v * er;

          let afterPdrMdr = cap.afterElement;
          const tr = cap.targetRates || {};
          if (cap.isPhysical && typeof tr.pdr === "number") afterPdrMdr *= tr.pdr;
          if (cap.isMagical && typeof tr.mdr === "number") afterPdrMdr *= tr.mdr;
          cap.afterPdrMdr = afterPdrMdr;

          let afterRec = afterPdrMdr;
          if (v < 0 && tr && typeof tr.rec === "number") afterRec *= tr.rec;
          cap.afterRec = afterRec;

          let afterCritical = afterRec;
          if (cap.critical && typeof orig.applyCritical === "function") afterCritical = orig.applyCritical.call(action, afterRec);
          cap.afterCritical = afterCritical;

          if (cap.afterVariance == null) cap.afterVariance = afterCritical;
          if (cap.afterGuard == null) cap.afterGuard = cap.afterVariance;

          cap.rounded = Math.round(cap.afterGuard);
          if (typeof cap.finalValue === "number" && typeof cap.rounded === "number") cap.finalMinusRounded = cap.finalValue - cap.rounded;
        }
      } catch (_) {}

      action._ktProbeDmgMap.set(target, cap);
      return finalValue;
    } finally {
      try { action.evalDamageFormula = orig.evalDamageFormula; } catch (_) {}
      try { action.calcElementRate = orig.calcElementRate; } catch (_) {}
      try { action.applyVariance = orig.applyVariance; } catch (_) {}
      try { action.applyGuard = orig.applyGuard; } catch (_) {}
      try { action.applyCritical = orig.applyCritical; } catch (_) {}
    }
  };

  // ----------------------------
  // window errors
  // ----------------------------
  window.addEventListener("error", (ev) => {
    dump("window_error", {
      message: safeString(ev.message),
      filename: safeString(ev.filename),
      lineno: ev.lineno ?? null,
      colno: ev.colno ?? null,
      stack: ev.error && ev.error.stack ? safeString(ev.error.stack) : null,
    });
  });

  window.addEventListener("unhandledrejection", (ev) => {
    dump("unhandledrejection", { reason: safeString(ev.reason) });
  });

  // ----------------------------
  // console intercept（warn/errorをログ化。再帰ガード）
  // ----------------------------
  if (INTERCEPT_CONSOLE) {
    let _ktConsoleHookBusy = false;

    const _cerr = console.error.bind(console);
    console.error = (...args) => {
      if (!_ktConsoleHookBusy) {
        _ktConsoleHookBusy = true;
        try { dump("console_error", { args: args.map(safeString) }); }
        finally { _ktConsoleHookBusy = false; }
      }
      _cerr(...args);
    };

    const _cwarn = console.warn.bind(console);
    console.warn = (...args) => {
      if (!_ktConsoleHookBusy) {
        _ktConsoleHookBusy = true;
        try { writeLog("console_warn", { args: args.map(safeString) }); }
        finally { _ktConsoleHookBusy = false; }
      }
      _cwarn(...args);
    };
  }

  // ----------------------------
  // Battle apply logging
  // ----------------------------
  function writeApplyLog(kind, payload) {
    if (!isRuntimeEnabled()) return;
    if (APPLY_LOG_MODE === "none") return;
    if (APPLY_ONLY_BATTLE && !isInBattle()) return;

    const entry = {
      kind,
      at: nowIso(),
      session: SESSION,
      scene: sceneName(),
      battle: battleMeta(),
      payload,
    };
    Writer.write(attachHumanMsg(entry), false);
  }

  const _apply = Game_Action.prototype.apply;
  Game_Action.prototype.apply = function (target) {
    const subject = this.subject ? this.subject() : null;
    const item = this.item ? this.item() : null;

    const subj = battlerInfo(subject);
    const targBefore = battlerInfo(target);
    const it = itemInfo(item);

    if (APPLY_LOG_MODE === "begin_end") {
      writeApplyLog("apply_begin", { subject: subj, item: it, target: targBefore });
    }
    pushRecent({ kind: "apply_begin", subject: subj, item: it, target: targBefore });

    try {
      _apply.call(this, target);

      const res = target && target.result ? target.result() : null;
      const targAfter = battlerInfo(target);

      let damageBreakdown = null;
      try {
        if (this._ktProbeDmgMap && typeof this._ktProbeDmgMap.get === "function") {
          damageBreakdown = this._ktProbeDmgMap.get(target) || null;
          this._ktProbeDmgMap.delete(target);
        }
      } catch (_) { damageBreakdown = null; }

      const endPayload = {
        subject: subj,
        item: it,
        targetBefore: targBefore,
        targetAfter: targAfter,
        result: res ? {
          used: !!res.used,
          missed: !!res.missed,
          evaded: !!res.evaded,
          critical: !!res.critical,
          hpDamage: (typeof res.hpDamage === "number") ? res.hpDamage : null,
          mpDamage: (typeof res.mpDamage === "number") ? res.mpDamage : null,
          tpDamage: (typeof res.tpDamage === "number") ? res.tpDamage : null,
          addedStates: res.addedStates ?? null,
          removedStates: res.removedStates ?? null,
        } : null,
        damageBreakdown,
      };

      if (APPLY_LOG_MODE === "end" || APPLY_LOG_MODE === "begin_end") {
        writeApplyLog("apply_end", endPayload);
      }

      pushRecent({
        kind: "apply_end",
        damage: res ? res.hpDamage : null,
        critical: res ? !!res.critical : null,
        addedStates: res ? res.addedStates : null,
        removedStates: res ? res.removedStates : null,
        afterHp: target ? target.hp : null,
      });

      if (
        res && res.used && it && it.isSkill &&
        it.damageType != null && Number(it.damageType) === 1 &&
        res.hpDamage === 0 && !res.missed && !res.evaded
      ) {
        dump("suspicious_zero_damage", endPayload);
      }
    } catch (e) {
      dump("apply_exception", { err: safeString(e) });
      throw e;
    }
  };

  // ----------------------------
  // Battle start hook
  // ----------------------------
  if (typeof BattleManager !== "undefined" && BattleManager && BattleManager.startBattle) {
    const _startBattle = BattleManager.startBattle;
    BattleManager.startBattle = function() {
      _startBattle.call(this);
      try { logBattleStartSnapshot(); } catch (_) {}
    };
  }

  // ----------------------------
  // plugin commands
  // ----------------------------
  PluginManager.registerCommand(PLUGIN_NAME, "ShowLogInfo", () => {
    console.log("[KT_AutoBugProbe] info:", Writer.info());
  });

  PluginManager.registerCommand(PLUGIN_NAME, "ClearLocalStorageLogs", () => {
    console.log("[KT_AutoBugProbe] ClearLocalStorageLogs:", Writer.clearLocalStorage());
  });

  PluginManager.registerCommand(PLUGIN_NAME, "ClearFileLogs", () => {
    console.log("[KT_AutoBugProbe] ClearFileLogs:", Writer.clearFile());
  });

})();