/*:
 * @target MZ
 * @plugindesc v1.0.0 KT_DamageDebugLogger: 戦闘中のダメージ0ケースをconsoleに記録（デバッグ用）
 * @author ChatGPT
 *
 * @param LogMode
 * @type select
 * @option FinalZeroOnly
 * @option Always
 * @default FinalZeroOnly
 *
 * @param LogMissEvade
 * @type boolean
 * @default true
 *
 * @help
 * ■使い方
 * - テストプレー中に F8 -> Console を開く
 * - ダメージが0になったケース（または常時）で内訳をconsoleに出力します
 *
 * ■見るべき項目
 * - baseValue（ダメージ式の結果）
 * - elementId / elementRate（属性IDと有効度）
 * - pdr/mdr/rec（物理/魔法/回復倍率）
 * - variance（分散）
 * - expectedNoVariance（分散なしでの丸め結果の目安）
 * - final（最終ダメージ：実際に適用された値）
 */

(() => {
  const PLUGIN_NAME = "KT_DamageDebugLogger";
  const params = PluginManager.parameters(PLUGIN_NAME);
  const logMode = String(params.LogMode || "FinalZeroOnly");
  const logMissEvade = String(params.LogMissEvade || "true") === "true";

  const _makeDamageValue = Game_Action.prototype.makeDamageValue;
  Game_Action.prototype.makeDamageValue = function(target, critical) {
    // まず“実際の最終値”（他プラグインの加工も含む）を取得
    const final = _makeDamageValue.call(this, target, critical);

    // 追加の乱数消費を避けるため、ここでは分散の再計算はしない（推定だけ出す）
    const item = this.item();
    const subject = this.subject();
    const result = target.result?.();

    let baseValue;
    try {
      baseValue = this.evalDamageFormula(target);
    } catch (e) {
      console.error("[DamageDebug] evalDamageFormula error", e, {
        item: item?.name, itemId: item?.id,
        subject: subject?.name?.(), target: target?.name?.()
      });
      return final;
    }

    const elementId = item?.damage?.elementId;
    const variance = item?.damage?.variance ?? 0;
    const dmgType = item?.damage?.type;
    const isPhysical = this.isPhysical();
    const isMagical = this.isMagical();

    const elementRate = this.calcElementRate(target);
    let value = baseValue * elementRate;

    if (isPhysical) value *= target.pdr;
    if (isMagical) value *= target.mdr;

    // 回復系はrecの影響を受ける（baseValueが負になるケースが多い）
    if (baseValue < 0) value *= target.rec;

    if (critical) value = this.applyCritical(value);

    // 分散なしの「目安」を出す（実際はここから分散→ガード→丸め）
    const afterGuard = this.applyGuard(value, target);
    const expectedNoVariance = Math.round(afterGuard);

    const missed = !!result?.missed;
    const evaded = !!result?.evaded;
    const hit = result?.isHit ? !!result.isHit() : undefined;

    const shouldLog =
      logMode === "Always" ||
      final === 0 ||
      (logMissEvade && (missed || evaded));

    if (shouldLog) {
      console.log("[DamageDebug]", {
        item: item ? `${item.name}(${item.id})` : "(unknown item)",
        kind: item ? (DataManager.isSkill(item) ? "skill" : "item") : "unknown",
        subject: subject?.name?.(),
        target: target?.name?.(),
        critical,
        missed, evaded, hit,
        dmgType,
        elementId,
        baseValue,
        elementRate,
        pdr: isPhysical ? target.pdr : "(n/a)",
        mdr: isMagical ? target.mdr : "(n/a)",
        rec: baseValue < 0 ? target.rec : "(n/a)",
        variance,
        expectedNoVariance,
        final
      });
      if (target?.states) {
        console.log("[DamageDebug] target states", target.states().map(s => `${s.name}(${s.id})`));
      }
      if (subject?.states) {
        console.log("[DamageDebug] subject states", subject.states().map(s => `${s.name}(${s.id})`));
      }
    }

    return final;
  };
})();
