/*:
 * @target MZ
 * @plugindesc v1.5.3 KT_EnemyTamaKyoudaReact: 敵がタマ強打ステートになったときトースト表示（同時表示なし／行動者の取りこぼし対策（保持強化））
 * @author ChatGPT
 *
 * @help
 * ■概要
 * - 戦闘中、敵（エネミー）が指定ステート（タマ強打／複数指定可）になったタイミングでトースト表示します。
 * - 表示順（キューで順番に表示・同時表示はしません）
 *   1) 敵リアクション（左寄り）
 *   2) 付与した味方（行動者）リアクション（右寄り／顔あり）
 *   3) 追加味方リアクション（右寄り／顔あり）
 *
 * ■置換
 * - {target}  = 対象（敵）の名前
 * - {speaker} = 発言者（アクター）の名前
 *
 * ■基本動作（このプラグインの主目的）
 * - 味方（アクター）の行動で、敵がタマ強打ステートになった（新規付与）とき表示
 *
 * ■追加動作（v1.3.0）
 * - 敵がすでにタマ強打状態のとき、「タマ強打付与」効果を持つ技が命中した場合にも表示します。
 * - （新規）タマ強打付与効果のある行動が命中し、そのまま戦闘不能になった場合、
 *   ステートが付かなかった（無効/耐性など）場合でも表示できます（パラメータでON/OFF）。
 *
 * ■追加点（v1.5.1）
 * - タマ強打として扱うステートごとに、"新規付与時は表示しない" を設定できます（ステート設定リスト）。
 *
 * ■戦闘不能同一行動
 * - タマ強打が付いたうえで同じ行動内で戦闘不能になっても表示されます。
 * - 「タマ強打付与効果はあるが、タマ強打にならないまま戦闘不能」ケースは表示しません。
 *
 * ■行動者取りこぼし対策（v1.1.2）
 * - スキル効果が「コモンイベント／スクリプト等」で遅れてステート付与される場合、
 *   apply() 内の subject を取れず「行動者が誰か分からない」ことがありました。
 * - v1.1.2 では BattleManager.startAction の行動者を「次の行動開始まで」保持し、
 *   その行動中〜直後に発生したステート付与でも行動者（味方）を拾いやすくしています。
 *
 * ■注意
 * - 敵は顔画像を持たないため、敵リアクションは顔無し表示です（テキストのみ）。
 * - 味方リアクションは顔指定（Override）またはDBの顔設定を使用します。
 *
 * @param TamaStateRules
 * @text タマ強打ステート設定（複数）
 * @type struct<TamaStateRule>[]
 * @desc 各ステートごとに『新規付与時は表示しない』等を設定できます。空の場合は下の StateIds/StateId を使用します。
 * @default []
 *
 * @param StateIds
 * @text タマ強打ステートID（複数）
 * @type state[]
 * @desc タマ強打として扱うステートIDを複数指定できます。空の場合は旧「タマ強打ステートID」を使用します。
 * @default ["33"]

 *
 * @param StateId
 * @text タマ強打ステートID（旧・単体）
 * @type state
 * @desc （互換用）StateIds が空のときに使用されます。
 * @default 33

 *
 * @param TamaOwnerStateIds
 * @text タマ持ち判定ステートID（敵）
 * @type number[]
 * @desc 対象（敵）がこのいずれかのステートを持つ場合のみ、タマ強打メッセージ判定を有効にします（例：31,32）。空配列 [] にすると無効化。
 * @default ["31","32"]

 *
 * @param DeathStateId
 * @text 戦闘不能ステートID
 * @type state
 * @default 1
 *
 * @param OnlyWhenSubjectIsActor
 * @text 味方が付与したときのみ
 * @type boolean
 * @desc trueなら「行動者がアクター」の場合だけ反応（推奨）
 * @default true
 *
 * @param EnableDeathSameFrame
 * @text 同フレーム戦闘不能でも表示
 * @type boolean
 * @default true
 *
 * @param ShowOnKillWithoutState
 * @text 付与前に倒しても表示
 * @type boolean
 * @desc タマ強打付与効果のある行動が命中し、相手が戦闘不能になった場合、ステートが付かなかった（無効/耐性など）場合でも表示します。
 * @default true
 *
 * @param DebugLog
 * @text デバッグログ
 * @type boolean
 * @default false
 *
 * @param EnemyDefaultMessages
 * @text 敵リアクション（共通）のメッセージ（複数）
 * @type string[]
 * @default ["うがあっ・・・！！"]
 *
 * @param EnemyMessages
 * @text 敵リアクション（敵データ別）
 * @type struct<EnemyMsg>[]
 * @default []
 *
 * @param ActorDefaultMessages
 * @text 付与した味方リアクション（共通）のメッセージ（複数）
 * @type string[]
 * @default ["うっわ～痛そう～！{target}、だいじょうぶ？"]
 *
 * @param ActorMessages
 * @text 付与した味方リアクション（アクター別）
 * @type struct<ActorMsg>[]
 * @default []
 *
 * @param ExtraReactions
 * @text 追加味方リアクション（複数）
 * @type struct<ExtraReact>[]
 * @desc 付与した味方の後に、別の味方リアクションを追加で出す設定
 * @default []
 *
 * @param ActorFaceOverrides
 * @text アクター顔指定（複数）
 * @type struct<ActorFaceEntry>[]
 * @desc アクターIDごとに顔（img/faces）を指定。未指定ならDB設定の顔を使用。
 * @default []
 *
 * @param MaxSimultaneousAllyToasts
 * @text 味方トースト同時表示数（未使用）
 * @type number
 * @min 1
 * @max 4
 * @default 2
 *
 * @param ToastFrames
 * @text 表示フレーム数
 * @type number
 * @min 1
 * @default 120
 *
 * @param ToastOpenCloseSpeed
 * @text 開閉速度（1-64）
 * @type number
 * @min 1
 * @max 64
 * @default 16
 *
 * @param AutoFitHeight
 * @text 全文が見えるように自動高さ
 * @type boolean
 * @default true
 *
 * @param AllyToastWidth
 * @text 味方トースト幅
 * @type number
 * @min 240
 * @default 560
 *
 * @param AllyToastMinHeight
 * @text 味方トースト最小高さ
 * @type number
 * @min 96
 * @default 160
 *
 * @param AllyToastMaxHeight
 * @text 味方トースト最大高さ
 * @type number
 * @min 96
 * @default 360
 *
 * @param AllyRightHalfDefault
 * @text 味方：右半分を基本にする
 * @type boolean
 * @desc trueなら、位置範囲未指定時に「画面右半分」内でランダム表示
 * @default true
 *
 * @param AllyXMin
 * @text 味方 X最小（範囲指定）
 * @type number
 * @min -99999
 * @max 99999
 * @default 0
 *
 * @param AllyXMax
 * @text 味方 X最大（範囲指定）
 * @type number
 * @min -99999
 * @max 99999
 * @default 0
 *
 * @param AllyYMin
 * @text 味方 Y最小（範囲指定）
 * @type number
 * @min -99999
 * @max 99999
 * @default 0
 *
 * @param AllyYMax
 * @text 味方 Y最大（範囲指定）
 * @type number
 * @min -99999
 * @max 99999
 * @default 0
 *
 * @param EnemyToastWidth
 * @text 敵トースト幅
 * @type number
 * @min 180
 * @default 420
 *
 * @param EnemyToastMinHeight
 * @text 敵トースト最小高さ
 * @type number
 * @min 72
 * @default 120
 *
 * @param EnemyToastMaxHeight
 * @text 敵トースト最大高さ
 * @type number
 * @min 72
 * @default 220
 *
 * @param EnemyAreaMode
 * @text 敵：自動配置モード
 * @type select
 * @option 中央より少し左（推奨）
 * @value centerLeft
 * @option 左半分
 * @value leftHalf
 * @option 全画面
 * @value full
 * @default centerLeft
 *
 * @param EnemyXMin
 * @text 敵 X最小（範囲指定）
 * @type number
 * @min -99999
 * @max 99999
 * @default 0
 *
 * @param EnemyXMax
 * @text 敵 X最大（範囲指定）
 * @type number
 * @min -99999
 * @max 99999
 * @default 0
 *
 * @param EnemyYMin
 * @text 敵 Y最小（範囲指定）
 * @type number
 * @min -99999
 * @max 99999
 * @default 0
 *
 * @param EnemyYMax
 * @text 敵 Y最大（範囲指定）
 * @type number
 * @min -99999
 * @max 99999
 * @default 0
 *
 * @param FlushToastOnEndBattle
 * @text 戦闘終了直前：未表示分をまとめてトーストに移す
 * @type boolean
 * @default true
 *
 * @param PreloadActorFaces
 * @text 戦闘開始時：顔を先読み
 * @type boolean
 * @default true
 */

/*~struct~TamaStateRule:
 * @param StateId
 * @text ステートID
 * @type state
 * @default 33
 *
 * @param SuppressOnNewApply
 * @text 新規付与時は表示しない
 * @type boolean
 * @desc trueなら、このステートが新規で付与された瞬間（新規付与）ではメッセージを表示しません。
 * @default false
 */

/*~struct~EnemyMsg:
 * @param EnemyId
 * @text 敵データID
 * @type enemy
 * @default 1
 *
 * @param Messages
 * @text この敵用メッセージ（複数）
 * @type string[]
 * @default ["うがあっ・・・！！"]
 */

/*~struct~ActorMsg:
 * @param ActorId
 * @text アクターID
 * @type actor
 * @default 1
 *
 * @param Messages
 * @text このアクター用メッセージ（複数）
 * @type string[]
 * @default ["うっわ～痛そう～！{target}、だいじょうぶ？"]
 */

/*~struct~ExtraReact:
 * @param SubjectActorId
 * @text 付与者アクターID（0=誰でも）
 * @type number
 * @min 0
 * @default 0
 *
 * @param TargetEnemyId
 * @text 対象敵データID（0=誰でも）
 * @type number
 * @min 0
 * @default 0
 *
 * @param SpeakerActorId
 * @text 発言者アクターID（0=自動で別メンバー）
 * @type number
 * @min 0
 * @default 0
 *
 * @param RequireSpeakerInParty
 * @text 発言者がパーティにいる時のみ
 * @type boolean
 * @default true
 *
 * @param RequireSpeakerAlive
 * @text 発言者が生存時のみ
 * @type boolean
 * @default true
 *
 * @param Messages
 * @text 発言メッセージ（複数）
 * @type string[]
 * @default ["そ、それはやりすぎです・・・。かわいそうですよ・・・。"]
 */

/*~struct~ActorFaceEntry:
 * @param ActorId
 * @text アクターID
 * @type actor
 * @default 1
 *
 * @param FaceFile
 * @text 顔ファイル（img/faces）
 * @type file
 * @dir img/faces
 * @default
 *
 * @param FaceIndex
 * @text 顔インデックス（0-7）
 * @type number
 * @min 0
 * @max 7
 * @default 0
 */

(() => {
  "use strict";

  function detectPluginName() {
    try {
      const cs = document.currentScript;
      if (cs && cs.src) {
        const m = cs.src.match(/([^/\\]+)\.js(?:\?.*)?$/);
        if (m) return m[1];
      }
    } catch {}
    const KEY = "KT_EnemyTamaKyoudaReact";
    try {
      if (Array.isArray(window.$plugins)) {
        const hit = window.$plugins.find(p => p && typeof p.description === "string" && p.description.includes(KEY));
        if (hit && hit.name) return String(hit.name);
      }
    } catch {}
    return "KT_EnemyTamaKyoudaReact";
  }

  const PLUGIN_NAME = detectPluginName();
  const P = PluginManager.parameters(PLUGIN_NAME) || {};

  const U = {
    bool(v, def) { if (v === undefined || v === null || v === "") return !!def; return String(v) === "true"; },
    num(v, def) { const n = Number(v); return Number.isFinite(n) ? n : def; },
    clamp(n, a, b) { n = Number(n); if (!Number.isFinite(n)) return a; return Math.max(a, Math.min(b, n)); },
    strArray(v) {
      if (!v) return [];
      try {
        const arr = JSON.parse(v);
        if (!Array.isArray(arr)) return [];
        return arr.map(s => String(s)).map(s => s.trim()).filter(s => s.length > 0);
      } catch { return []; }
    },
    numArray(v, defArr = []) {
      if (v === undefined || v === null || v === "") return Array.isArray(defArr) ? defArr.slice() : [];
      try {
        const arr = JSON.parse(v);
        if (!Array.isArray(arr)) return Array.isArray(defArr) ? defArr.slice() : [];
        return arr.map(n => Number(n)).filter(n => Number.isFinite(n));
      } catch {
        return Array.isArray(defArr) ? defArr.slice() : [];
      }
    },
    pickRandom(arr) {
      if (!arr || arr.length === 0) return "";
      const i = (Math.randomInt ? Math.randomInt(arr.length) : Math.floor(Math.random() * arr.length));
      return arr[i];
    },
    inBattle() { try { return $gameParty && $gameParty.inBattle(); } catch { return false; } },
    isActor(b) { try { return !!(b && b.isActor && b.isActor()); } catch { return false; } },
    isEnemy(b) { try { return !!(b && b.isEnemy && b.isEnemy()); } catch { return false; } },
    actorId(b) { try { return b && b.actorId ? b.actorId() : 0; } catch { return 0; } },
    enemyId(b) { try { return b && b.enemyId ? b.enemyId() : 0; } catch { return 0; } },
    battlerName(b) { try { return b && b.name ? b.name() : ""; } catch { return ""; } },
    frameCount() {
      try { return (typeof Graphics !== "undefined" && Number.isFinite(Graphics.frameCount)) ? Graphics.frameCount : 0; }
      catch { return 0; }
    },
    randInt(min, max) {
      min = Math.floor(Number(min)); max = Math.floor(Number(max));
      if (!Number.isFinite(min) || !Number.isFinite(max)) return 0;
      if (min > max) [min, max] = [max, min];
      return min + Math.floor(Math.random() * (max - min + 1));
    },
  };
  // ------------------------------------------------------------
  // Tama state rules (multi, per-state settings)
  // ------------------------------------------------------------
  function parseStructListFast(rawJson) {
    const raw = rawJson || "[]";
    let list = [];
    try {
      list = JSON.parse(raw);
      if (!Array.isArray(list)) list = [];
    } catch {
      list = [];
    }
    return list.map(s => {
      try { return JSON.parse(s); } catch { return {}; }
    });
  }

  const TAMA_STATE_RULES = (() => {
    // Preferred: per-state rule list
    const rulesRaw = parseStructListFast(P.TamaStateRules);
    const map = new Map();
    for (const o of rulesRaw) {
      const id = U.num(o.StateId, 0);
      if (id <= 0) continue;
      const suppressOnNewApply = U.bool(o.SuppressOnNewApply, false);
      // last wins
      map.set(id, { stateId: id, suppressOnNewApply });
    }
    if (map.size > 0) return Array.from(map.values());

    // Fallback: legacy ID list
    const ids = U.numArray(P.StateIds, []);
    const legacy = U.num(P.StateId, 33);
    const base = (ids && ids.length) ? ids : [legacy];
    const uniq = Array.from(new Set(base
      .map(n => Math.floor(Number(n)))
      .filter(n => Number.isFinite(n) && n > 0)));
    const finalIds = uniq.length ? uniq : [33];
    return finalIds.map(id => ({ stateId: id, suppressOnNewApply: false }));
  })();

  const TAMA_STATE_IDS = TAMA_STATE_RULES.map(r => r.stateId);
  const TAMA_STATE_ID_SET = new Set(TAMA_STATE_IDS);
  const TAMA_STATE_RULE_MAP = (() => {
    const m = Object.create(null);
    for (const r of TAMA_STATE_RULES) if (r && r.stateId > 0) m[r.stateId] = r;
    return m;
  })();

  function tamaRuleById(stateId) {
    const id = Number(stateId);
    return TAMA_STATE_RULE_MAP[id] || null;
  }

  const CFG = {
    stateIds: TAMA_STATE_IDS,
    stateId: TAMA_STATE_IDS[0],
    stateRules: TAMA_STATE_RULES,
    stateRuleMap: TAMA_STATE_RULE_MAP,
    tamaOwnerStateIds: U.numArray(P.TamaOwnerStateIds, [31, 32]),
    deathStateId: U.num(P.DeathStateId, 1),
    onlyWhenSubjectIsActor: U.bool(P.OnlyWhenSubjectIsActor, true),
    enableDeathSameFrame: U.bool(P.EnableDeathSameFrame, true),
    showOnKillWithoutState: U.bool(P.ShowOnKillWithoutState, true),
    debug: U.bool(P.DebugLog, false),

    enemyDefaultMessages: U.strArray(P.EnemyDefaultMessages),
    enemyMessageMap: Object.create(null),

    actorDefaultMessages: U.strArray(P.ActorDefaultMessages),
    actorMessageMap: Object.create(null),

    extraReactions: [],
    actorFaceOverrideMap: Object.create(null),

    toastFrames: Math.max(1, U.num(P.ToastFrames, 120)),
    toastSpeed: U.clamp(U.num(P.ToastOpenCloseSpeed, 16), 1, 64),
    autoFitHeight: U.bool(P.AutoFitHeight, true),

    allyW: Math.max(240, U.num(P.AllyToastWidth, 560)),
    allyMinH: Math.max(96, U.num(P.AllyToastMinHeight, 160)),
    allyMaxH: Math.max(96, U.num(P.AllyToastMaxHeight, 360)),
    allyRightHalfDefault: U.bool(P.AllyRightHalfDefault, true),
    allyXMin: U.num(P.AllyXMin, 0),
    allyXMax: U.num(P.AllyXMax, 0),
    allyYMin: U.num(P.AllyYMin, 0),
    allyYMax: U.num(P.AllyYMax, 0),

    enemyW: Math.max(180, U.num(P.EnemyToastWidth, 420)),
    enemyMinH: Math.max(72, U.num(P.EnemyToastMinHeight, 120)),
    enemyMaxH: Math.max(72, U.num(P.EnemyToastMaxHeight, 220)),
    enemyAreaMode: String(P.EnemyAreaMode || "centerLeft"),
    enemyXMin: U.num(P.EnemyXMin, 0),
    enemyXMax: U.num(P.EnemyXMax, 0),
    enemyYMin: U.num(P.EnemyYMin, 0),
    enemyYMax: U.num(P.EnemyYMax, 0),

    flushToastOnEndBattle: U.bool(P.FlushToastOnEndBattle, true),
    preloadActorFaces: U.bool(P.PreloadActorFaces, true),
  };

  // Safety: if plugin parameters failed to load (PluginManager.parameters returned empty),
  // message lists can become empty and nothing will be shown. Provide hardcoded fallbacks.
  if (!CFG.enemyDefaultMessages || CFG.enemyDefaultMessages.length === 0) {
    CFG.enemyDefaultMessages = ["うがあっ・・・！！"];
  }
  if (!CFG.actorDefaultMessages || CFG.actorDefaultMessages.length === 0) {
    CFG.actorDefaultMessages = ["うっわ～痛そう～！{target}、だいじょうぶ？"];
  }


  function dlog(label, payload) {
    if (!CFG.debug) return;
    try { console.log("[" + PLUGIN_NAME + "] " + label, payload); } catch {}
  }

  function parseStructList(rawJson) {
    const raw = rawJson || "[]";
    let list = [];
    try { list = JSON.parse(raw); if (!Array.isArray(list)) list = []; } catch { list = []; }
    return list.map(s => { try { return JSON.parse(s); } catch { return {}; } });
  }

  function parseEnemyMsgList(rawJson, outMap) {
    const list = parseStructList(rawJson);
    for (const o of list) {
      const eid = U.num(o.EnemyId, 0);
      const msgs = U.strArray(o.Messages);
      if (eid > 0 && msgs.length > 0) outMap[eid] = msgs;
    }
  }

  function parseActorMsgList(rawJson, outMap) {
    const list = parseStructList(rawJson);
    for (const o of list) {
      const aid = U.num(o.ActorId, 0);
      const msgs = U.strArray(o.Messages);
      if (aid > 0 && msgs.length > 0) outMap[aid] = msgs;
    }
  }

  function parseActorFaceList(rawJson, outMap) {
    const list = parseStructList(rawJson);
    for (const o of list) {
      const aid = U.num(o.ActorId, 0);
      const faceName = String(o.FaceFile || "").trim();
      const faceIndex = U.clamp(U.num(o.FaceIndex, 0), 0, 7);
      if (aid > 0 && faceName) outMap[aid] = { faceName, faceIndex };
    }
  }

  function parseExtraReactions(rawJson, outList) {
    const list = parseStructList(rawJson);
    for (const o of list) {
      const subjectActorId = Math.max(0, U.num(o.SubjectActorId, 0));
      const targetEnemyId = Math.max(0, U.num(o.TargetEnemyId, 0));
      const speakerActorId = Math.max(0, U.num(o.SpeakerActorId, 0));
      const requireInParty = U.bool(o.RequireSpeakerInParty, true);
      const requireAlive = U.bool(o.RequireSpeakerAlive, true);
      const msgs = U.strArray(o.Messages);
      if (msgs.length <= 0) continue;
      outList.push({
        subjectActorId,
        targetEnemyId,
        speakerActorId,
        requireInParty,
        requireAlive,
        messages: msgs,
      });
    }
  }

  parseEnemyMsgList(P.EnemyMessages, CFG.enemyMessageMap);
  parseActorMsgList(P.ActorMessages, CFG.actorMessageMap);
  parseExtraReactions(P.ExtraReactions, CFG.extraReactions);
  parseActorFaceList(P.ActorFaceOverrides, CFG.actorFaceOverrideMap);

  const RT = {
    reset() {
      if (!$gameTemp) return;
      $gameTemp._kt_etkr_guard = Object.create(null);
      $gameTemp._kt_etkr_pendingToast = [];
      $gameTemp._kt_etkr_uidCounter = 1;

      $gameTemp._kt_etkr_actionSerial = 0;
      $gameTemp._kt_etkr_actionSubject = null;
      $gameTemp._kt_etkr_actionSubjectSerial = 0;

      $gameTemp._kt_etkr_tamaAction = Object.create(null);

      $gameTemp._kt_etkr_ctxStack = [];
    },

    ensure() {
      if (!$gameTemp) return;
      if (!$gameTemp._kt_etkr_guard) this.reset();
    },

    guardMap() {
      this.ensure();
      return $gameTemp._kt_etkr_guard;
    },

    // ---------- action serial / subject ----------
    actionSerial() {
      this.ensure();
      return Number($gameTemp._kt_etkr_actionSerial || 0);
    },
    bumpActionSerial() {
      this.ensure();
      $gameTemp._kt_etkr_actionSerial = Number($gameTemp._kt_etkr_actionSerial || 0) + 1;
      return $gameTemp._kt_etkr_actionSerial;
    },
    setActionSubject(subject) {
      this.ensure();
      $gameTemp._kt_etkr_actionSubject = subject || null;
      // Keep the subject until the next action starts (tracked by actionSerial).
      // This makes delayed state application (common events / script) still attributable.
      $gameTemp._kt_etkr_actionSubjectSerial = this.actionSerial();
    },
    actionSubject() {
      this.ensure();
      const cur = this.actionSerial();
      const ss = Number($gameTemp._kt_etkr_actionSubjectSerial || 0);
      // If we can track actionSerial, only use the subject for the current action.
      if (cur > 0 && ss > 0 && cur !== ss) return null;
      return $gameTemp._kt_etkr_actionSubject || null;
    },

    // ---------- context stack (nested apply safety) ----------
    ctxStack() {
      this.ensure();
      return $gameTemp._kt_etkr_ctxStack;
    },
    pushCtx(ctx) { this.ctxStack().push(ctx); },
    popCtx() { const st = this.ctxStack(); return st.length ? st.pop() : null; },
    peekCtx() { const st = this.ctxStack(); return st.length ? st[st.length - 1] : null; },

    // ---------- battler uid ----------
    battlerUid(b) {
      if (!b) return 0;
      if (b._kt_etkr_uid) return b._kt_etkr_uid;
      this.ensure();
      const n = Number($gameTemp._kt_etkr_uidCounter || 1);
      $gameTemp._kt_etkr_uidCounter = n + 1;
      b._kt_etkr_uid = n;
      return n;
    },
    targetKey(b) {
      const uid = this.battlerUid(b);
      return uid ? ("U" + uid) : "";
    },

    // ---------- guard (per action if possible) ----------
    passGuardEvent(target, eventKey) {
      const key = this.targetKey(target);
      if (!key) return true;

      const serial = this.actionSerial();
      const suffix = serial > 0 ? ("S" + serial) : ("F" + U.frameCount());
      const k = key + ":" + eventKey + ":" + suffix;

      const m = this.guardMap();
      if (m[k]) return false;
      m[k] = true;
      return true;
    },

    // ---------- pending toast ----------
    pendingToast() {
      this.ensure();
      if (!$gameTemp._kt_etkr_pendingToast) $gameTemp._kt_etkr_pendingToast = [];
      return $gameTemp._kt_etkr_pendingToast;
    },
    pushToast(entry) { if (entry && entry.text) this.pendingToast().push(entry); },
    drainToast() {
      const q = this.pendingToast();
      const out = q.slice(0);
      q.length = 0;
      return out;
    },

    // ---------- tama/death same action ----------
    tamaActionMap() {
      this.ensure();
      if (!$gameTemp._kt_etkr_tamaAction) $gameTemp._kt_etkr_tamaAction = Object.create(null);
      return $gameTemp._kt_etkr_tamaAction;
    },
    markTamaThisAction(target) {
      const k = this.targetKey(target);
      if (!k) return;
      const serial = this.actionSerial();
      this.tamaActionMap()[k] = serial > 0 ? serial : U.frameCount();
    },
    wasTamaThisAction(target) {
      const k = this.targetKey(target);
      if (!k) return false;
      const cur = this.actionSerial();
      const v = this.tamaActionMap()[k];
      return (cur > 0) ? (v === cur) : (v === U.frameCount());
    },
  };

  // ------------------------------------------------------------
  // Text helpers
  // ------------------------------------------------------------
  function actorNameById(actorId) {
    const aid = Number(actorId || 0);
    if (aid <= 0) return "";
    try {
      const a = $gameActors && $gameActors.actor(aid);
      return a ? a.name() : "";
    } catch {}
    return "";
  }

  function formatText(raw, targetEnemy, speakerActorId) {
    const tname = U.battlerName(targetEnemy);
    const sname = speakerActorId ? actorNameById(speakerActorId) : "";
    return String(raw || "")
      .replace(/{target}/g, tname)
      .replace(/{speaker}/g, sname);
  }

  function resolveActorFaceById(actorId) {
    const aid = Number(actorId || 0);
    if (aid <= 0) return { faceName: "", faceIndex: 0 };

    const ov = CFG.actorFaceOverrideMap[aid];
    if (ov && ov.faceName) return { faceName: ov.faceName, faceIndex: U.clamp(ov.faceIndex, 0, 7) };

    try {
      const a = $gameActors && $gameActors.actor(aid);
      if (a && a.faceName && a.faceName()) {
        return { faceName: a.faceName(), faceIndex: U.clamp(a.faceIndex(), 0, 7) };
      }
    } catch {}
    return { faceName: "", faceIndex: 0 };
  }

  function currentSubjectFallback() {
    const ctx = RT.peekCtx();
    if (ctx && ctx.subject) return ctx.subject;

    // Additional fallback: BattleManager current subject (some setups keep this valid)
    try {
      if (typeof BattleManager !== "undefined" && BattleManager._subject && U.isActor(BattleManager._subject)) {
        return BattleManager._subject;
      }
    } catch {}

    return RT.actionSubject();
  }

  // ------------------------------------------------------------
  // Unified Toast Window (no simultaneous)
  // ------------------------------------------------------------
  class Window_KT_EnemyTamaToastUnified extends Window_Base {
    initialize() {
      // start with ally size, will resize per entry
      const rect = new Rectangle(0, 0, CFG.allyW, CFG.allyMinH);
      super.initialize(rect);
      this.openness = 0;

      this._speed = CFG.toastSpeed;
      this._remain = 0;
      this._entry = null;
      this._needClearOnClosed = false;
      this._queue = [];
      this._activeCfg = null;
    }

    setSpeed(n) { this._speed = U.clamp(n, 1, 64); }

    updateOpen() {
      if (this._opening) {
        this.openness = Math.min(255, this.openness + this._speed);
        if (this.openness >= 255) this._opening = false;
      }
    }
    updateClose() {
      if (this._closing) {
        this.openness = Math.max(0, this.openness - this._speed);
        if (this.openness <= 0) this._closing = false;
      }
    }

    isBusy() {
      return !!(this._entry || this._remain > 0 || this.openness > 0 || this._opening || this._closing);
    }

    enqueue(entry) {
      if (!entry || !entry.text) return;
      this._queue.push(entry);
    }
    enqueueMany(list) {
      if (!Array.isArray(list) || list.length === 0) return;
      for (const e of list) if (e && e.text) this._queue.push(e);
    }
    _dequeue() { return this._queue.length > 0 ? this._queue.shift() : null; }

    open() { this._opening = true; this._closing = false; }
    close() { this._closing = true; this._opening = false; this._needClearOnClosed = true; }

    update() {
      super.update();

      if (this._needClearOnClosed && this.openness === 0 && !this._closing && !this._opening) {
        this._needClearOnClosed = false;
        this._entry = null;
        this._remain = 0;
        try { this.contents.clear(); } catch {}
      }

      if (this._entry && !this._closing) {
        this._remain--;
        if (this._remain <= 0) this.close();
      }

      if (!this._entry && this.openness === 0 && !this._opening && !this._closing) {
        let next = this._dequeue();
        if (!next) next = RT.pendingToast().shift();
        if (next) this._startEntry(next);
      }
    }

    showEntry(entry) {
      if (!entry || !entry.text) return;
      if (this.isBusy()) this.enqueue(entry);
      else this._startEntry(entry);
    }

    _cfgForKind(kind) {
      if (kind === "enemy") {
        return {
          kind: "enemy",
          width: CFG.enemyW, minH: CFG.enemyMinH, maxH: CFG.enemyMaxH,
          frames: CFG.toastFrames,
          autoFitHeight: CFG.autoFitHeight,
          areaMode: (CFG.enemyXMin === 0 && CFG.enemyXMax === 0) ? CFG.enemyAreaMode : "full",
          xMin: CFG.enemyXMin, xMax: CFG.enemyXMax, yMin: CFG.enemyYMin, yMax: CFG.enemyYMax,
        };
      } else {
        return {
          kind: "ally",
          width: CFG.allyW, minH: CFG.allyMinH, maxH: CFG.allyMaxH,
          frames: CFG.toastFrames,
          autoFitHeight: CFG.autoFitHeight,
          areaMode: (CFG.allyXMin === 0 && CFG.allyXMax === 0) ? (CFG.allyRightHalfDefault ? "rightHalf" : "full") : "full",
          xMin: CFG.allyXMin, xMax: CFG.allyXMax, yMin: CFG.allyYMin, yMax: CFG.allyYMax,
        };
      }
    }

    _startEntry(entry) {
      this._entry = entry;
      this._activeCfg = this._cfgForKind(entry.kind);

      this._remain = Math.max(1, entry.frames || this._activeCfg.frames);

      this._resizeForText(entry, this._activeCfg);
      this._placeRandom(entry, this._activeCfg);

      this.refresh();
      this.open();
    }

    _resolveRange(cfg) {
      const bw = Graphics.boxWidth;
      const bh = Graphics.boxHeight;

      const userXSpecified = !(cfg.xMin === 0 && cfg.xMax === 0);
      const userYSpecified = !(cfg.yMin === 0 && cfg.yMax === 0);

      let xMin, xMax, yMin, yMax;

      if (userXSpecified) {
        xMin = cfg.xMin;
        xMax = cfg.xMax;
      } else {
        if (cfg.areaMode === "rightHalf") {
          xMin = Math.floor(bw / 2);
          xMax = Math.max(xMin, bw - this.width);
        } else if (cfg.areaMode === "leftHalf") {
          xMin = 0;
          xMax = Math.max(0, Math.floor(bw / 2) - this.width);
        } else if (cfg.areaMode === "centerLeft") {
          const margin = 24;
          xMax = Math.max(0, Math.floor(bw / 2) - margin);
          xMin = Math.max(0, xMax - this.width);
          xMax = Math.max(xMin, xMax);
        } else {
          xMin = 0;
          xMax = Math.max(0, bw - this.width);
        }
      }

      if (userYSpecified) {
        yMin = cfg.yMin;
        yMax = cfg.yMax;
      } else {
        yMin = 0;
        yMax = Math.max(0, bh - this.height);
      }

      return { xMin, xMax, yMin, yMax, bw, bh };
    }

    _placeRandom(entry, cfg) {
      const r = this._resolveRange(cfg);
      let x = U.randInt(r.xMin, r.xMax);
      let y = U.randInt(r.yMin, r.yMax);

      x = U.clamp(x, 0, Math.max(0, r.bw - this.width));
      y = U.clamp(y, 0, Math.max(0, r.bh - this.height));

      this.move(x, y, this.width, this.height);
    }

    _resizeForText(entry, cfg) {
      if (!cfg.autoFitHeight) {
        if (this.width !== cfg.width || this.height !== cfg.minH) {
          this.move(this.x, this.y, cfg.width, cfg.minH);
          this.createContents();
        }
        return;
      }

      const faceName = entry.faceName || "";
      const text = this.convertEscapeCharacters(String(entry.text));

      const newW = cfg.width;
      const pad = this.padding;
      const innerW = newW - pad * 2;

      const hasFace = !!faceName && cfg.kind === "ally";
      const faceW = (hasFace ? ImageManager.faceWidth : 0);
      const textW = Math.max(48, innerW - (hasFace ? (faceW + this.itemPadding()) : 0));

      const lines = this._wrapLinesForMeasure(text, textW);
      const lineH = this.lineHeight();
      const textH = Math.max(1, lines.length) * lineH;

      const faceH = (hasFace ? ImageManager.faceHeight : 0);
      const innerHNeed = Math.max(faceH, textH);

      let newH = innerHNeed + pad * 2;
      newH = U.clamp(newH, cfg.minH, cfg.maxH);

      if (this.width !== newW || this.height !== newH) {
        this.move(this.x, this.y, newW, newH);
        this.createContents();
      }
    }

    refresh() {
      this.contents.clear();
      if (!this._entry) return;

      const cfg = this._activeCfg || this._cfgForKind(this._entry.kind);
      const text = this.convertEscapeCharacters(String(this._entry.text));
      const faceName = this._entry.faceName || "";
      const faceIndex = U.clamp(U.num(this._entry.faceIndex, 0), 0, 7);

      const pad = this.itemPadding();
      const innerW = this.innerWidth;
      const innerH = this.innerHeight;

      let textX = 0;
      const hasFace = (cfg.kind === "ally") && !!faceName;
      if (hasFace) {
        try { ImageManager.loadFace(faceName); } catch {}
        this.drawFace(faceName, faceIndex, 0, 0, ImageManager.faceWidth, ImageManager.faceHeight);
        textX = ImageManager.faceWidth + pad;
      }

      const textW = innerW - textX;
      this._drawWrappedText(text, textX, 0, textW, innerH);
    }

    _drawWrappedText(text, x, y, w, h) {
      const lineH = this.lineHeight();
      const maxLines = Math.max(1, Math.floor(h / lineH));
      const lines = this._wrapLinesForDraw(text, w, maxLines);
      for (let i = 0; i < lines.length; i++) this.drawTextEx(lines[i], x, y + i * lineH, w);
    }

    _wrapLinesForMeasure(text, w) {
      const tokens = this._tokenizeForWrap(String(text));
      const lines = [];
      let cur = "";
      for (const token of tokens) {
        const next = cur ? (cur + token) : token;
        if (this._measureWidth(next) <= w) cur = next;
        else { if (cur) lines.push(cur); cur = token; }
      }
      if (cur) lines.push(cur);
      return lines.length > 0 ? lines : [String(text)];
    }

    _wrapLinesForDraw(text, w, maxLines) {
      const tokens = this._tokenizeForWrap(String(text));
      const lines = [];
      let cur = "";
      for (const t of tokens) {
        const next = cur ? (cur + t) : t;
        if (this._measureWidth(next) <= w) cur = next;
        else {
          if (cur) lines.push(cur);
          cur = t;
          if (lines.length >= maxLines) break;
        }
      }
      if (lines.length < maxLines && cur) lines.push(cur);
      return lines;
    }

    _measureWidth(s) {
      const str = String(s);
      let w = 0;
      const icons = str.match(/\x1bI\[\d+\]/gi);
      if (icons && icons.length) w += icons.length * (ImageManager.iconWidth || 32);
      w += this.textWidth(this._stripEscape(str));
      return w;
    }

    _stripEscape(s) {
      return String(s)
        .replace(/\x1bC\[\d+\]/gi, "")
        .replace(/\x1bI\[\d+\]/gi, "")
        .replace(/\x1bV\[\d+\]/gi, "")
        .replace(/\x1bN\[\d+\]/gi, "")
        .replace(/\x1bG/gi, "")
        .replace(/\x1b\{/g, "")
        .replace(/\x1b\}/g, "")
        .replace(/\x1b/g, "");
    }

    _tokenizeForWrap(text) {
      const s = String(text);
      const re = /\x1b[A-Z](?:\[\d+\])?|\x1bG|\x1b\{|\x1b\}|\s+|./g;
      return s.match(re) || [];
    }
  }

  function entryOf(kind, text, faceName, faceIndex) {
    return {
      kind: kind || "ally",
      text: String(text),
      faceName: faceName ? String(faceName) : "",
      faceIndex: Number.isFinite(faceIndex) ? faceIndex : 0,
      frames: CFG.toastFrames,
    };
  }

  function enqueueToast(entry) {
    if (!entry || !entry.text) return;
    try {
      const sc = SceneManager._scene;

      // If another plugin overwrote Scene_Battle.createAllWindows without calling original,
      // our toast window may not be created. Create it lazily on first use.
      try {
        const isBattleScene = sc && ((typeof Scene_Battle !== "undefined" && sc instanceof Scene_Battle) || (sc.constructor && sc.constructor.name === "Scene_Battle"));
        if (isBattleScene && !sc._ktEtkrToastWindow && typeof Window_KT_EnemyTamaToastUnified !== "undefined") {
          sc._ktEtkrToastWindow = new Window_KT_EnemyTamaToastUnified();
          sc._ktEtkrToastWindow.setSpeed(CFG.toastSpeed);
          if (typeof sc.addWindow === "function") sc.addWindow(sc._ktEtkrToastWindow);
          const pend = RT.drainToast();
          if (pend.length) sc._ktEtkrToastWindow.enqueueMany(pend);
        }
      } catch {}

      if (sc && sc._ktEtkrToastWindow) sc._ktEtkrToastWindow.showEntry(entry);
      else RT.pushToast(entry);
    } catch { RT.pushToast(entry); }
  }

  // ------------------------------------------------------------
  // Message selection
  // ------------------------------------------------------------
  function pickEnemyMessage(enemyTarget) {
    const eid = U.enemyId(enemyTarget);
    const list = CFG.enemyMessageMap[eid] || CFG.enemyDefaultMessages;
    if (!list || list.length === 0) return "";
    return formatText(U.pickRandom(list), enemyTarget, 0);
  }

  function pickActorMessage(subjectActor, enemyTarget) {
    const aid = U.actorId(subjectActor);
    const list = CFG.actorMessageMap[aid] || CFG.actorDefaultMessages;
    if (!list || list.length === 0) return "";
    return formatText(U.pickRandom(list), enemyTarget, aid);
  }

  function actorInParty(actorId) {
    try {
      const ms = $gameParty && $gameParty.members ? $gameParty.members() : [];
      return ms.some(a => a && a.actorId && a.actorId() === actorId);
    } catch {}
    return false;
  }

  function actorIsAlive(actorId) {
    try {
      const a = $gameActors && $gameActors.actor(actorId);
      return a ? a.isAlive() : false;
    } catch {}
    return false;
  }

  function autoPickSpeakerId(excludeActorId, requireAlive) {
    try {
      const ms = $gameParty && $gameParty.members ? $gameParty.members() : [];
      for (const a of ms) {
        if (!a || !a.actorId) continue;
        const aid = a.actorId();
        if (aid === excludeActorId) continue;
        if (requireAlive && !a.isAlive()) continue;
        return aid;
      }
    } catch {}
    return 0;
  }

  // ------------------------------------------------------------
  // Show sequence
  // ------------------------------------------------------------
  function showEnemyTamaSequence(enemyTarget, subject, reason) {
    if (!U.inBattle()) return;
    if (!enemyTarget || !U.isEnemy(enemyTarget)) return;

    // subject requirement (with fallback)
    let subj = subject;
    if (CFG.onlyWhenSubjectIsActor && (!subj || !U.isActor(subj))) {
      subj = currentSubjectFallback();
    }
    if (CFG.onlyWhenSubjectIsActor && (!subj || !U.isActor(subj))) return;

    RT.ensure();

    // guard per enemy per action (or per frame fallback)
    if (!RT.passGuardEvent(enemyTarget, "ENEMY_TAMA_TOAST")) {
      dlog("Guarded", { reason, enemy: U.battlerName(enemyTarget) });
      return;
    }

    // (1) enemy reaction (left-ish)
    const eMsg = pickEnemyMessage(enemyTarget);
    if (eMsg) enqueueToast(entryOf("enemy", eMsg, "", 0));

    // (2) subject actor reaction (right)
    if (subj && U.isActor(subj)) {
      const aMsg = pickActorMessage(subj, enemyTarget);
      const face = resolveActorFaceById(U.actorId(subj));
      if (aMsg) enqueueToast(entryOf("ally", aMsg, face.faceName, face.faceIndex));
    }

    // (3) extra ally reactions
    const subjectActorId = (subj && U.isActor(subj)) ? U.actorId(subj) : 0;
    const targetEnemyId = U.enemyId(enemyTarget);

    if (Array.isArray(CFG.extraReactions) && CFG.extraReactions.length) {
      for (const r of CFG.extraReactions) {
        if (!r) continue;
        if (r.subjectActorId > 0 && r.subjectActorId !== subjectActorId) continue;
        if (r.targetEnemyId > 0 && r.targetEnemyId !== targetEnemyId) continue;

        let speakerId = Number(r.speakerActorId || 0);
        if (speakerId <= 0) speakerId = autoPickSpeakerId(subjectActorId, !!r.requireAlive);
        if (speakerId <= 0) continue;

        if (r.requireInParty && !actorInParty(speakerId)) continue;
        if (r.requireAlive && !actorIsAlive(speakerId)) continue;

        const line = U.pickRandom(r.messages);
        const text = formatText(line, enemyTarget, speakerId);
        const face = resolveActorFaceById(speakerId);
        if (text) enqueueToast(entryOf("ally", text, face.faceName, face.faceIndex));
      }
    }

    dlog("Shown", {
      reason,
      enemy: U.battlerName(enemyTarget),
      subject: subject ? U.battlerName(subject) : "",
      subjectActorId,
      targetEnemyId,
    });
  }

  // ------------------------------------------------------------
  // Battle hooks (start/end)
  // ------------------------------------------------------------
  const _BattleManager_startBattle = BattleManager.startBattle;
  BattleManager.startBattle = function() {
    RT.reset();
    _BattleManager_startBattle.apply(this, arguments);

    dlog("CFG", { plugin: PLUGIN_NAME, stateIds: CFG.stateIds, ownerIds: CFG.tamaOwnerStateIds, enemyMsgN: (CFG.enemyDefaultMessages||[]).length, actorMsgN: (CFG.actorDefaultMessages||[]).length });

    if (CFG.preloadActorFaces) {
      try {
        const ms = $gameParty && $gameParty.members ? $gameParty.members() : [];
        for (const a of ms) if (a && a.faceName && a.faceName()) ImageManager.loadFace(a.faceName());
      } catch {}
      try {
        for (const k of Object.keys(CFG.actorFaceOverrideMap)) {
          const ov = CFG.actorFaceOverrideMap[k];
          if (ov && ov.faceName) ImageManager.loadFace(ov.faceName);
        }
      } catch {}
    }
  };

  if (typeof BattleManager.startAction === "function") {
    const _BattleManager_startAction = BattleManager.startAction;
    BattleManager.startAction = function() {
      RT.ensure();
      RT.bumpActionSerial();

      // Capture subject AFTER the original startAction (some setups set BattleManager._subject inside it).
      const ret = _BattleManager_startAction.apply(this, arguments);
      try { RT.setActionSubject(this._subject); } catch {}
      return ret;
    };
  }

  const _BattleManager_endBattle = BattleManager.endBattle;
  BattleManager.endBattle = function(result) {
    if (CFG.flushToastOnEndBattle) {
      try {
        const sc = SceneManager._scene;
        const toastW = sc && sc._ktEtkrToastWindow;
        if (toastW) {
          const all = RT.drainToast();
          if (all.length) toastW.enqueueMany(all);
        }
      } catch (e) {
        dlog("endBattle flush error", e);
      }
    }
    return _BattleManager_endBattle.apply(this, arguments);
  };

  if (typeof Scene_Battle !== "undefined") {
    const _Scene_Battle_createAllWindows = Scene_Battle.prototype.createAllWindows;
    Scene_Battle.prototype.createAllWindows = function() {
      _Scene_Battle_createAllWindows.apply(this, arguments);

      this._ktEtkrToastWindow = new Window_KT_EnemyTamaToastUnified();
      this._ktEtkrToastWindow.setSpeed(CFG.toastSpeed);
      this.addWindow(this._ktEtkrToastWindow);

      try {
        const pend = RT.drainToast();
        if (pend.length) this._ktEtkrToastWindow.enqueueMany(pend);
      } catch {}

      const first = this._ktEtkrToastWindow._dequeue && this._ktEtkrToastWindow._dequeue();
      if (first) this._ktEtkrToastWindow.showEntry(first);
    };

    const _Scene_Battle_terminate = Scene_Battle.prototype.terminate;
    Scene_Battle.prototype.terminate = function() {
      _Scene_Battle_terminate.apply(this, arguments);
      try { RT.reset(); } catch {}
    };
  }


  // ------------------------------------------------------------
  // Tama owner gate
  // ------------------------------------------------------------
  function hasTamaOwner(battler) {
    try {
      const ids = CFG.tamaOwnerStateIds || [];
      if (ids.length === 0) return true; // disabled
      return ids.some(id => battler && battler.isStateAffected && battler.isStateAffected(id));
    } catch { return true; }
  }


  // ------------------------------------------------------------
  // Tama state helpers (multi)
  // ------------------------------------------------------------
  function isTamaStateId(stateId) {
    return TAMA_STATE_ID_SET.has(Number(stateId));
  }

  function hasAnyTamaState(battler) {
    try {
      return TAMA_STATE_IDS.some(id => battler && battler.isStateAffected && battler.isStateAffected(id));
    } catch {
      return false;
    }
  }

  function actionMayAddAnyTamaState(action) {
    try {
      return TAMA_STATE_IDS.some(id => actionMayAddState(action, id));
    } catch {
      return false;
    }
  }

  function isAnyTamaStateAddedByResult(result) {
    try {
      return TAMA_STATE_IDS.some(id => result && typeof result.isStateAdded === 'function' && result.isStateAdded(id));
    } catch {
      return false;
    }
  }
  // ------------------------------------------------------------
  // Hooks (state)
  // ------------------------------------------------------------
  const _Game_BattlerBase_addNewState = Game_BattlerBase.prototype.addNewState;
  Game_BattlerBase.prototype.addNewState = function(stateId) {
    _Game_BattlerBase_addNewState.apply(this, arguments);

    if (!U.inBattle()) return;
    if (!U.isEnemy(this)) return;

    // new state event (best effort)
    if (isTamaStateId(stateId)) {
      const rule = tamaRuleById(stateId);
      if (rule && rule.suppressOnNewApply) return;
      if (!hasTamaOwner(this)) return;
      if (CFG.enableDeathSameFrame) RT.markTamaThisAction(this);
      const subject = currentSubjectFallback();
      showEnemyTamaSequence(this, subject, "tamaApplied(addNewState)");
    }

    if (CFG.enableDeathSameFrame && stateId === CFG.deathStateId) {
      if (RT.wasTamaThisAction(this) && hasTamaOwner(this)) {
        const subject = currentSubjectFallback();
        showEnemyTamaSequence(this, subject, "deathSameAction(addNewState)");
      }
    }
  };

  const _Game_Battler_addState = Game_Battler.prototype.addState;
  Game_Battler.prototype.addState = function(stateId) {
    const had = this.isStateAffected(stateId);
    _Game_Battler_addState.apply(this, arguments);

    if (!U.inBattle()) return;
    if (!U.isEnemy(this)) return;

    // new only
    if (had) return;
    if (!this.isStateAffected(stateId)) return;

    if (isTamaStateId(stateId)) {
      const rule = tamaRuleById(stateId);
      if (rule && rule.suppressOnNewApply) return;
      if (!hasTamaOwner(this)) return;
      if (CFG.enableDeathSameFrame) RT.markTamaThisAction(this);
      const subject = currentSubjectFallback();
      showEnemyTamaSequence(this, subject, "tamaApplied(addState)");
    }

    if (CFG.enableDeathSameFrame && stateId === CFG.deathStateId) {
      if (RT.wasTamaThisAction(this) && hasTamaOwner(this)) {
        const subject = currentSubjectFallback();
        showEnemyTamaSequence(this, subject, "deathSameAction(addState)");
      }
    }
  };

  function actionHasAddStateEffect(action, stateId) {
    try {
      const item = action && action.item ? action.item() : null;
      const effects = item && item.effects ? item.effects : null;
      if (!effects) return false;
      for (let i = 0; i < effects.length; i++) {
        const ef = effects[i];
        if (ef && ef.code === Game_Action.EFFECT_ADD_STATE && ef.dataId === stateId) return true;
      }
    } catch {}
    return false;
  }

  // action has explicit add-state effect only (do NOT treat normal attackStates as tama-add)
  function actionMayAddState(action, stateId) {
    return actionHasAddStateEffect(action, stateId);
  }

  // ------------------------------------------------------------
  // Hooks (apply/result) - reliable "subject" path
  // ------------------------------------------------------------
  const _Game_Action_apply = Game_Action.prototype.apply;
  Game_Action.prototype.apply = function(target) {
    const inBattle = U.inBattle();
    const isEnemyTarget = inBattle && target && U.isEnemy(target);

    let hadAnyTamaBefore = false;
    let hadTamaOwnerBefore = false;
    let mayAddTama = false;
    let wasAliveBefore = false;
    let hadTamaById = null;
    let mayAddTamaIds = null;

    if (isEnemyTarget) {
      try {
        hadTamaOwnerBefore = hasTamaOwner(target);

        hadTamaById = Object.create(null);
        hadAnyTamaBefore = false;
        for (const id of TAMA_STATE_IDS) {
          const v = !!(target && target.isStateAffected && target.isStateAffected(id));
          hadTamaById[id] = v;
          if (v) hadAnyTamaBefore = true;
        }

        mayAddTamaIds = [];
        for (const id of TAMA_STATE_IDS) {
          if (actionMayAddState(this, id)) mayAddTamaIds.push(id);
        }
        mayAddTama = mayAddTamaIds.length > 0;

        wasAliveBefore = target.isAlive ? target.isAlive() : (target.isDead ? !target.isDead() : true);
      } catch {}
    }

    RT.ensure();
    const subject = (() => { try { return this.subject ? this.subject() : null; } catch { return null; } })();
    RT.pushCtx({ subject, mayAddTama });

    _Game_Action_apply.apply(this, arguments);

    RT.popCtx();

    if (!isEnemyTarget) return;

    // hit?
    let hit = false;
    try {
      const r = target.result && target.result();
      if (r) {
        if (typeof r.isHit === "function") hit = r.isHit();
        else hit = !r.missed && !r.evaded;
      }
    } catch {}


    // state added? (per-state)
    let addedNewIds = [];
    let addedAllowedNewIds = [];
    try {
      const r = target.result && target.result();
      if (r) {
        const isAdded = (id) => {
          try {
            if (typeof r.isStateAdded === "function") return !!r.isStateAdded(id);
            if (Array.isArray(r.addedStates)) return r.addedStates.includes(id);
          } catch {}
          return false;
        };

        for (const id of TAMA_STATE_IDS) {
          const had = hadTamaById ? !!hadTamaById[id] : false;
          if (!had && isAdded(id)) {
            addedNewIds.push(id);
            const rule = tamaRuleById(id);
            const suppress = rule ? !!rule.suppressOnNewApply : false;
            if (!suppress) addedAllowedNewIds.push(id);
          }
        }
      }
    } catch {}

    // became dead after this action? (for kill-without-state case)
    let becameDeadAfter = false;
    if (CFG.showOnKillWithoutState && isEnemyTarget && wasAliveBefore) {
      try { becameDeadAfter = target.isDead ? target.isDead() : false; } catch {}
    }

    // trigger:
    // A) 設定されたステートのうち「新規付与」されたものがあり、かつそのステートが抑制設定でない
    // B) すでにそのステート中に、同じステート付与効果を持つ技が命中した
    if (hit && (hadTamaOwnerBefore || hadAnyTamaBefore)) {
      if (addedAllowedNewIds.length > 0) {
        if (CFG.enableDeathSameFrame) RT.markTamaThisAction(target);
        showEnemyTamaSequence(target, subject, "tamaAddedByResult(apply):" + addedAllowedNewIds.join(","));
      } else {
        // B) repeat hit while already in the same configured state
        let repeatHit = false;
        try {
          // If the target is already in ANY configured tama state, a tama-add skill counts as a repeat hit
          // ONLY when the skill attempts to add at least one non-suppressed tama state.
          // This prevents states like 37 (SuppressOnNewApply:true) from triggering messages by themselves.
          if (mayAddTama && hadAnyTamaBefore) {
            let mayAddAllowed = false;
            if (Array.isArray(mayAddTamaIds) && mayAddTamaIds.length) {
              for (const id of mayAddTamaIds) {
                const rule = tamaRuleById(id);
                const suppress = rule ? !!rule.suppressOnNewApply : false;
                if (!suppress) { mayAddAllowed = true; break; }
              }
            }
            if (mayAddAllowed) repeatHit = true;
          }
        } catch {}

        if (repeatHit) {
          showEnemyTamaSequence(target, subject, "tamaHitWhileAlready(apply)");
        } else if (CFG.showOnKillWithoutState && mayAddTama && !hadAnyTamaBefore && becameDeadAfter) {
          // C) tama skill hit and killed the target, but the tama state did not get applied (immune/resist etc.)
          //    respect per-state suppression (new-apply suppression)
          let attemptedAllowedNew = false;
          try {
            if (Array.isArray(mayAddTamaIds) && mayAddTamaIds.length && hadTamaById) {
              for (const id of mayAddTamaIds) {
                if (hadTamaById[id]) continue; // not a new apply attempt
                const rule = tamaRuleById(id);
                const suppress = rule ? !!rule.suppressOnNewApply : false;
                if (!suppress) { attemptedAllowedNew = true; break; }
              }
            }
          } catch {}

          if (attemptedAllowedNew && addedNewIds.length === 0) {
            showEnemyTamaSequence(target, subject, "tamaKillWithoutState(apply)");
          }
        }
      }
    }
  };

  RT.reset();
})();
