/*:
 * @target MZ
 * @plugindesc v1.0.1 KT_FemaleKintekiNoTamaReact: <Sex: Female>の「タマ無し金的」リアクション専用トースト
 * @author ChatGPT
 *
 * @help
 * ■概要
 * - 対象アクターがメモ欄に <Sex: Female> を持つ（女性扱い）場合に、
 *   「金的扱い（デフォルト：属性ID10）」を受けたタイミングで専用リアクションをトースト表示します。
 * - ただし「タマ生えステート（デフォルト：ID31）」を受けている場合は、この専用リアクションは出しません。
 *
 * ■使い方
 * 1) このプラグインをON
 * 2) 女性扱いにしたいアクターのメモ欄に <Sex: Female> を記述
 * 3) 金的扱いの判定を「属性ID」または「スキルID」で調整（プラグインパラメータ）
 *
 * ■補足
 * - これは「女性（タマ無し）リアクション」専用のため、既存のタマ強打系プラグインと併用できます。
 * - もし同一攻撃で別プラグイン側のメッセージも出て二重になる場合は、
 *   (A) 女性にはタマ強打ステートを付与しない運用にする
 *   (B) 既存プラグイン側に「女性時は出さない」条件を追加する
 *   のどちらかが確実です。
 *
 * @param kintekiElementIds
 * @text 金的属性ID一覧
 * @type number[]
 * @default [10]
 * @desc この属性IDが適用される攻撃を「金的扱い」とみなします（複数可）。
 *
 * @param triggerSkillIds
 * @text 発動スキルID限定（任意）
 * @type number[]
 * @default []
 * @desc 空なら全スキル対象。指定すると、このIDのスキル/アイテムのみで判定します。
 *
 * @param requireHit
 * @text 命中時のみ
 * @type boolean
 * @default true
 * @desc Miss/Evade時は出さない（true推奨）。
 *
 * @param triggerEvenIfDead
 * @text 戦闘不能でも表示
 * @type boolean
 * @default true
 * @desc ダメージで戦闘不能になっても（命中していれば）表示します。
 *
 * @param femaleMetaKey
 * @text 女性タグキー
 * @type string
 * @default Sex
 * @desc メモ欄タグのキー。例：<Sex: Female> の「Sex」
 *
 * @param femaleMetaValue
 * @text 女性タグ値
 * @type string
 * @default Female
 * @desc メモ欄タグの値。例：<Sex: Female> の「Female」（大文字小文字は無視）
 *
 * @param ignoreIfStateId
 * @text 無効化ステート（タマ生え）
 * @type state
 * @default 31
 * @desc このステートを受けている場合、女性（タマ無し）リアクションは出しません。
 *
 * @param messages
 * @text 女性（タマ無し）リアクション文
 * @type string[]
 * @default ["・・・？","あの、そこは男性ほどは痛くなくて・・・。ごめんなさい。"]
 * @desc ランダムに1つ表示。置換：{target} {subject}
 *
 * @param showFace
 * @text 顔グラ表示
 * @type boolean
 * @default false
 * @desc 対象アクターの顔グラを左に表示します。
 *
 * @param durationFrames
 * @text 表示時間（フレーム）
 * @type number
 * @min 1
 * @default 120
 * @desc 表示維持時間。60=約1秒。
 *
 * @param fadeOutFrames
 * @text フェードアウト（フレーム）
 * @type number
 * @min 0
 * @default 30
 * @desc 透明に消える時間。0で即消え。
 *
 * @param maxWidthRatio
 * @text 最大横幅（画面比）
 * @type number
 * @decimals 2
 * @min 0.2
 * @max 1.0
 * @default 0.50
 * @desc ウィンドウ最大幅（画面幅に対する比率）。
 *
 * @param xMin
 * @text X最小（px）
 * @type number
 * @min 0
 * @default 600
 * @desc 表示Xの最小値。ウィンドウがはみ出す場合は自動調整します。
 *
 * @param xMax
 * @text X最大（px）
 * @type number
 * @min 0
 * @default 1200
 * @desc 表示Xの最大値。
 *
 * @param yMin
 * @text Y最小（px）
 * @type number
 * @min 0
 * @default 40
 * @desc 表示Yの最小値。
 *
 * @param yMax
 * @text Y最大（px）
 * @type number
 * @min 0
 * @default 620
 * @desc 表示Yの最大値。
 *
 * @param blockAddedStateIds
 * @text （任意）付与ステートのブロック
 * @type number[]
 * @default []
 * @desc この専用リアクションが出た時に「付与された扱い」になったステートIDを打ち消します（空=何もしない）
 */

(() => {
  "use strict";

  const PLUGIN_NAME = "KT_FemaleKintekiNoTamaReact";

  // ------------------------
  // Param helpers
  // ------------------------
  const params = PluginManager.parameters(PLUGIN_NAME);

  const parseNumberArray = (s) => {
    try {
      const a = JSON.parse(s || "[]");
      return Array.isArray(a) ? a.map(n => Number(n)).filter(n => Number.isFinite(n)) : [];
    } catch (e) {
      return [];
    }
  };

  const kintekiElementIds = new Set(parseNumberArray(params.kintekiElementIds));
  const triggerSkillIds = new Set(parseNumberArray(params.triggerSkillIds));
  const requireHit = String(params.requireHit) === "true";
  const triggerEvenIfDead = String(params.triggerEvenIfDead) === "true";
  const femaleMetaKey = String(params.femaleMetaKey || "Sex");
  const femaleMetaValue = String(params.femaleMetaValue || "Female").trim().toLowerCase();
  const ignoreIfStateId = Number(params.ignoreIfStateId || 31);
  const messages = (() => {
    try {
      const a = JSON.parse(params.messages || "[]");
      return Array.isArray(a) && a.length ? a.map(x => String(x)) : ["・・・？"];
    } catch (e) {
      return ["・・・？"];
    }
  })();
  const showFace = String(params.showFace) === "true";
  const durationFrames = Math.max(1, Number(params.durationFrames || 120));
  const fadeOutFrames = Math.max(0, Number(params.fadeOutFrames || 30));
  const maxWidthRatio = Math.min(1.0, Math.max(0.2, Number(params.maxWidthRatio || 0.5)));
  const xMinParam = Math.max(0, Number(params.xMin || 0));
  const xMaxParam = Math.max(0, Number(params.xMax || 0));
  const yMinParam = Math.max(0, Number(params.yMin || 0));
  const yMaxParam = Math.max(0, Number(params.yMax || 0));
  const blockAddedStateIds = new Set(parseNumberArray(params.blockAddedStateIds));

  const randInt = (min, max) => {
    const a = Math.min(min, max);
    const b = Math.max(min, max);
    return a + Math.floor(Math.random() * (b - a + 1));
  };

  const choice = (arr) => arr[Math.floor(Math.random() * arr.length)];

  const isSceneBattle = () => SceneManager._scene && SceneManager._scene instanceof Scene_Battle;

  const metaValueOf = (battler, key) => {
    // For actors, read from actor().meta; for enemies, enemy().meta (not used here)
    if (battler && battler.isActor && battler.isActor()) {
      const a = battler.actor();
      return a && a.meta ? a.meta[key] : undefined;
    }
    return undefined;
  };

  const isFemaleTagged = (actorBattler) => {
    const v = metaValueOf(actorBattler, femaleMetaKey);
    if (v == null) return false;
    return String(v).trim().toLowerCase() === femaleMetaValue;
  };

  // ------------------------
  // Toast window
  // ------------------------
  class Window_KT_FemaleToast extends Window_Base {
    initialize(rect) {
      super.initialize(rect);
      this._text = "";
      this._actor = null;
      this._life = 0;
      this._fade = 0;
      this.openness = 255;
    }

    setToast(text, actor, lifeFrames, fadeFrames) {
      this._text = String(text || "");
      this._actor = actor || null;
      this._life = Math.max(1, lifeFrames | 0);
      this._fade = Math.max(0, fadeFrames | 0);
      this.opacity = 255;
      this.contentsOpacity = 255;
      this.refresh();
    }

    isFinished() {
      return this._life <= 0;
    }

    update() {
      super.update();
      if (this._life > 0) {
        this._life--;
        if (this._fade > 0 && this._life <= this._fade) {
          const t = this._life / this._fade; // 1 -> 0
          const op = Math.floor(255 * Math.max(0, Math.min(1, t)));
          this.opacity = op;
          this.contentsOpacity = op;
        }
      }
    }

    fittingHeight(numLines) {
      return this.itemHeight() * numLines + this.padding * 2;
    }

    refresh() {
      this.contents.clear();

      const text = this.convertEscapeCharacters(this._text);
      const faceW = showFace && this._actor ? ImageManager.faceWidth + 12 : 0;

      // Draw face
      let x = 0;
      if (showFace && this._actor) {
        this.drawFace(this._actor.faceName(), this._actor.faceIndex(), 0, 0, ImageManager.faceWidth, ImageManager.faceHeight);
        x = faceW;
      }

      // Draw text
      const lines = text.split("\n");
      let y = 0;
      for (const line of lines) {
        this.drawTextEx(line, x, y, this.contentsWidth() - x);
        y += this.lineHeight();
      }
    }
  }

  class KT_FemaleToastManager {
    constructor() {
      this._queue = [];
      this._window = null;
    }

    reset() {
      this._queue.length = 0;
      this._window = null;
    }

    _isWindowAlive() {
      const w = this._window;
      if (!w) return false;
      // PIXI sets destroyed/detached states during scene teardown.
      if (w.destroyed || w._destroyed) return false;
      if (!w.parent) return false;
      // Some PIXI versions null out transform-related fields after destroy.
      if (!w.transform) return false;
      return true;
    }

    request(text, actor) {
      if (!isSceneBattle()) return;
      this._queue.push({ text, actor });
      this._ensure();
    }

    _ensure() {
      if (!isSceneBattle()) return;

      // If the scene/window layer has been torn down, drop the stale reference.
      if (this._window && !this._isWindowAlive()) {
        this._window = null;
      }

      const scene = SceneManager._scene;
      if (!scene || !scene.addWindow) return;

      if (!this._window) {
        // Create a minimal rect first, will be resized per toast
        const rect = new Rectangle(0, 0, 240, 120);
        this._window = new Window_KT_FemaleToast(rect);
        scene.addWindow(this._window);
        this._window.hide();
      }
    }

    _calcWindowRect(text, actor) {
      const temp = this._window;
      const converted = temp.convertEscapeCharacters(String(text || ""));
      const lines = converted.split("\n");

      // Rough width estimation using textSizeEx (max per line)
      let maxW = 0;
      for (const line of lines) {
        const sz = temp.textSizeEx(line);
        maxW = Math.max(maxW, sz.width);
      }

      const faceW = (showFace && actor) ? (ImageManager.faceWidth + 12) : 0;
      const padding = temp.padding * 2;
      const maxWpx = Math.floor(Graphics.boxWidth * maxWidthRatio);

      const w = Math.min(maxWpx, Math.max(180, Math.ceil(maxW + faceW + padding + 24)));
      const h = Math.min(Graphics.boxHeight - 24, Math.max(temp.fittingHeight(lines.length), 96));

      // random position within bounds, auto clamp so it doesn't go out
      const xMin = xMinParam;
      const xMax = xMaxParam > 0 ? xMaxParam : (Graphics.boxWidth - 24);
      const yMin = yMinParam;
      const yMax = yMaxParam > 0 ? yMaxParam : (Graphics.boxHeight - 24);

      let x = randInt(xMin, xMax);
      let y = randInt(yMin, yMax);

      x = Math.max(0, Math.min(x, Graphics.boxWidth - w));
      y = Math.max(0, Math.min(y, Graphics.boxHeight - h));

      return new Rectangle(x, y, w, h);
    }

    update() {
      if (!isSceneBattle()) {
        this.reset();
        return;
      }

      this._ensure();
      if (!this._window) return;
      if (!this._isWindowAlive()) {
        this._window = null;
        return;
      }

      if (!this._window.visible) {
        if (this._queue.length > 0) {
          const req = this._queue.shift();
          const rect = this._calcWindowRect(req.text, req.actor);
          try {
            this._window.move(rect.x, rect.y, rect.width, rect.height);
            this._window.setToast(req.text, req.actor, durationFrames + fadeOutFrames, fadeOutFrames);
            this._window.show();
          } catch (e) {
            // If the window got disposed during teardown, avoid crashing the game.
            this.reset();
          }
        }
        return;
      }

      this._window.update();
      if (this._window.isFinished()) {
        this._window.hide();
      }
    }
  }

  const KT_TOAST = new KT_FemaleToastManager();

  // Hook Scene_Battle update to drive toast manager
  const _Scene_Battle_update = Scene_Battle.prototype.update;
  Scene_Battle.prototype.update = function() {
    _Scene_Battle_update.call(this);
    KT_TOAST.update();
  };

  // Clear stale references on scene teardown (prevents PIXI "position is null" type errors)
  const _Scene_Battle_terminate = Scene_Battle.prototype.terminate;
  Scene_Battle.prototype.terminate = function() {
    KT_TOAST.reset();
    if (_Scene_Battle_terminate) _Scene_Battle_terminate.call(this);
  };

  // ------------------------
  // Action detection
  // ------------------------
  const matchesSkillRestriction = (action) => {
    if (!action || !action.item) return false;
    if (triggerSkillIds.size === 0) return true;
    const item = action.item();
    return !!item && triggerSkillIds.has(Number(item.id));
  };

  const actionHasKintekiElement = (action) => {
    if (!action || !action.item) return false;
    const item = action.item();
    if (!item || !item.damage) return false;

    const eid = Number(item.damage.elementId);
    if (!Number.isFinite(eid)) return false;

    // elementId: 0 = Normal Attack (uses subject attack elements). Positive = fixed element.
    if (eid > 0) return kintekiElementIds.has(eid);

    if (eid === 0) {
      const subject = action.subject && action.subject();
      if (!subject || !subject.attackElements) return false;
      const elems = subject.attackElements();
      return Array.isArray(elems) && elems.some(e => kintekiElementIds.has(Number(e)));
    }

    return false;
  };

  const shouldTriggerForTarget = (action, target) => {
    if (!target || !target.isActor || !target.isActor()) return false;
    if (!isFemaleTagged(target)) return false;
    if (ignoreIfStateId > 0 && target.isStateAffected(ignoreIfStateId)) return false;
    if (!matchesSkillRestriction(action)) return false;
    if (!actionHasKintekiElement(action)) return false;
    if (!triggerEvenIfDead && !target.isAlive()) return false;
    return true;
  };

  const makeMessage = (text, subject, target) => {
    return String(text || "")
      .replace(/\{subject\}/gi, subject ? subject.name() : "")
      .replace(/\{target\}/gi, target ? target.name() : "");
  };

  // Apply hook
  const _Game_Action_apply = Game_Action.prototype.apply;
  Game_Action.prototype.apply = function(target) {
    _Game_Action_apply.call(this, target);

    try {
      if (!isSceneBattle()) return;

      if (!shouldTriggerForTarget(this, target)) return;

      const result = target.result && target.result();
      if (requireHit && result && !result.isHit()) return;

      // Optional: block added states if requested
      if (result && blockAddedStateIds.size > 0 && Array.isArray(result.addedStates)) {
        for (const sid of result.addedStates) {
          if (blockAddedStateIds.has(Number(sid))) {
            // remove state if it was added by this action
            target.removeState(Number(sid));
          }
        }
      }

      const subject = this.subject ? this.subject() : null;
      const raw = choice(messages);
      const msg = makeMessage(raw, subject, target);
      const actor = target.actor ? target.actor() : null;

      KT_TOAST.request(msg, actor);
    } catch (e) {
      // no-op (avoid hard crash)
      // console.warn(`[${PLUGIN_NAME}] error`, e);
    }
  };

})();
