/*:
 * @target MZ
 * @plugindesc v1.0.0 KT_SexFemaleElementGuard - <Sex: Female> の対象は指定属性ダメージを1/100（タマ生えステート中は無効）
 * @author ChatGPT
 *
 * @param elementId
 * @text 対象属性ID
 * @type number
 * @min 1
 * @default 10
 *
 * @param rateMultiplier
 * @text 倍率（1/100=0.01）
 * @type number
 * @decimals 4
 * @default 0.01
 *
 * @param bypassStateId
 * @text 無効化ステートID（タマ生え）
 * @type state
 * @default 31
 *
 * @param sexMetaKey
 * @text ノートタグキー
 * @type string
 * @default Sex
 *
 * @param femaleValue
 * @text Female判定値
 * @type string
 * @default Female
 *
 * @param applyToActors
 * @text アクターに適用
 * @type boolean
 * @default true
 *
 * @param applyToEnemies
 * @text 敵に適用
 * @type boolean
 * @default true
 *
 * @help
 * ■機能
 * - メモ欄に <Sex: Female> が付いている対象は、指定属性ID（既定:10）のダメージ倍率がさらに 0.01 倍されます（=最終ダメージが1/100）。
 * - ただし対象が「無効化ステートID（既定:31 / タマ生え）」を受けている間は、この 0.01 倍を無効化します。
 *
 * ■設定
 * - アクター or 敵キャラのメモ欄に <Sex: Female> を付けてください。
 *
 * ■補足
 * - 複数属性のスキル等で「実際に採用される（最大倍率になる）属性」が対象属性IDのときだけ 0.01 倍します。
 */

(() => {
  "use strict";

  const PLUGIN_NAME = (() => {
    const script = document.currentScript;
    if (script && script.src) {
      const name = script.src.split("/").pop().replace(/\.js$/i, "");
      return name || "KT_SexFemaleElementGuard";
    }
    return "KT_SexFemaleElementGuard";
  })();

  const p = PluginManager.parameters(PLUGIN_NAME);

  const ELEMENT_ID = Number(p.elementId || 10);
  const MULTIPLIER = Number(p.rateMultiplier || 0.01);
  const BYPASS_STATE_ID = Number(p.bypassStateId || 31);

  const SEX_META_KEY = String(p.sexMetaKey || "Sex");
  const FEMALE_VALUE = String(p.femaleValue || "Female").trim().toLowerCase();

  const APPLY_TO_ACTORS = String(p.applyToActors || "true") === "true";
  const APPLY_TO_ENEMIES = String(p.applyToEnemies || "true") === "true";

  function isFemaleByNoteTag(battler) {
    if (!battler) return false;

    if (battler.isActor && battler.isActor()) {
      if (!APPLY_TO_ACTORS) return false;
      const data = battler.actor && battler.actor();
      const v = data?.meta?.[SEX_META_KEY];
      return v != null && String(v).trim().toLowerCase() === FEMALE_VALUE;
    }

    if (battler.isEnemy && battler.isEnemy()) {
      if (!APPLY_TO_ENEMIES) return false;
      const data = battler.enemy && battler.enemy();
      const v = data?.meta?.[SEX_META_KEY];
      return v != null && String(v).trim().toLowerCase() === FEMALE_VALUE;
    }

    return false;
  }

  function actionElements(action) {
    if (action.elements) return action.elements();

    const item = action.item?.();
    const id = item?.damage?.elementId ?? 0;
    if (id < 0) {
      const subject = action.subject?.();
      return subject?.attackElements?.() ?? [];
    }
    return [id];
  }

  const _Game_Action_calcElementRate = Game_Action.prototype.calcElementRate;
  Game_Action.prototype.calcElementRate = function (target) {
    const baseRate = _Game_Action_calcElementRate.call(this, target);

    if (!target) return baseRate;
    if (!Number.isFinite(ELEMENT_ID) || ELEMENT_ID <= 0) return baseRate;
    if (!Number.isFinite(MULTIPLIER) || MULTIPLIER === 1) return baseRate;

    if (BYPASS_STATE_ID > 0 && target.isStateAffected?.(BYPASS_STATE_ID)) {
      return baseRate; // タマ生え中は無効
    }

    if (!isFemaleByNoteTag(target)) return baseRate;

    const elements = actionElements(this);
    if (!elements || elements.length === 0) return baseRate;
    if (!elements.includes(ELEMENT_ID)) return baseRate;

    // 複数属性の場合：実際に採用される（最大倍率になる）属性が対象属性IDのときだけ適用
    let maxR = -Infinity;
    let targetElemR = null;

    for (const e of elements) {
      const r = target.elementRate?.(e) ?? 1;
      if (e === ELEMENT_ID) targetElemR = r;
      if (r > maxR) maxR = r;
    }

    if (targetElemR == null) return baseRate;

    const EPS = 1e-10;
    if (targetElemR + EPS < maxR) return baseRate;

    return baseRate * MULTIPLIER;
  };
})();
