/*:
 * @target MZ
 * @plugindesc v1.2.6 KT_SkillImageFlash: 指定スキル使用時に画像を短時間表示（中央±ランダム/スキルDB選択/表示ディレイ）
 * @author ChatGPT
 *
 * @help
 * ■追加点（v1.2.6）
 * - 画像表示にディレイ（遅延）秒数を指定できます。
 *
 * ■概要
 * 戦闘中、指定したスキルが使用されたタイミングで、指定した画像（img/pictures）を一定時間表示します。
 * 表示位置は画面中央付近＋指定範囲でランダムにズラします。
 *
 * ■設定方法（推奨：設定リスト）
 * 1) 「設定リスト」に要素を追加
 * 2) 画像ファイル名を選択
 * 3) 対象スキル（複数）で、DBからスキルを追加して選択
 * 4) 表示ディレイ秒数（必要なら）を設定
 *
 * ■同時表示
 * - 1枚のみ。新規発火で上書き（ディレイ中も含めて上書き）します。
 *
 * ■もし「設定リスト」がテキスト1本のままで項目が出ない場合
 * - 「代替設定（テキスト）」mappingsText を使えます。
 *
 * ■代替設定（mappingsText）の書式（1行=1設定）
 * 【新形式（ディレイあり）】
 * picture|skillIds|delay|duration|randX|randY|offX|offY|scale|opacity
 * 例：
 * FlashA|5,6|0.20|0.30|24|16|0|0|100|255
 *
 * 【旧形式（ディレイなし）】※互換
 * picture|skillIds|duration|randX|randY|offX|offY|scale|opacity
 *
 * @param entries
 * @text 設定リスト
 * @type struct<KTSIFEntry>[]
 * @default []
 *
 * @param mappingsText
 * @text 代替設定（テキスト）
 * @type multiline_string
 * @default
 *
 * @param defaultDelaySec
 * @text デフォルト表示ディレイ秒数
 * @type number
 * @decimals 2
 * @min 0
 * @default 0
 *
 * @param defaultDurationSec
 * @text デフォルト表示秒数
 * @type number
 * @decimals 2
 * @min 0
 * @default 0.30
 *
 * @param baseOffsetX
 * @text ベースX（中央からの固定オフセット）
 * @type number
 * @min -99999
 * @default 0
 *
 * @param baseOffsetY
 * @text ベースY（中央からの固定オフセット）
 * @type number
 * @min -99999
 * @default 0
 *
 * @param randomRangeX
 * @text ランダム範囲X（±ピクセル）
 * @type number
 * @min 0
 * @default 24
 *
 * @param randomRangeY
 * @text ランダム範囲Y（±ピクセル）
 * @type number
 * @min 0
 * @default 16
 *
 * @param defaultScale
 * @text デフォルト拡大率(%)
 * @type number
 * @min 1
 * @default 100
 *
 * @param defaultOpacity
 * @text デフォルト不透明度
 * @type number
 * @min 0
 * @max 255
 * @default 255
 *
 * @param fadeOutSec
 * @text フェードアウト秒数
 * @type number
 * @decimals 2
 * @min 0
 * @default 0.10
 */
/*~struct~KTSIFEntry:
 * @param pictureName
 * @text 画像ファイル名
 * @type file
 * @dir img/pictures
 * @default
 *
 * @param skillIds
 * @text 対象スキル（複数）
 * @type skill[]
 * @default []
 *
 * @param delaySec
 * @text 表示ディレイ秒数（-1=全体デフォルト）
 * @type number
 * @decimals 2
 * @min -1
 * @default -1
 *
 * @param durationSec
 * @text 表示秒数（-1=全体デフォルト）
 * @type number
 * @decimals 2
 * @min -1
 * @default -1
 *
 * @param offsetX
 * @text 追加X
 * @type number
 * @min -99999
 * @default 0
 *
 * @param offsetY
 * @text 追加Y
 * @type number
 * @min -99999
 * @default 0
 *
 * @param randX
 * @text ランダム範囲X（-1=全体/0=なし/1+=±）
 * @type number
 * @min -1
 * @default -1
 *
 * @param randY
 * @text ランダム範囲Y（-1=全体/0=なし/1+=±）
 * @type number
 * @min -1
 * @default -1
 *
 * @param scale
 * @text 拡大率%（0=全体デフォルト）
 * @type number
 * @min 0
 * @default 0
 *
 * @param opacity
 * @text 不透明度（-1=全体デフォルト）
 * @type number
 * @min -1
 * @max 255
 * @default -1
 *
 * @param skillIdsCsv
 * @text （旧）対象スキルID（CSV）※互換用
 * @default
 */
(() => {
  "use strict";

  const pluginName = (() => {
    const cs = document.currentScript;
    if (cs?.src) return cs.src.split("/").pop().replace(/\.js$/i, "");
    return "KT_SkillImageFlash";
  })();

  const params = PluginManager.parameters(pluginName);

  const toNumber = (v, def) => {
    const n = Number(v);
    return Number.isFinite(n) ? n : def;
  };
  const clamp = (n, min, max) => Math.max(min, Math.min(max, n));

  const sanitizePictureName = (name) => {
    const s = String(name || "");
    const base = s.split("/").pop();
    return base.replace(/\.(png|jpg|jpeg|webp)$/i, "");
  };

  const parseStructArray = (v) => {
    if (!v) return [];
    let a;
    try { a = JSON.parse(v); } catch { return []; }
    if (!Array.isArray(a)) return [];
    return a.map(e => {
      if (typeof e === "string") { try { return JSON.parse(e); } catch { return null; } }
      return e;
    }).filter(Boolean);
  };

  const parseNumberArrayLoose = (v) => {
    if (Array.isArray(v)) return v.map(Number).filter(n => Number.isFinite(n) && n > 0);
    if (typeof v === "string") {
      const t = v.trim();
      if (!t) return [];
      try {
        const a = JSON.parse(t);
        if (Array.isArray(a)) return a.map(Number).filter(n => Number.isFinite(n) && n > 0);
      } catch {
        return t.split(/[,\s]+/).map(Number).filter(n => Number.isFinite(n) && n > 0);
      }
    }
    return [];
  };

  const parseSkillIdsCsv = (text) => {
    if (!text) return [];
    return String(text)
      .split(/[,\s]+/)
      .map(s => s.trim())
      .filter(Boolean)
      .map(Number)
      .filter(n => Number.isFinite(n) && n > 0);
  };

  const defaultDelaySec = Math.max(0, toNumber(params.defaultDelaySec, 0));
  const defaultDurationSec = Math.max(0, toNumber(params.defaultDurationSec, 0.30));
  const baseOffsetX = toNumber(params.baseOffsetX, 0);
  const baseOffsetY = toNumber(params.baseOffsetY, 0);
  const randomRangeX = Math.max(0, toNumber(params.randomRangeX, 24));
  const randomRangeY = Math.max(0, toNumber(params.randomRangeY, 16));
  const defaultScale = Math.max(1, toNumber(params.defaultScale, 100));
  const defaultOpacity = clamp(toNumber(params.defaultOpacity, 255), 0, 255);
  const fadeOutSec = Math.max(0, toNumber(params.fadeOutSec, 0.10));

  const buildSkillMap = () => {
    const map = new Map();

    // 1) entries（struct）から読む
    const entries = parseStructArray(params.entries);
    for (const e of entries) {
      const pictureName = sanitizePictureName(e.pictureName || "");
      if (!pictureName) continue;

      let ids = parseNumberArrayLoose(e.skillIds);
      if (!ids.length) ids = parseSkillIdsCsv(e.skillIdsCsv);
      if (!ids.length) continue;

      const delayRaw = toNumber(e.delaySec, -1);
      const delaySec = Math.max(0, delayRaw >= 0 ? delayRaw : defaultDelaySec);

      const durationRaw = toNumber(e.durationSec, -1);
      const durationSec = Math.max(0, durationRaw >= 0 ? durationRaw : defaultDurationSec);

      const offsetX = toNumber(e.offsetX, 0);
      const offsetY = toNumber(e.offsetY, 0);

      const randXRaw = toNumber(e.randX, -1);
      const randYRaw = toNumber(e.randY, -1);
      const randX = Math.max(0, randXRaw >= 0 ? randXRaw : randomRangeX);
      const randY = Math.max(0, randYRaw >= 0 ? randYRaw : randomRangeY);

      const scaleRaw = toNumber(e.scale, 0);
      const scale = clamp(scaleRaw > 0 ? scaleRaw : defaultScale, 1, 10000);

      const opacityRaw = toNumber(e.opacity, -1);
      const opacity = clamp(opacityRaw >= 0 ? opacityRaw : defaultOpacity, 0, 255);

      const cfg = { pictureName, delaySec, durationSec, offsetX, offsetY, randX, randY, scale, opacity };
      for (const id of ids) map.set(id, cfg);
    }

    // 2) entriesが効かない/空の時は mappingsText（代替）
    if (map.size === 0) {
      const text = String(params.mappingsText || "").replace(/\r\n/g, "\n");
      const lines = text.split("\n").map(l => l.trim()).filter(l => l && !l.startsWith("#"));

      for (const line of lines) {
        const cols = line.split("|").map(s => s.trim());

        // 新形式: 10列以上 picture|ids|delay|duration|randX|randY|offX|offY|scale|opacity
        // 旧形式:  9列     picture|ids|duration|randX|randY|offX|offY|scale|opacity
        const isNew = cols.length >= 10;

        const pictureName = sanitizePictureName(cols[0] || "");
        const ids = parseSkillIdsCsv(cols[1] || "");
        if (!pictureName || !ids.length) continue;

        const delaySec = Math.max(0, toNumber(isNew ? cols[2] : 0, 0));
        const durationSec = Math.max(0, toNumber(isNew ? cols[3] : cols[2], defaultDurationSec));

        const randX = Math.max(0, toNumber(isNew ? cols[4] : cols[3], randomRangeX));
        const randY = Math.max(0, toNumber(isNew ? cols[5] : cols[4], randomRangeY));

        const offsetX = toNumber(isNew ? cols[6] : cols[5], 0);
        const offsetY = toNumber(isNew ? cols[7] : cols[6], 0);

        const scale = clamp(toNumber(isNew ? cols[8] : cols[7], defaultScale), 1, 10000);
        const opacity = clamp(toNumber(isNew ? cols[9] : cols[8], defaultOpacity), 0, 255);

        const cfg = { pictureName, delaySec, durationSec, offsetX, offsetY, randX, randY, scale, opacity };
        for (const id of ids) map.set(id, cfg);
      }
    }

    return map;
  };

  const skillMap = buildSkillMap();

  const randIntBetween = (min, max) => {
    const span = max - min + 1;
    return min + Math.randomInt(Math.max(1, span));
  };

  class KTSIF_Controller {
    constructor(scene) {
      this._sprite = new Sprite();
      this._sprite.anchor.set(0.5, 0.5);
      this._sprite.visible = false;

      this._remainFrames = 0;
      this._fadeOutFrames = 0;
      this._baseOpacity = 255;

      // ディレイ用
      this._pendingCfg = null;
      this._delayFrames = 0;

      const wl = scene._windowLayer;
      if (wl) {
        const idx = scene.getChildIndex(wl);
        scene.addChildAt(this._sprite, idx);
      } else {
        scene.addChild(this._sprite);
      }
    }

    preloadAll() {
      const set = new Set();
      for (const [, cfg] of skillMap) set.add(cfg.pictureName);
      for (const name of set) ImageManager.loadPicture(name);
    }

    show(skillId) {
      const cfg = skillMap.get(skillId);
      if (!cfg) return;

      // 上書き：現在表示中/ディレイ待ちを停止して新しい要求に切り替え
      this._sprite.visible = false;
      this._sprite.bitmap = null;
      this._remainFrames = 0;
      this._fadeOutFrames = 0;

      this._pendingCfg = cfg;
      this._delayFrames = Math.max(0, Math.round(cfg.delaySec * 60));

      if (this._delayFrames === 0) {
        const c = this._pendingCfg;
        this._pendingCfg = null;
        this._startDisplay(c);
      }
    }

    _startDisplay(cfg) {
      this._sprite.bitmap = ImageManager.loadPicture(cfg.pictureName);

      const rx = cfg.randX > 0 ? randIntBetween(-cfg.randX, cfg.randX) : 0;
      const ry = cfg.randY > 0 ? randIntBetween(-cfg.randY, cfg.randY) : 0;

      this._sprite.x = Graphics.width / 2 + baseOffsetX + cfg.offsetX + rx;
      this._sprite.y = Graphics.height / 2 + baseOffsetY + cfg.offsetY + ry;

      const s = cfg.scale / 100;
      this._sprite.scale.set(s, s);

      this._baseOpacity = cfg.opacity;
      this._sprite.opacity = this._baseOpacity;
      this._sprite.visible = true;

      this._remainFrames = Math.max(0, Math.round(cfg.durationSec * 60));
      this._fadeOutFrames = Math.max(0, Math.round(fadeOutSec * 60));
      if (this._fadeOutFrames > this._remainFrames) this._fadeOutFrames = this._remainFrames;
    }

    update() {
      // ディレイ待ち
      if (this._pendingCfg) {
        if (this._delayFrames > 0) {
          this._delayFrames--;
        }
        if (this._delayFrames <= 0) {
          const cfg = this._pendingCfg;
          this._pendingCfg = null;
          this._startDisplay(cfg);
        }
        return;
      }

      // 表示中
      if (!this._sprite.visible) return;

      if (this._remainFrames > 0) {
        this._remainFrames--;
        if (this._fadeOutFrames > 0 && this._remainFrames < this._fadeOutFrames) {
          const t = this._remainFrames / this._fadeOutFrames;
          this._sprite.opacity = Math.round(this._baseOpacity * t);
        } else {
          this._sprite.opacity = this._baseOpacity;
        }
      } else {
        this._sprite.visible = false;
        this._sprite.bitmap = null;
      }
    }
  }

  const _Scene_Battle_create = Scene_Battle.prototype.create;
  Scene_Battle.prototype.create = function() {
    _Scene_Battle_create.call(this);
    this._ktsif = new KTSIF_Controller(this);
    this._ktsif.preloadAll();
  };

  const _Scene_Battle_update = Scene_Battle.prototype.update;
  Scene_Battle.prototype.update = function() {
    _Scene_Battle_update.call(this);
    this._ktsif?.update();
  };

  const _BattleManager_startAction = BattleManager.startAction;
  BattleManager.startAction = function() {
    const action = this._subject?.currentAction?.() || null;

    _BattleManager_startAction.call(this);

    if (!action?.isSkill?.()) return;
    const item = action.item();
    if (!item?.id) return;

    const scene = SceneManager._scene;
    if (scene instanceof Scene_Battle && scene._ktsif) {
      scene._ktsif.show(item.id);
    }
  };
})();
