/*:
 * @target MZ
 * @plugindesc v1.0 MV→MZ 互換コア：MV式pluginCommandのブリッジ、コマンド登録ヘルパ、メタ/注釈解析・画面サイズユーティリティ等
 * @author ChatGPT
 * @help MV2MZ_CompatCore.js
 *
 * このプラグインは “MV時代のプラグイン” を “MZで動かす” ための薄い互換レイヤーです。
 * 主な機能：
 *  - MV式 pluginCommand(command, args) を MZ で受けるブリッジ
 *  - MZ式 registerCommand と MV式 pluginCommand を同時登録するヘルパ
 *  - <RateMap: x> などメタタグの note/注釈(108/408)補完解析
 *  - 画面サイズユーティリティ（MV/MZ両対応）
 *  - マップ遷移での一時差し替え解除用の簡易フック
 *
 * ▼使い方（例：MapGenerator）
 * 1) 依存側プラグインで以下のように書くだけで、
 *    - エディタUIからは MZコマンドとして呼べる
 *    - MV式「MapGenerator RoomAndPass」でも呼べる
 *
 *   const P = "YourPluginName";
 *   MVCompat.addDualCommand(P, "RoomAndPass", () => $gameMap.generateMap?.("RoomAndPass"));
 *   MVCompat.addDualCommand(P, "FillRoom",    () => $gameMap.generateMap?.("FillRoom"));
 *   MVCompat.addMVAlias("MapGenerator", (args)=>{ // MV式: MapGenerator <subcmd>
 *     const sub = (args?.[0]||"").trim();
 *     if (sub==="RoomAndPass") MVCompat.run(P,"RoomAndPass");
 *     if (sub==="FillRoom")    MVCompat.run(P,"FillRoom");
 *   });
 *
 * 2) メタ解析：
 *   const meta = MVCompat.eventMeta(evData); // note/注釈から <RateMap:x> 等を補完
 *
 * 3) 画面サイズ：
 *   const w = MVCompat.screenW(), h = MVCompat.screenH();
 *
 * ■注意
 * - これは“コア”です。実処理は依存プラグイン側で書いてください。
 * - 競合が心配なら本コアを最上段に配置してください。
 */

(function(){
  "use strict";

  // 防重複
  if (window.MVCompat) return;

  const _registry = {
    // cmdMap: { pluginName: { commandName: handler } }
    cmdMap: Object.create(null),
    // mvAlias: { mvCommandName: handler(argsArray, interpreterThis) }
    mvAlias: Object.create(null),
  };

  //==================================================
  // 公開API
  //==================================================
  const MVCompat = {
    version: "1.0.0",

    /** MZ式コマンド登録（PluginManager.registerCommand） */
    addMZCommand(pluginName, commandName, handler){
      if (!pluginName || !commandName || typeof handler!=="function") return;
      (_registry.cmdMap[pluginName] ||= Object.create(null))[commandName] = handler;
      if (PluginManager && PluginManager.registerCommand){
        PluginManager.registerCommand(pluginName, commandName, args=>{
          try{ handler(args); }catch(e){ console.error(e); }
        });
      }
    },

    /** MV式 pluginCommand の別名（入口）を登録 */
    addMVAlias(mvCommandName, handler){
      if (!mvCommandName || typeof handler!=="function") return;
      _registry.mvAlias[mvCommandName] = handler;
    },

    /** 両方式を一度に：MZ名で登録し、run()可能にする */
    addDualCommand(pluginName, commandName, handler){
      this.addMZCommand(pluginName, commandName, handler);
    },

    /** 登録済みハンドラを実行（依存プラグインから呼べる） */
    run(pluginName, commandName, args){
      const h = _registry.cmdMap?.[pluginName]?.[commandName];
      if (typeof h === "function"){
        return h(args);
      }
      console.warn(`[MVCompat] handler not found: ${pluginName}.${commandName}`);
    },

    /** パラメータ取得（MV/MZ両対応・安全ラップ） */
    params(pluginName){
      try{
        const p = PluginManager.parameters(pluginName) || {};
        return new Proxy(p, {
          get(t, k){
            if (!(k in t)) return undefined;
            // 自動で型推定（数値/真偽/JSON）
            const v = t[k];
            if (v===undefined || v===null) return v;
            if (v==="true") return true;
            if (v==="false") return false;
            if (/^-?\d+(\.\d+)?$/.test(v)) return Number(v);
            if ((v.startsWith("{")&&v.endsWith("}")) || (v.startsWith("[")&&v.endsWith("]"))){
              try{ return JSON.parse(v); }catch{ /* noop */ }
            }
            return v;
          }
        });
      }catch(e){
        console.warn(e);
        return {};
      }
    },

    /** イベント/データのメタ解析（noteと注釈108/408から補完） */
    meta(obj){
      if (!obj) return {};
      const meta = Object.assign({}, obj.meta||{});
      const text = (obj.note||"") + this._extractNotesFromPages(obj.pages);
      // 一般形： <Key: value> を抽出
      const re = /<\s*([A-Za-z0-9_]+)\s*:\s*([^>]+)\s*>/g;
      let m;
      while ((m = re.exec(text)) !== null){
        const key = m[1];
        const val = m[2].trim();
        if (meta[key]==null) meta[key]=val;
      }
      return meta;
    },

    /** イベント専用ショートカット */
    eventMeta(evData){ return this.meta(evData); },

    /** 注釈抽出（コード108/408） */
    _extractNotesFromPages(pages){
      if (!pages) return "";
      let s = "";
      for (const p of pages){
        if (!p || !p.list) continue;
        for (const cmd of p.list){
          if (!cmd) continue;
          if (cmd.code===108 || cmd.code===408){
            s += "\n" + (cmd.parameters?.[0]||"");
          }
        }
      }
      return s;
    },

    /** 画面サイズ（MV/MZ両対応） */
    screenW(){ return Graphics?.width || Graphics?.boxWidth || 816; },
    screenH(){ return Graphics?.height|| Graphics?.boxHeight|| 624; },

    /** 一時差し替え解除用：マップ遷移で呼ばれる Game_Map.setup をラップ */
    onMapReset(clearFn){
      // clearFn: () => void  生成結果の解除など
      if (typeof clearFn!=="function") return;
      _mapResetHooks.add(clearFn);
    },
  };

  window.MVCompat = MVCompat;

  //==================================================
  // MV式 pluginCommand ブリッジ
  //==================================================
  // MZでも MVの入口を生かす
  const _GE_pc = Game_Interpreter.prototype.pluginCommand;
  Game_Interpreter.prototype.pluginCommand = function(command, args){
    if (_GE_pc) _GE_pc.call(this, command, args);
    try{
      const h = _registry.mvAlias[command];
      if (typeof h === "function"){
        h.call(this, args);
      }
    }catch(e){
      console.error(e);
    }
  };

  //==================================================
  // マップ遷移時の一括解除フック
  //==================================================
  const _mapResetHooks = new Set();

  const _GM_setup = Game_Map.prototype.setup;
  Game_Map.prototype.setup = function(mapId){
    _GM_setup.call(this, mapId);
    // 登録された解除処理を呼ぶ（生成データや一時差し替えをリセットしたい場合に）
    try{
      for (const fn of _mapResetHooks) fn();
    }catch(e){ console.error(e); }
  };

})();
