/*:
 * @target MZ
 * @plugindesc v1.14 《オスのちから》と《一時的オスのちから》を同時に保持可能にする拡張版。4つ状態（悲惨な重複）を再現します。（一時オス解除履歴のバグ修正版）
 * @author ChatGPT
 * @help
 * 【概要】
 * - 《オスのちから》（永続）と《一時的オスのちから》を同時に発生させることができます。
 * - 両方が存在すると「4つある状態（悲惨なシチュ）」を表現可能です。
 * - 《タマの痛み》はどちらかが残っていれば維持され、両方なくなった時のみ解除されます。
 *
 * 【基本仕様】
 * - HasBallsタグがあるキャラには《オスのちから》が常時保証されます。
 * - ProtectMalePower=true の場合、《オスのちから》は外せません。
 * - 一時的オスのちからが切れても、永続はそのまま維持されます。
 *
 * 【仕様変更点】
 * - 従来の相互排他ロジックを撤廃。
 * - 永続・一時が共存可能（ダブル付与OK）。
 * - 「タマの痛み」は両方のステートが失われたときにのみ解除。
 *
 * 【v1.14 での修正】
 * - 一時的オスのちから解除時に記録している
 *   $gameTemp._recentlyRemovedTempMale が、ゲーム開始後に
 *   Game_Temp に差し替えられることで undefined になり、
 *   「Cannot set property 'E9' of undefined」が発生していた問題を修正。
 * - 以後は、使用するたびに _recentlyRemovedTempMale を安全に初期化します。
 *
 * @param MalePowerStateId
 * @text 《オスのちから》ステートID（永続）
 * @type state
 * @default 0
 *
 * @param TempMalePowerStateId
 * @text 《一時的オスのちから》ステートID（一時）
 * @type state
 * @default 0
 *
 * @param BallPainStateId
 * @text 《タマの痛み》ステートID
 * @type state
 * @default 0
 *
 * @param AutoApplyOnBattleStart
 * @text 戦闘開始時に《オスのちから》を保証
 * @type boolean
 * @on する
 * @off しない
 * @default true
 *
 * @param AutoApplyOnMapStart
 * @text マップ開始時に《オスのちから》を保証
 * @type boolean
 * @on する
 * @off しない
 * @default true
 *
 * @param ProtectMalePower
 * @text 《オスのちから》を保護（解除不可）
 * @type boolean
 * @on 保護する
 * @off 保護しない
 * @default true
 */

(() => {
  //============================================================================
  // パラメータ取得
  //============================================================================
  const PLUGIN = (function () {
    try {
      const src = document.currentScript.src;
      return src.split("/").pop().replace(/\.js$/, "");
    } catch (e) {
      return "MalePowerAndBallPain";
    }
  })();

  const P = PluginManager.parameters(PLUGIN);
  const MALE_POWER = Number(P.MalePowerStateId || 0);
  const TEMP_MALE  = Number(P.TempMalePowerStateId || 0);
  const BALL_PAIN  = Number(P.BallPainStateId || 0);
  const AUTO_BTL   = P.AutoApplyOnBattleStart === "true";
  const AUTO_MAP   = P.AutoApplyOnMapStart === "true";
  const PROTECT_MP = P.ProtectMalePower === "true";

  //============================================================================
  // 汎用ヘルパ
  //============================================================================
  const hasMeta = (obj, key) => obj && obj.meta && (obj.meta[key] !== undefined);

  function poolNotesOf(battler) {
    const pool = [];
    if (battler.isActor && battler.isActor()) {
      const a = battler.actor(); if (a) pool.push(a);
      const c = battler.currentClass(); if (c) pool.push(c);
      battler.equips().forEach(e => { if (e) pool.push(e); });
    } else {
      const e = battler.enemy(); if (e) pool.push(e);
    }
    battler.states().forEach(s => { if (s) pool.push(s); });
    return pool;
  }

  function hasHasBallsSource(battler) {
    return poolNotesOf(battler).some(d => hasMeta(d, "HasBalls"));
  }

  function battlerHasMalePower(battler) {
    if (hasHasBallsSource(battler)) return true;
    if (MALE_POWER > 0 && battler.isStateAffected(MALE_POWER)) return true;
    if (TEMP_MALE  > 0 && battler.isStateAffected(TEMP_MALE )) return true;
    return false;
  }

function battlerUniqueKey(battler) {
  if (!battler) return null;
  try {
    if (battler.isActor && battler.isActor()) {
      return "A" + battler.actorId();
    } else {
      return "E" + battler.enemyId();
    }
  } catch (e) {
    return null;
  }
}

  // 一時的オス解除履歴用マップを安全に取得
  function tempMaleHistoryMap() {
    // ゲーム開始前などで $gameTemp が未生成の場合は null を返す
    if (typeof $gameTemp === "undefined" || !$gameTemp) return null;
    if (!$gameTemp._recentlyRemovedTempMale) {
      $gameTemp._recentlyRemovedTempMale = {};
    }
    return $gameTemp._recentlyRemovedTempMale;
  }

  //============================================================================
  // addState：相互排他を撤廃
  //============================================================================
  const _addState = Game_Battler.prototype.addState;
  Game_Battler.prototype.addState = function (stateId) {
    const sid = Number(stateId || 0);

    // 「タマの痛み」はタマ持ち以外には付与しない
    if (BALL_PAIN > 0 && sid === BALL_PAIN) {
      const hasBalls =
        hasHasBallsSource(this) ||
        (MALE_POWER > 0 && this.isStateAffected(MALE_POWER)) ||
        (TEMP_MALE  > 0 && this.isStateAffected(TEMP_MALE ));
      if (!hasBalls) return;
    }

    return _addState.apply(this, arguments);
  };

  //============================================================================
  // removeState：痛み解除条件を「両方なくなったときのみ」に変更
  //============================================================================
  const _removeState = Game_Battler.prototype.removeState;
  Game_Battler.prototype.removeState = function (stateId) {
    const sid = Number(stateId || 0);

    // 保護ロジック（永続）
    if (PROTECT_MP && sid === MALE_POWER) {
      if ($gameTemp && $gameTemp._forceRemoveMalePower) {
        $gameTemp._forceRemoveMalePower = false;
      } else {
        return;
      }
    }

    const res = _removeState.apply(this, arguments);

    // 一時的オスのちからが外れた時刻を記録
    if (sid === TEMP_MALE && TEMP_MALE > 0) {
      const key = battlerUniqueKey(this);
      const map = tempMaleHistoryMap();
      if (key && map) {
        map[key] = Date.now();
      }
    }

    // 両方なくなったら《タマの痛み》も解除
    if (BALL_PAIN > 0) {
      const hasMale = this.isStateAffected(MALE_POWER);
      const hasTemp = this.isStateAffected(TEMP_MALE );
      if (!hasMale && !hasTemp && this.isStateAffected(BALL_PAIN)) {
        _removeState.call(this, BALL_PAIN);
      }
    }

    return res;
  };

  //============================================================================
  // ensureMalePower：HasBalls持ちには永続を保証
  //============================================================================
  function ensureMalePower(battler) {
    if (MALE_POWER <= 0) return;
    if (hasHasBallsSource(battler) && !battler.isStateAffected(MALE_POWER)) {
      battler.addState(MALE_POWER);
    }
  }

  //============================================================================
  // 自動保証（マップ開始・戦闘開始・リフレッシュ）
  //============================================================================
  if (AUTO_MAP) {
    const _Scene_Map_start = Scene_Map.prototype.start;
    Scene_Map.prototype.start = function () {
      _Scene_Map_start.apply(this, arguments);
      $gameParty.members().forEach(a => ensureMalePower(a));
    };
  }

  if (AUTO_BTL) {
    const _BattleManager_startBattle = BattleManager.startBattle;
    BattleManager.startBattle = function () {
      _BattleManager_startBattle.apply(this, arguments);
      const all = $gameParty.members().concat($gameTroop.members());
      all.forEach(b => ensureMalePower(b));
    };
  }

  const _Game_Actor_refresh = Game_Actor.prototype.refresh;
  Game_Actor.prototype.refresh = function () {
    _Game_Actor_refresh.apply(this, arguments);
    ensureMalePower(this);
  };

})();
