//=============================================================================
// Mano_EnemyHelpWindow.js (RPGツクールMZ用改造版)
// ----------------------------------------------------------------------------
// 元: Manosasayaki_EnemyInformationWindow.js (by しぐれん / 魔のささやき)
// 改造: MZ用ウィンドウ仕様に対応
// MIT License
//=============================================================================

/*:
 * @target MZ
 * @plugindesc 攻撃対象選択中に、敵の弱点・耐性を表示するウィンドウを追加します（MZ用移植版）
 * @author しぐれん（魔のささやき）＋改造：あなたの名前
 * 
 * @param WindowRect
 * @text ウィンドウ位置・サイズ
 * 
 * @param WindowWidth
 * @text ウィンドウ幅
 * @type number
 * @default 300
 * @parent WindowRect
 * 
 * @param WindowHeight
 * @text ウィンドウ高さ
 * @type number
 * @default 150
 * @parent WindowRect
 * 
 * @param WindowX
 * @text X座標（空欄で自動）
 * @type number
 * @default
 * @parent WindowRect
 * 
 * @param WindowY
 * @text Y座標（空欄で自動）
 * @type number
 * @default
 * @parent WindowRect
 * 
 * @param WeakLine
 * @text 弱点判定ライン（％）
 * @type number
 * @desc 属性有効度がこの数値を上回ると弱点として表示します。
 * @default 100
 * 
 * @param ResistanceLine
 * @text 耐性判定ライン（％）
 * @type number
 * @desc 属性有効度がこの数値を下回ると耐性として表示します。
 * @default 100
 * 
 * @param WeakName
 * @text 弱点ラベル
 * @desc 弱点行の先頭に表示するテキスト
 * @default 弱点：
 * 
 * @param ResistanceName
 * @text 耐性ラベル
 * @desc 耐性行の先頭に表示するテキスト
 * @default 耐性：
 * 
 * @param displayMode
 * @text 表示モード
 * @type select
 * @option text
 * @option icon
 * @desc 属性を文字で表示するか、アイコンで表示するか
 * @default text
 * 
 * @param elementIconList
 * @text 属性アイコン用スキルリスト
 * @desc 属性アイコン用に参照するスキルIDの配列（属性が重複している場合は未定義）
 * @type skill[]
 * @default []
 *  
 * @param hiddenState
 * @text 非表示ステート
 * @desc このステートが付与されている敵は弱点・耐性を隠します。
 * @type state
 * @default 0
 * 
 * @param hiddenIcon
 * @text 非表示時アイコン
 * @desc 弱点が隠されている場合に代わりに表示するアイコンID（0で何も表示しない）
 * @type number
 * @default 0
 * 
 * @help
 * ■概要
 * バトル中、敵を選択しているときに、
 * その敵の「属性弱点」と「属性耐性」を表示するウィンドウを追加します。
 * 
 * 弱点・耐性は「属性有効度」を元に判定します。
 *   ・ 属性有効度 > WeakLine％ → 弱点
 *   ・ 属性有効度 < ResistanceLine％ → 耐性
 * 
 * 表示方法は「文字（属性名）」または「アイコン」を選べます。
 * 
 * ■メモ欄タグ
 * エネミー（またはアクター）のメモ欄に以下を書くと、
 * そのバトラーについては弱点・耐性を表示しません。
 *   <EnemyHelpHide>
 * 
 * ■注意
 * ・本プラグインは、元々MV用のものをMZ向けに書き換えたものです。
 * ・他のバトル系プラグインと競合する可能性があります。
 * 
 * ■更新履歴
 * v1.0(MZ)  MZ用にウィンドウ初期化・位置計算等を修正
 */

(() => {
    'use strict';

    // プラグイン名取得（ファイル名と合わせてください）
    const pluginName = 'Mano_EnemyHelpWindow';
    function myParams() {
        return PluginManager.parameters(pluginName);
    }

    // 表示用ヘルパー
    function toIcon(elementId) {
        return '\\[' + setting.elementIcon[elementId] + ']';
    }
    function toElementName(elementId) {        
        return $dataSystem.elements[elementId];
    }

    // 設定読み込み
    const setting = (() => {
        const param = myParams();
        const result = {
            weakLine: Number(param.WeakLine || 100) / 100,
            resistanceLine: Number(param.ResistanceLine || 100) / 100,
            weakName: String(param.WeakName || '弱点：'),
            resistanceName: String(param.ResistanceName || '耐性：'),
            displayMode: String(param.displayMode || 'text'),
            rect: {
                x: param.WindowX === '' ? NaN : Number(param.WindowX),
                y: param.WindowY === '' ? NaN : Number(param.WindowY),
                width: Number(param.WindowWidth || 300),
                height: Number(param.WindowHeight || 150)
            },
            hiddenState: Number(param.hiddenState || 0),
            hiddenIcon: Number(param.hiddenIcon || 0),
            elementIcon: []
        };
        return result;
    })();

    const elementItemFunc = setting.displayMode === 'text' ? toElementName : toIcon;

    // 属性→アイコンID対応表を作る
    function iconSampling() {
        const param = myParams();
        const array = [];
        array.length = $dataSystem.elements.length;
        for (let i = 0; i < array.length; ++i) {
            array[i] = 0;
        }

        const list = JSON.parse(param.elementIconList || '[]');
        const listLen = list.length;
        for (let i = 0; i < listLen; ++i) {
            const skillID = Number(list[i]);
            const skill = $dataSkills[skillID];
            if (!skill) continue;
            const elm = skill.damage.elementId;
            array[elm] = skill.iconIndex;
        }

        setting.elementIcon = array;
    }

    // Scene_Bootでデータ読み込み後に一度だけセットアップ
    const _Scene_Boot_start = Scene_Boot.prototype.start;
    Scene_Boot.prototype.start = function() {
        _Scene_Boot_start.call(this);
        iconSampling();
    };

    //-----------------------------------------------------------------------------
    // Window_EenmyHelp（敵情報ウィンドウ / MZ用）
    //-----------------------------------------------------------------------------

    class Window_EenmyHelp extends Window_Selectable {
        /**
         * @param {Rectangle} rect
         */
        constructor(rect) {
            super(rect);
            this._textCache = [];
            this._infoX = Math.max(
                this.textWidth(setting.weakName),
                this.textWidth(setting.resistanceName)
            );
            this._batller = null;
            this.refresh();
        }

        /**
         * MZでは windowWidth/windowHeight が必須ではないが、
         * 既存コード互換のために定義しておく
         */
        windowWidth() {
            return this.width;
        }

        windowHeight() {
            return this.height;
        }

        /**
         * @param {Game_Battler} battler 
         */
        isBattlerDataHidden(battler) {
            if (!battler) return true;
            // メモ欄タグでも非表示にする（拡張）
            const enemy = battler.enemy && battler.enemy();
            const actor = battler.actor && battler.actor();
            const meta = (enemy && enemy.meta) || (actor && actor.meta) || {};
            if (meta.EnemyHelpHide) return true;

            if (setting.hiddenState > 0 && battler.isStateAffected(setting.hiddenState)) {
                return true;
            }
            return false;
        }

        /**
         * @return {Game_Battler}
         */
        battler() {
            return this._batller;
        }

        /**
         * @param {Game_Battler} battler 
         */
        createWeakList(battler) {
            const len = $dataSystem.elements.length;
            const weakList = [];
            const resistanceList = [];
            for (let i = 1; i < len; i += 1) {
                const rate = battler.elementRate(i);
                if (rate > setting.weakLine) {
                    weakList.push(i);
                } else if (rate < setting.resistanceLine) {
                    resistanceList.push(i);
                }
            }
            return {
                weakList: weakList,
                resistanceList: resistanceList
            };
        }

        clearCache() {
            this._textCache.length = 0;
        }

        drawBaseText() {
            this.contents.clear();
            this.drawText(setting.weakName, 0, 0, this.contentsWidth());
            this.drawText(setting.resistanceName, 0, this.lineHeight(), this.contentsWidth());
        }

        /**
         * @param {number} x 
         * @param {number} y 
         * @param {number[]} elementList 
         */
        drawIcons(x, y, elementList) {
            for (let i = 0; i < elementList.length; ++i) {
                const iconId = setting.elementIcon[elementList[i]];
                if (iconId > 0) {
                    this.drawIcon(iconId, x + ImageManager.iconWidth * i, y);
                }
            }
        }

        /**
         * @param {number} x 
         * @param {number} y 
         * @param {number[]} elementList 
         */
        drawElementText(x, y, elementList) {
            let str = '';
            for (let i = 0; i < elementList.length; ++i) {
                const index = elementList[i];
                str += $dataSystem.elements[index];
                if (i < elementList.length - 1) {
                    str += ' ';
                }
            }
            this.drawText(str, x, y, this.contentsWidth() - x);
        }

        drawDataUnknow() {
            if (setting.displayMode === 'icon') {
                const iconId = setting.hiddenIcon;
                if (iconId) {
                    this.drawIcon(iconId, this._infoX, 0);
                    this.drawIcon(iconId, this._infoX, this.lineHeight());
                }
            } else {
                const textUnknow = '????';
                this.drawText(textUnknow, this._infoX, 0, this.contentsWidth() - this._infoX);
                this.drawText(textUnknow, this._infoX, this.lineHeight(), this.contentsWidth() - this._infoX);
            }
        }

        drawItem() {
            const battler = this.battler();
            if (!battler || this.isBattlerDataHidden(battler)) {
                this.drawDataUnknow();
                return;
            }

            const list = this.createWeakList(battler);
            if (setting.displayMode === 'icon') {
                this.drawIcons(this._infoX, 0, list.weakList);
                this.drawIcons(this._infoX, this.lineHeight(), list.resistanceList);
            } else {
                this.drawElementText(this._infoX, 0, list.weakList);
                this.drawElementText(this._infoX, this.lineHeight(), list.resistanceList);
            }
        }

        refresh() {
            this.drawBaseText();
            this.drawItem();
        }

        /**
         * @param {Game_Battler} enemyBattler 
         */
        setBattler(enemyBattler) {
            this._batller = enemyBattler || null;
            this.refresh();
        }
    }

    //-----------------------------------------------------------------------------
    // Scene_Battle への組み込み（MZ用）
    //-----------------------------------------------------------------------------

    const _Scene_Battle_createAllWindows = Scene_Battle.prototype.createAllWindows;
    Scene_Battle.prototype.createAllWindows = function() {
        _Scene_Battle_createAllWindows.call(this);

        const r = setting.rect;
        const autoX = Graphics.boxWidth - r.width;
        const autoY = this._enemyWindow ? (this._enemyWindow.y - r.height) : 0;
        const x = !Number.isNaN(r.x) ? r.x : autoX;
        const y = !Number.isNaN(r.y) ? r.y : autoY;

        const rect = new Rectangle(x, y, r.width, r.height);
        const help = new Window_EenmyHelp(rect);
        help.hide();
        this.addWindow(help);
        this._enemyHelpWindow_MA = help;

        if (this._enemyWindow) {
            this._enemyWindow.setEnemyHelpWindow(help);
        }
    };

    Scene_Battle.prototype.isInputCompleted = function() {
        return !BattleManager.isInputting();
    };

    Scene_Battle.prototype.onInputComplete = function() {
        if (this._enemyHelpWindow_MA) {
            this._enemyHelpWindow_MA.clearCache();
        }
    };

    const _Scene_Battle_selectNextCommand = Scene_Battle.prototype.selectNextCommand;
    Scene_Battle.prototype.selectNextCommand = function() {
        _Scene_Battle_selectNextCommand.call(this);
        if (this.isInputCompleted()) {
            this.onInputComplete();
        }
    };

    //-----------------------------------------------------------------------------
    // Window_BattleEnemy 拡張（ヘルプウィンドウ連動）
    //-----------------------------------------------------------------------------

    /**
     * @param {Window_EenmyHelp} help
     */
    Window_BattleEnemy.prototype.setEnemyHelpWindow = function(help) {
        const w = help.windowWidth();
        const h = help.windowHeight();
        const x = this.x + this.width - w;
        const y = this.y - h;
        help.move(x, y, w, h);
        this._enemyHelpWindow_MA = help;
    };

    const _Window_BattleEnemy_show = Window_BattleEnemy.prototype.show;
    Window_BattleEnemy.prototype.show = function() {
        _Window_BattleEnemy_show.call(this);
        if (this._enemyHelpWindow_MA) {
            this._enemyHelpWindow_MA.show();
        }
    };

    const _Window_BattleEnemy_hide = Window_BattleEnemy.prototype.hide;
    Window_BattleEnemy.prototype.hide = function() {
        _Window_BattleEnemy_hide.call(this);
        if (this._enemyHelpWindow_MA) {
            this._enemyHelpWindow_MA.hide();
        }
    };

    const _Window_BattleEnemy_callUpdateHelp = Window_BattleEnemy.prototype.callUpdateHelp;
    Window_BattleEnemy.prototype.callUpdateHelp = function() {
        _Window_BattleEnemy_callUpdateHelp.call(this);
        if (this._enemyHelpWindow_MA) {
            this._enemyHelpWindow_MA.setBattler(this.enemy());
        }
    };

})();
