/*:
 * @target MZ
 * @plugindesc [Step Random CE] 歩行時ランダムでコモンイベント発動＋クールタイム／ビジー時キュー対応 v1.1.0＋既存セーブ自動反映
 * @author You
 *
 * @help
 * 旧セーブをロードしても、現行のプラグインパラメータ値を自動でセーブに適用します。
 * 手動で適用したい場合はプラグインコマンド「ReloadFromParams」を利用してください。
 *
 * ■元の説明は割愛（機能は従来どおり）■
 *
 * @param EnableOnNewGame
 * @text 新規ゲームで有効化
 * @type boolean @on 有効 @off 無効
 * @default true
 *
 * @param EnableSwitchId
 * @text 有効/無効スイッチID
 * @type switch
 * @desc 0なら無視。指定すると、ONのときのみ機能します。
 * @default 0
 *
 * @param CommonEventId
 * @text コモンイベントID
 * @type common_event
 * @default 1
 *
 * @param TriggerChance
 * @text 抽選確率（%）
 * @type number @min 0 @max 100
 * @default 10
 *
 * @param CooldownSteps
 * @text クールタイム歩数
 * @type number @min 0
 * @default 30
 *
 * @param MirrorToVariableId
 * @text 残りクールタイムを書き出す変数
 * @type variable
 * @default 0
 *
 * @param QueueIfBusy
 * @text ビジー時は後で発動(キュー)
 * @type boolean @on する @off しない
 * @default true
 *
 * @param DebugLog
 * @text デバッグログ
 * @type boolean @on 出力する @off 出力しない
 * @default false
 *
 * @command SetEnabled
 * @text 有効/無効の切替
 * @arg enabled
 * @text 有効にする
 * @type boolean
 * @default true
 *
 * @command SetChance
 * @text 抽選確率(%)を変更
 * @arg chance
 * @type number @min 0 @max 100
 * @default 10
 *
 * @command SetCooldown
 * @text クールタイム歩数を変更
 * @arg steps
 * @type number @min 0
 * @default 30
 *
 * @command SetCommonEvent
 * @text コモンイベントIDを変更
 * @arg commonEventId
 * @type common_event
 * @default 1
 *
 * @command ResetCooldown
 * @text クールタイムを即時リセット
 *
 * @command ForceTrigger
 * @text 即時発動（確率・クール無視）
 *
 * @command ReloadFromParams
 * @text パラメータをセーブに再適用
 * @desc 現在のプラグインパラメータ値を、既存セーブの設定へ反映します。
 */

(() => {
  const PLUGIN_NAME = "OnStepCommonEventMZ";

  // ---- Parameters ----
  const params = PluginManager.parameters(PLUGIN_NAME);
  const P = {
    enableOnNewGame: String(params.EnableOnNewGame || "true") === "true",
    enableSwitchId: Number(params.EnableSwitchId || 0),
    commonEventId: Number(params.CommonEventId || 1),
    chance: Number(params.TriggerChance || 10),
    cooldownSteps: Number(params.CooldownSteps || 30),
    mirrorVarId: Number(params.MirrorToVariableId || 0),
    queueIfBusy: String(params.QueueIfBusy || "true") === "true",
    debug: String(params.DebugLog || "false") === "true",
  };

  const log = (...args) => { if (P.debug && console && console.log) console.log(`[${PLUGIN_NAME}]`, ...args); };

  function clampPercent(n) {
    n = Number(n || 0);
    if (n < 0) n = 0;
    if (n > 100) n = 100;
    return n;
  }

  function isBusyNow() {
    return !!(
      ($gameMap.isEventRunning && $gameMap.isEventRunning()) ||
      ($gameMessage.isBusy && $gameMessage.isBusy()) ||
      ($gameTemp.isCommonEventReserved && $gameTemp.isCommonEventReserved())
    );
  }

  function mirrorCooldownVariable() {
    if (P.mirrorVarId > 0) {
      const s = $gameSystem.stepCEData();
      $gameVariables.setValue(P.mirrorVarId, s.cooldownRemain);
    }
  }

  function tryReserveCE(ceId, s) {
    if (!ceId || ceId <= 0) return false;
    if (isBusyNow()) return false;
    $gameTemp.reserveCommonEvent(ceId);
    s.cooldownRemain = s.cooldownSteps;
    mirrorCooldownVariable();
    log(`Triggered CE ${ceId}. Cooldown set to ${s.cooldownRemain} steps.`);
    return true;
  }

  // ---- Init holder on Game_System ----
  const sysInit = function(system) {
    system._stepCE = system._stepCE || {};
    const s = system._stepCE;
    if (s._initialized) return;

    s.enabled = !!P.enableOnNewGame;
    s.commonEventId = Math.max(0, Number(P.commonEventId || 0));
    s.chance = clampPercent(P.chance);
    s.cooldownSteps = Math.max(0, Number(P.cooldownSteps || 0));
    s.cooldownRemain = 0;
    s.pendingTrigger = false; // ビジーで見送った当選を1回だけ後で実行するフラグ
    s._initialized = true;

    try { log("initialized", JSON.parse(JSON.stringify(s))); } catch (_e) { log("initialized"); }
  };

  // ---- Game_System patch ----
  const _Game_System_initialize = Game_System.prototype.initialize;
  Game_System.prototype.initialize = function() {
    _Game_System_initialize.call(this);
    sysInit(this);
  };

  Game_System.prototype.stepCEData = function() {
    sysInit(this);
    return this._stepCE;
  };

  Game_System.prototype.stepCEEnabled = function() {
    const s = this.stepCEData();
    if (P.enableSwitchId > 0) {
      return s.enabled && $gameSwitches.value(P.enableSwitchId);
    }
    return s.enabled;
  };

  Game_System.prototype.setStepCEEnabled = function(flag) {
    const s = this.stepCEData();
    s.enabled = !!flag;
  };

  Game_System.prototype.setStepCEChance = function(chance) {
    const s = this.stepCEData();
    s.chance = clampPercent(chance);
  };

  Game_System.prototype.setStepCECooldown = function(steps) {
    const s = this.stepCEData();
    s.cooldownSteps = Math.max(0, Number(steps || 0));
  };

  Game_System.prototype.setStepCECommonEvent = function(id) {
    const s = this.stepCEData();
    s.commonEventId = Math.max(0, Number(id || 0));
  };

  Game_System.prototype.resetStepCECooldown = function() {
    const s = this.stepCEData();
    s.cooldownRemain = 0;
    mirrorCooldownVariable();
  };

  // ---- ★ 追加：既存セーブへ現行パラメータを適用するAPI ----
  Game_System.prototype.applyStepCEParams = function() {
    const s = this.stepCEData();
    s.enabled       = !!P.enableOnNewGame;
    s.commonEventId = Math.max(0, Number(P.commonEventId || 0));
    s.chance        = clampPercent(P.chance);
    s.cooldownSteps = Math.max(0, Number(P.cooldownSteps || 0));
    // 必要なら残りCTもリセット：
    // s.cooldownRemain = 0;
    mirrorCooldownVariable();
    try { log("Applied params to save", JSON.parse(JSON.stringify(s))); } catch (_e) { log("Applied params to save"); }
  };

  // ---- Core: called on each player step ----
  Game_System.prototype.onPlayerStepForCE = function() {
    const s = this.stepCEData();

    // クールタイム進行
    if (s.cooldownRemain > 0) {
      s.cooldownRemain = Math.max(0, s.cooldownRemain - 1);
      mirrorCooldownVariable();
      return;
    }

    // 有効判定
    if (!this.stepCEEnabled()) return;

    const busy = isBusyNow();

    // 既に保留（pending）があり、今は非ビジーなら先に消化
    if (s.pendingTrigger && !busy) {
      if (tryReserveCE(s.commonEventId, s)) {
        s.pendingTrigger = false;
        return;
      }
    }

    // 通常抽選
    const ceId = s.commonEventId;
    if (!ceId || ceId <= 0) return;

    const roll = Math.random() * 100;
    if (roll < s.chance) {
      if (!busy) {
        tryReserveCE(ceId, s);
      } else if (P.queueIfBusy) {
        s.pendingTrigger = true;
        log("Queued trigger while busy.");
      }
    }
  };

  // ---- Hook: Game_Player.increaseSteps ----
  const _Game_Player_increaseSteps = Game_Player.prototype.increaseSteps;
  Game_Player.prototype.increaseSteps = function() {
    _Game_Player_increaseSteps.call(this);
    // 乗り物移動中はスキップ（乗り物でも抽選したい場合は下行を削除）
    if (this.isInVehicle && this.isInVehicle()) return;
    $gameSystem.onPlayerStepForCE();
  };

  // ---- Plugin Commands ----
  PluginManager.registerCommand(PLUGIN_NAME, "SetEnabled", args => {
    const enabled = String(args.enabled || "true") === "true";
    $gameSystem.setStepCEEnabled(enabled);
    log("SetEnabled:", enabled);
  });

  PluginManager.registerCommand(PLUGIN_NAME, "SetChance", args => {
    $gameSystem.setStepCEChance(Number(args.chance || 0));
    log("SetChance:", $gameSystem.stepCEData().chance);
  });

  PluginManager.registerCommand(PLUGIN_NAME, "SetCooldown", args => {
    $gameSystem.setStepCECooldown(Number(args.steps || 0));
    log("SetCooldown:", $gameSystem.stepCEData().cooldownSteps);
  });

  PluginManager.registerCommand(PLUGIN_NAME, "SetCommonEvent", args => {
    $gameSystem.setStepCECommonEvent(Number(args.commonEventId || 0));
    log("SetCommonEvent:", $gameSystem.stepCEData().commonEventId);
  });

  PluginManager.registerCommand(PLUGIN_NAME, "ResetCooldown", _args => {
    $gameSystem.resetStepCECooldown();
    log("ResetCooldown");
  });

  PluginManager.registerCommand(PLUGIN_NAME, "ForceTrigger", _args => {
    const s = $gameSystem.stepCEData();
    const ceId = s.commonEventId;
    if (!ceId || ceId <= 0) return;
    if (tryReserveCE(ceId, s)) {
      s.pendingTrigger = false;
    }
  });

  // ---- ★ 追加：現行パラメータを手動適用するコマンド ----
  PluginManager.registerCommand(PLUGIN_NAME, "ReloadFromParams", _args => {
    if ($gameSystem && $gameSystem.applyStepCEParams) {
      $gameSystem.applyStepCEParams();
    }
  });

  // ---- ★ 追加：セーブロード時に自動適用 ----
  const _DataManager_extractSaveContents = DataManager.extractSaveContents;
  DataManager.extractSaveContents = function(contents) {
    _DataManager_extractSaveContents.call(this, contents);
    if ($gameSystem && $gameSystem.applyStepCEParams) {
      $gameSystem.applyStepCEParams();
    }
  };

})();
