/*:
 * @target MZ
 * @plugindesc v1.0 使用中プラグインの「名前 と 制作者」を列挙してテキスト出力（クリップボード＆txt保存）
 * @author ChatGPT
 * @help PluginListerMZ.js
 *
 * プロジェクトで「プラグイン管理」に登録されているプラグインを走査し、
 * それぞれの「プラグイン名」「制作者（作者名）」をテキストで列挙します。
 * 生成したテキストはクリップボードにコピーされ、任意でプロジェクト直下に
 * テキストファイルとして保存できます（NW.js実行時）。
 *
 * ■プラグインコマンド
 *  @command CopyPluginList
 *  @text 一覧を生成してコピー/保存
 *  @desc 使用中プラグイン一覧（名前 と 制作者）を生成し、クリップボードにコピー。任意でtxt保存。
 *
 *  @arg includeDisabled
 *  @text 無効プラグインも含める
 *  @type boolean
 *  @default false
 *
 *  @arg includeVersion
 *  @text バージョンを併記（@plugindesc から推定）
 *  @type boolean
 *  @default true
 *
 *  @arg separator
 *  @text 区切り文字
 *  @type string
 *  @default  - 
 *
 *  @arg saveFileName
 *  @text 保存ファイル名（空欄で保存しない）
 *  @type string
 *  @default PluginList.txt
 *
 */
(function(){
  'use strict';
  const PN='PluginListerMZ';

  const readTextHead = function(path){
    try{
      if (!Utils.isNwjs()) return null;
      const fs = require('fs');
      const fd = fs.openSync(path, 'r');
      const buf = Buffer.alloc(64*1024);
      const bytes = fs.readSync(fd, buf, 0, buf.length, 0);
      fs.closeSync(fd);
      return buf.slice(0, bytes).toString('utf8');
    }catch(e){ return null; }
  };

  const guessAuthor = function(text){
    if (!text) return null;
    const mHead = text.match(/\/\*[\s\S]*?\*\//);
    const head = mHead ? mHead[0] : text;
    const tries = [
      /@author\s*([^\r\n]+)/i,
      /^\s*\*\s*author\s*[:：]\s*([^\r\n]+)/im,
      /^\s*\/\/\s*author\s*[:：]\s*([^\r\n]+)/im,
      /作者\s*[:：]\s*([^\r\n]+)/i,
      /Author\s*[:：]\s*([^\r\n]+)/i,
    ];
    for (const rx of tries){
      const m = head.match(rx);
      if (m && m[1]) return m[1].trim();
    }
    return null;
  };

  const guessVersion = function(text){
    if (!text) return null;
    const mHead = text.match(/\/\*[\s\S]*?\*\//);
    const head = mHead ? mHead[0] : text;
    const m1 = head.match(/@plugindesc[^\r\n]*?(v[\d._\-]+[^ \r\n]*)/i);
    if (m1 && m1[1]) return m1[1].trim();
    const m2 = head.match(/\bv\d+(?:\.\d+)*\b/i);
    if (m2 && m2[0]) return m2[0].trim();
    return null;
  };

  const buildList = function(includeDisabled, includeVersion, sep){
    const arr = [];
    const base = (Utils.isNwjs() ? require('path').join(require('path').dirname(process.mainModule.filename), 'js', 'plugins') : null);
    for (const p of $plugins){
      if (!includeDisabled && !p.status) continue;
      const name = p.name || '(NoName)';
      let author = null, version = null;
      if (Utils.isNwjs()){
        try{
          const path = require('path').join(base, name + '.js');
          const text = readTextHead(path);
          author = guessAuthor(text);
          if (includeVersion) version = guessVersion(text);
        }catch(e){}
      }
      if (!author || author.length===0) author = 'Unknown';
      const left = includeVersion && version ? `${name} ${version}` : `${name}`;
      arr.push(`${left}${sep}${author}`);
    }
    arr.sort((a,b)=> a.localeCompare(b,'ja'));
    return arr.join('\n');
  };

  const copyToClipboard = function(text){
    if (!text || !Utils.isNwjs()) return false;
    try{
      if (window.nw && nw.Clipboard){
        const clip = nw.Clipboard.get();
        clip.set(text, 'text');
        return true;
      }
    }catch(e){ return false; }
  };

  const saveToFile = function(text, fileName){
    if (!text || !fileName || !Utils.isNwjs()) return null;
    try{
      const path = require('path');
      const fs = require('fs');
      const root = path.dirname(process.mainModule.filename);
      const out = path.join(root, fileName);
      fs.writeFileSync(out, text, 'utf8');
      return out;
    }catch(e){ return null; }
  };

  PluginManager.registerCommand(PN, 'CopyPluginList', args=>{
    const includeDisabled = String(args.includeDisabled||'false').toLowerCase()==='true';
    const includeVersion  = String(args.includeVersion||'true').toLowerCase()==='true';
    const separator       = String(args.separator ?? ' - ');
    const saveFileName    = String(args.saveFileName||'').trim();

    const text = buildList(includeDisabled, includeVersion, separator);
    if (copyToClipboard(text)) $gameMessage.add('プラグイン一覧をクリップボードにコピーしました。');
    const saved = saveToFile(text, saveFileName);
    if (saved) $gameMessage.add(`${saveFileName} を保存しました。`);
    console.log('=== PluginListerMZ ===\n' + text);
  });
})();