/*:
 * @target MZ
 * @plugindesc v1.2 戦闘敗北→ゲームオーバー遷移をフック。マップ復帰後に“空き”を待ってコモンイベントを確実起動
 * @author you
 *
 * @help
 * 敗北時にGameOverへの遷移を横取りし、マップへ戻して指定のコモンイベントを実行します。
 * v1.2では、マップ復帰直後に他イベントで詰まっていても、一定フレーム待って
 * “空いている瞬間”に予約→実行する遅延キューを実装しました。
 *
 * @param commonEventId
 * @text 実行するコモンイベントID
 * @type common_event
 * @default 0
 *
 * @param applyWhenCanLoseIsFalse
 * @text canLose=FALSE時のみ適用
 * @type boolean
 * @on はい
 * @off いいえ（常に適用）
 * @default true
 *
 * @param reviveMode
 * @text 敗北後の味方状態
 * @type select
 * @option 触らない（戦闘時のまま） @value none
 * @option 全員をHP1で蘇生       @value hp1
 * @option 全回復で蘇生           @value full
 * @default hp1
 *
 * @param fadeOutIn
 * @text マップ復帰時にフェード
 * @type boolean
 * @on する
 * @off しない
 * @default false
 *
 * @param setSwitchId
 * @text 実行前にONにするスイッチ
 * @type switch
 * @default 0
 *
 * @param deferFrames
 * @text 予約を開始するまでの待ちフレーム
 * @type number
 * @min 0
 * @desc マップに戻ってから予約監視を開始するまでの待ち。0〜10程度推奨。
 * @default 1
 *
 * @param waitUntilNoEvent
 * @text 実行中イベントが無い瞬間まで待つ
 * @type boolean
 * @on はい（安全）
 * @off いいえ（即予約）
 * @default true
 *
 * @param waitTimeoutFrames
 * @text 最大待ちフレーム
 * @type number
 * @min 1
 * @desc 空き待ちの上限（超えたら強制予約）
 * @default 600
 *
 * @param debugLog
 * @text デバッグログ
 * @type boolean
 * @on 出す
 * @off 出さない
 * @default false
 */

(() => {
  'use strict';

  const PLUGIN_NAME = document.currentScript.src.split('/').pop().replace(/\.js$/,'');
  const P = PluginManager.parameters(PLUGIN_NAME);
  const CE_ID = Number(P.commonEventId || 0);
  const ONLY_CANLOSE_FALSE = P.applyWhenCanLoseIsFalse === 'true';
  const REVIVE_MODE = String(P.reviveMode || 'hp1'); // none/hp1/full
  const FADE = P.fadeOutIn === 'true';
  const SWITCH_ID = Number(P.setSwitchId || 0);
  const DEFER_FRAMES = Math.max(0, Number(P.deferFrames || 1));
  const WAIT_NO_EVENT = P.waitUntilNoEvent === 'true';
  const WAIT_TIMEOUT = Math.max(1, Number(P.waitTimeoutFrames || 600));
  const DEBUG = P.debugLog === 'true';
  const log = (...a)=>{ if(DEBUG) console.log(`[${PLUGIN_NAME}]`, ...a); };

  // グローバルな“起動待ち”ペンディングをGame_Systemに持たせる
  const keyPending = '_mdPostDefeatCEPending';
  const keyCounter = '_mdPostDefeatWaitCounter';
  const keyDelay   = '_mdPostDefeatDefer';

  function setPendingCommonEvent(id) {
    $gameSystem[keyPending] = id;
    $gameSystem[keyCounter] = 0;
    $gameSystem[keyDelay]   = DEFER_FRAMES;
    log('Pending CE set:', id, 'defer:', DEFER_FRAMES);
  }
  function clearPending() {
    $gameSystem[keyPending] = 0;
    $gameSystem[keyCounter] = 0;
    $gameSystem[keyDelay]   = 0;
  }
  function hasPending() {
    return !!$gameSystem[keyPending];
  }

  function reviveParty(mode) {
    if (mode === 'none') return;
    $gameParty.members().forEach(actor => {
      if (mode === 'full') {
        actor.recoverAll();
        if (actor.hp <= 0) actor.setHp(1);
      } else if (mode === 'hp1') {
        if (actor.hp <= 0) actor.setHp(1);
      }
    });
  }

  // canLoseのスナップショット保持
  const _BattleManager_setup = BattleManager.setup;
  BattleManager.setup = function(troopId, canEscape, canLose) {
    this._md_canLoseSnapshot = !!canLose;
    _BattleManager_setup.call(this, troopId, canEscape, canLose);
  };

  // GameOver遷移フック
  const _SceneManager_goto = SceneManager.goto;
  SceneManager.goto = function(sceneClass) {
    try {
      const inBattle = this._scene instanceof Scene_Battle;
      const toGameover = (sceneClass === Scene_Gameover);
      if (inBattle && toGameover) {
        const canLose = !!(BattleManager._md_canLoseSnapshot);
        if (!ONLY_CANLOSE_FALSE || !canLose) {
          if (CE_ID > 0) {
            // BGM/BGS復帰・蘇生・スイッチON
            if (BattleManager.replayBgmAndBgs) BattleManager.replayBgmAndBgs();
            reviveParty(REVIVE_MODE);
            if (SWITCH_ID > 0) $gameSwitches.setValue(SWITCH_ID, true);

            // 予約は“今”ではなく、マップ復帰後に空きを見て実行
            setPendingCommonEvent(CE_ID);
          }
          // 戦闘終了→マップへ
          if (BattleManager.endBattle) BattleManager.endBattle(2);
          if (FADE && this._scene) this._scene.startFadeOut(12, false);
          this.pop();
          if (FADE && this._scene) this._scene.startFadeIn(12, false);
          log('Intercepted GameOver -> return to map');
          return;
        }
      }
    } catch(e) {
      console.error(`[${PLUGIN_NAME}] hook error`, e);
    }
    _SceneManager_goto.call(this, sceneClass);
  };

  // マップ側で“空いている瞬間”に予約→実行させる監視
  const _Scene_Map_update = Scene_Map.prototype.update;
  Scene_Map.prototype.update = function() {
    _Scene_Map_update.call(this);
    if (!hasPending()) return;

    // 待機フレーム（復帰直後の読み込み/並列初期化のため）
    if ($gameSystem[keyDelay] > 0) {
      $gameSystem[keyDelay]--;
      return;
    }

    // 空き待ち
    const counter = ($gameSystem[keyCounter] || 0) + 1;
    $gameSystem[keyCounter] = counter;

    const noEventRunning =
      !$gameMap.isEventRunning() &&
      !this.isBusy?.() &&
      !$gameMessage.isBusy();

    if (WAIT_NO_EVENT && !noEventRunning && counter < WAIT_TIMEOUT) {
      // まだ詰まっているので待つ
      if (counter % 60 === 0) log('Waiting free slot...', counter, 'frames');
      return;
    }

    // 予約→即実行へ（Game_MapのStartingEventへ渡す）
    const pendingId = $gameSystem[keyPending];
    clearPending();

    if (pendingId > 0) {
      log('Reserve CommonEvent', pendingId, '(forced:', !noEventRunning || counter >= WAIT_TIMEOUT, ')');
      $gameTemp.reserveCommonEvent(pendingId);
      // 直ちに1回だけsetupStartingEventを促す
      if (!$gameMap.isEventRunning()) {
        $gameMap.setupStartingEvent();
      }
    }
  };

})();
