/*:
 * @target MZ
 * @plugindesc 重み付きランダムケース実行プラグイン（範囲指定 & default 対応版）
 * @author YourName
 *
 * @help
 * ■概要
 * イベント内で <case:n weight=X> ～ <endcase> で囲った複数ブロックのうち、
 * 指定範囲の番号から重み付きランダムで1つ選んで実行します。
 * 範囲に該当がなければ <case:default> にフォールバックします。
 *
 * ■書き方（イベントコマンドの「注釈」）
 *   <case:0 weight=10>
 *   （ここに実行したいコマンド）
 *   <case:1 weight=30>
 *   （ここに実行したいコマンド）
 *   <case:2>        ← weight省略可
 *   （ここに実行したいコマンド）
 *   <case:default>  ← 省略可（範囲外や該当なしの時に実行）
 *   （ここに実行したいコマンド）
 *   <endcase>
 *
 * ■使い方（配置が重要）
 * 1) プラグインコマンド「ランダムケース実行（randomCase）」を置く
 * 2) その“下”に上記の <case:…> ブロック群を書く（同じイベント内）
 *    ※本プラグインはコマンド行以降を前方に走査します。
 *
 * ■重みのルール
 * ・weight は小数OK。未指定のケースには「(100 - 指定合計)を等分」して自動付与。
 * ・指定合計が100を超えると、100を上限として先頭から切り上げられ、
 *   後続ケースに確率が回らないため、合計は100以下を推奨。
 *
 * ■範囲指定
 * ・rangeStart ～ rangeEnd は両端を含む数値ケースが対象。
 * ・範囲内に数値ケースが無い → <case:default> があればそれを実行、
 *   無ければ何も実行しません（ブロックはスキップされます）。
 *
 * ■注意
 * ・<endcase> は1つのグループの終端です。グループは複数置けますが、
 *   その都度プラグインコマンドを呼び出してください。
 * ・本プラグインはRPGツクールMZ専用です（MV未対応）。
 *
 * ■更新履歴
 * v1.0.0  初版（範囲指定・default対応／未指定重みの自動補填）
 *
 * @command randomCase
 * @text ランダムケース実行
 * @desc 指定範囲の <case:n> から重み付きランダムで1つ実行します
 *
 * @arg rangeStart
 * @text 開始ケース番号
 * @type number
 * @default 0
 *
 * @arg rangeEnd
 * @text 終了ケース番号
 * @type number
 * @default 3
 */

(() => {
    const pluginName = "RandomWeightedCase";

    // プラグインコマンド登録
    PluginManager.registerCommand(pluginName, "randomCase", function(args) {
        const interpreter = this;
        const list = getCurrentEventCommandList(interpreter);

        // ブロック解析
        const blocks = parseCaseBlocks(list, interpreter._index);

        // 範囲情報
        const rangeStart = Number(args.rangeStart || 0);
        const rangeEnd = Number(args.rangeEnd || 3);

        // 重み付き抽選
        const selectedKey = selectWeightedCase(blocks, rangeStart, rangeEnd);

        // 選ばれたブロックを実行
        if (selectedKey && blocks[selectedKey]) {
            executeEventCommandList(interpreter, blocks[selectedKey].list);
        }

        // 抽選対象ブロック全体をスキップ
        if (blocks._endIndex != null) {
            interpreter._index = blocks._endIndex;
        }
    });

    function getCurrentEventCommandList(interpreter) {
        return interpreter._list || [];
    }

    // parseCaseBlocks 修正版
    function parseCaseBlocks(list, startIndex) {
        const blocks = {};
        let currentCase = null;
        let buffer = [];
        let endIndex = startIndex;

        for (let i = startIndex; i < list.length; i++) {
            const cmd = list[i];
            const code = cmd.code;
            const params = cmd.parameters;

            let text = (code === 108 || code === 408) && params ? params[0] || params.text || "" : null;

            if (text) {
                const matchCase = text.match(/^<case:(\d+|default)(?:\s+weight=(\d+(\.\d+)?))?>$/i);
                const matchEnd = text.match(/^<endcase>$/i);

                if (matchCase) {
                    if (currentCase !== null) {
                        blocks[currentCase].list = buffer.slice();
                        buffer = [];
                    }
                    currentCase = matchCase[1];
                    const weight = parseFloat(matchCase[2] || "0");
                    blocks[currentCase] = { list: [], weight: weight };
                    endIndex = i + 1;
                    continue;
                } else if (matchEnd) {
                    if (currentCase !== null) {
                        blocks[currentCase].list = buffer.slice();
                        buffer = [];
                        currentCase = null;
                        endIndex = i + 1;
                    }
                    continue;
                }
            }

            if (currentCase !== null) {
                buffer.push(cmd);
            }

            endIndex = i + 1;
        }

        if (currentCase !== null) {
            blocks[currentCase].list = buffer.slice();
        }

        blocks._endIndex = endIndex;
        return blocks;
    }

    // 重み付き抽選（範囲指定 + default 対応）
    function selectWeightedCase(blocks, rangeStart, rangeEnd) {
        const allKeys = Object.keys(blocks).filter(k => k !== "_endIndex");
        const numericKeys = allKeys.filter(k => k !== "default").map(k => Number(k)).sort((a,b)=>a-b);

        // 範囲フィルタリング
        const filteredKeys = numericKeys.filter(k => k >= rangeStart && k <= rangeEnd).map(k=>k.toString());

        // 数値ケースがない場合は default にフォールバック
        if (filteredKeys.length === 0) {
            if (blocks['default']) return 'default';
            return null;
        }

        // 重み正規化
        let totalWeight = 0;
        filteredKeys.forEach(k => {
            totalWeight += blocks[k].weight || 0;
        });

        const unspecifiedKeys = filteredKeys.filter(k => !blocks[k].weight || blocks[k].weight <= 0);
        const remainingPercent = Math.max(0, 100 - totalWeight);
        const defaultWeight = unspecifiedKeys.length > 0 ? remainingPercent / unspecifiedKeys.length : 0;

        filteredKeys.forEach(k => {
            if (!blocks[k].weight || blocks[k].weight <= 0) {
                blocks[k].weight = defaultWeight;
            }
        });

        // 重み付き抽選
        const rnd = Math.random() * 100;
        let acc = 0;
        for (const k of filteredKeys) {
            acc += blocks[k].weight;
            if (rnd < acc) return k;
        }

        return filteredKeys[filteredKeys.length - 1];
    }

	function executeEventCommandList(interpreter, list) {
	  if (!list || list.length === 0) return;
	  // 元イベントIDを子インタプリタに引き継ぐ
	  interpreter.setupChild(list, interpreter._eventId);
	}

})();
