/*:
 * @target MZ
 * @plugindesc v0.9e ランダムダンジョンLite：階層/難易度/満腹度HUD・進行/自然回復・フラッシュ演出（HudColor/背景/フォント調整・動的変更・セーブ互換）＋満腹度減少ON/OFFコマンド
 * @author ChatGPT
 * @help RogueDungeonMZ.js
 *
 * ■概要
 * ダンジョン探索用の簡易ローグライクシステムを提供します。
 * ・階層を進むごとに難易度が自動で変動（確定+1 + 低確率で上下）
 * ・満腹度が減少、0で歩くとHPが減少
 * ・満腹度自然回復ゾーン（リージョンID）対応（※仕様そのまま）
 * ・低確率でHP/MPが難易度に応じて回復
 * ・HUDで階数/難易度/満腹度を左上に常時表示（難易度変動時に点滅）
 * ・プラグインコマンドで各値の操作／HUDスタイル変更／満腹度減少ON/OFFが可能
 *
 * ▼注意
 * - プラグインコマンドがエディタに出ない場合は、必ずファイル名を **RogueDungeonMZ.js** にしてください。
 *
 * ■パラメータ
 * @param StartFloor
 * @text 開始階
 * @type number
 * @default 1
 *
 * @param StartDifficulty
 * @text 開始難易度
 * @type number
 * @default 1
 *
 * @param MaxDifficulty
 * @text 最大難易度
 * @type number
 * @default 10000
 *
 * @param StartSatiety
 * @text 満腹度初期値(%)
 * @type number
 * @default 100
 *
 * @param HungerStepLoss
 * @text 歩行での満腹度減少(歩数)
 * @type number
 * @default 25
 *
 * @param StarveHpTick
 * @text 空腹ダメージ間隔(歩数)
 * @type number
 * @default 50
 * @desc 満腹度0時、この歩数ごとにHPを最大値の1%だけ減らします（最低1）。
 *
 * @param RecoverRegionId
 * @text 満腹度自然回復ゾーン用リージョンID
 * @type number
 * @default 5
 *
 * @param HudX
 * @text HUD X
 * @type number
 * @default 24
 *
 * @param HudY
 * @text HUD Y
 * @type number
 * @default 64
 *
 * @param HudColor
 * @text HUD文字色（#RRGGBB）
 * @type string
 * @default #FFFFFF
 *
 * @param HudBgOpacity
 * @text HUD背景の不透明度(0-255)
 * @type number
 * @min 0
 * @max 255
 * @default 0
 *
 * @param HudFontSize
 * @text HUDフォントサイズ
 * @type number
 * @min 8
 * @max 48
 * @default 18
 *
 * ------------------------------------------------------------
 * @command NextFloor
 * @text 階層を進める
 * @desc 次の階層へ進め、難易度変動と回復抽選を実行します。
 *
 * @command AddSatiety
 * @text 満腹度を加算
 * @desc 満腹度に値を加算します（負数で減算）
 * @arg value
 * @text 加算値
 * @type number
 * @default 10
 *
 * @command SubSatiety
 * @text 満腹度を減算
 * @desc 現在の満腹度から値を減算します
 * @arg value
 * @text 減算値
 * @type number
 * @default 1
 *
 * @command SetSatiety
 * @text 満腹度を設定
 * @desc 満腹度を指定値に設定します（0〜100）
 * @arg value
 * @text 値
 * @type number
 * @default 100
 *
 * @command SetDifficulty
 * @text 難易度を設定
 * @desc 難易度を指定値に設定します（1〜上限）
 * @arg value
 * @text 値
 * @type number
 * @default 1
 *
 * @command SetFloor
 * @text 階数を設定
 * @desc 階数を指定値に設定します（1〜）
 * @arg value
 * @text 値
 * @type number
 * @default 1
 *
 * @command AddDifficulty
 * @text 難易度を加算
 * @desc 現在の難易度に値を加算します
 * @arg value
 * @text 加算値
 * @type number
 * @default 1
 *
 * @command SubDifficulty
 * @text 難易度を減算
 * @desc 現在の難易度から値を減算します
 * @arg value
 * @text 減算値
 * @type number
 * @default 1
 *
 * @command SetHudStyle
 * @text HUDスタイル変更
 * @desc HUDの文字色/背景不透明度を変更します（空欄は変更しません）
 * @arg color
 * @text 文字色(#RRGGBB)
 * @type string
 * @default
 * @arg bgOpacity
 * @text 背景不透明度(0-255)
 * @type number
 * @min 0
 * @max 255
 * @default
 *
 * @command DisableHungerLoss
 * @text 満腹度減少を無効化
 * @desc 満腹度が歩行で減らなくなります。
 *
 * @command EnableHungerLoss
 * @text 満腹度減少を有効化
 * @desc 満腹度が歩行で通常どおり減少します。
 */

(() =>{
  'use strict';

  // -------------------------------------------------
  // Parameters
  // -------------------------------------------------
  const PN = 'RogueDungeonMZ';
  const P  = PluginManager.parameters(PN);

  const clamp =(v,a,b)=>Math.max(a,Math.min(b,v));
  const sanitizeColor = (c)=>{
    if (typeof c !== 'string') return '#FFFFFF';
    const t = c.trim();
    return /^#([0-9a-fA-F]{6}|[0-9a-fA-F]{3})$/.test(t) ? t : '#FFFFFF';
  };

  const S = {
    floor: Number(P.StartFloor||1),
    diff : Number(P.StartDifficulty||1),
    maxD : Number(P.MaxDifficulty||10000),
    sat  : Number(P.StartSatiety||100),
    stepLoss: Number(P.HungerStepLoss||25),
    starveTick: Number(P.StarveHpTick||50),
    recoverRegion: Number(P.RecoverRegionId||5),
    hudX: Number(P.HudX||24),
    hudY: Number(P.HudY||64),
    hudColor: sanitizeColor(String(P.HudColor||'#FFFFFF')),
    hudBgOpacity: Number(P.HudBgOpacity ?? 0),
    hudFontSize: Number(P.HudFontSize ?? 18),
    hungerLossEnabled: true, // 追加：満腹度減少のON/OFF（true:減る / false:減らない）
  };

  // -------------------------------------------------
  // Save & Accessor（既存セーブ互換：自己修復）
  // -------------------------------------------------
  const _setup = DataManager.setupNewGame;
  DataManager.setupNewGame = function(){
    _setup.call(this);
    $gameSystem._rogueLite = JSON.parse(JSON.stringify(S));
  };

  function st(){
    if (!$gameSystem._rogueLite) {
      $gameSystem._rogueLite = JSON.parse(JSON.stringify(S));
    }
    return $gameSystem._rogueLite;
  }

  // -------------------------------------------------
  // Steps: hunger & starving & recover region
  // （回復リージョン内はreturnで通常処理停止のまま＝仕様据え置き）
  // -------------------------------------------------
  const _increaseSteps = Game_Player.prototype.increaseSteps;
  Game_Player.prototype.increaseSteps = function(){
    _increaseSteps.call(this);
    const o = st();

    // 満腹度自然回復ゾーン（RecoverRegionId一致）なら+1（最大100）
    if ($gameMap.regionId(this.x, this.y) === o.recoverRegion){
      o.sat = clamp(o.sat+1,0,100);
      return; // 満腹/餓死カウンタも進めない
    }

    // 通常の満腹度減少（stepLoss歩ごとに-1）※減少無効時は減らさない
    o._hungerStep = (o._hungerStep||0)+1;
    if (o.hungerLossEnabled && o._hungerStep >= o.stepLoss){
      o._hungerStep = 0;
      o.sat = clamp(o.sat-1,0,100);
    }

    // 空腹（sat<=0）のとき一定歩数ごとに割合ダメージ
    o._starveStep = (o._starveStep||0)+1;
    if (o.sat<=0 && o._starveStep >= o.starveTick){
      o._starveStep = 0;
      $gameParty.members().forEach(a=>{
        a.gainHp(-Math.max(1,Math.floor(a.mhp*0.01)));
      });
    }
  };

  // -------------------------------------------------
  // API: floor progression & setters
  // -------------------------------------------------
  window.RogueLite = {
    nextFloor(){
      const o = st();
      o.floor += 1;

      // 難易度は確定で+1、10%でさらに+1〜+3、続く10%で-1〜-3
      let delta = 1;
      const r = Math.random();
      if (r < 0.10) delta += 1 + Math.floor(Math.random()*3);     // 難化
      else if (r < 0.20) delta -= (1 + Math.floor(Math.random()*3)); // 易化
      o.diff = clamp(o.diff + delta, 1, o.maxD);

      // 30%でHP/MP回復（難易度比で0.1〜最大0.5）
      if (Math.random()<0.30){
        const ratio = clamp(0.1 + o.diff/o.maxD*0.2, 0.05, 0.5);
        $gameParty.members().forEach(a=>{
          a.gainHp(Math.floor(a.mhp*ratio));
          a.gainMp(Math.floor(a.mmp*ratio));
        });
        o._lastRecover = Math.round(ratio*100);
      } else {
        o._lastRecover = 0;
      }

      o._lastDelta = delta;

      // HUD点滅
      if (SceneManager._scene && SceneManager._scene._rogueHud){
        SceneManager._scene._rogueHud.flashHud();
      }
    },

    addSatiety(n){ const o=st(); o.sat = clamp(o.sat + Number(n||0), 0, 100); },
    setSatiety(n){ const o=st(); o.sat = clamp(Number(n||0), 0, 100); },
    setDifficulty(n){ const o=st(); o.diff = clamp(Number(n||1), 1, o.maxD); o._lastDelta = 0; },
    setFloor(n){ const o=st(); o.floor = Math.max(1, Number(n||1)); }
  };

  // -------------------------------------------------
  // HUD (HudColor/HudBgOpacity/HudFontSize + auto resize + redraw optimize)
  // -------------------------------------------------
  function RogueHud(){ this.initialize(...arguments); }
  RogueHud.prototype = Object.create(Window_Base.prototype);
  RogueHud.prototype.constructor = RogueHud;

  RogueHud.prototype.initialize = function(){
    const o = st();
    const x = o.hudX, y = o.hudY;
    Window_Base.prototype.initialize.call(this, new Rectangle(x, y, 180, 72));
    this.opacity = clamp(o.hudBgOpacity, 0, 255); // 背景不透明度
    this.contents.fontSize = clamp(o.hudFontSize, 8, 48);
    this._flash = 0;
    this._hudColor = sanitizeColor(o.hudColor);
    this._lastKey = "";
    this._lastPaintOpacity = 255;
    this.refresh(true);
  };

  RogueHud.prototype.flashHud = function(){ this._flash = 30; };

  RogueHud.prototype.update = function(){
    Window_Base.prototype.update.call(this);

    // 点滅演出
    if (this._flash > 0){
      this._flash--;
      this.contents.paintOpacity = (this._flash % 6 < 3 ? 255 : 64);
    } else {
      this.contents.paintOpacity = 255;
    }

    const force = (this._flash > 0) || (this.contents.paintOpacity !== this._lastPaintOpacity);
    this._lastPaintOpacity = this.contents.paintOpacity;
    this.refresh(force);
  };

  RogueHud.prototype.refresh = function(forceLayout = false){
    const o = st();

    const textFloor = `階: ${o.floor}`;
    const textDiff  = `難易度: ${o.diff} (Δ${o._lastDelta||0})`;
    const textSat   = `満腹度: ${o.sat}%`;

    const key = [
      textFloor, textDiff, textSat,
      this.x, this.y, this._hudColor, this.contents.fontSize,
      this.opacity
    ].join('|');

    if (!forceLayout && key === this._lastKey){
      this.contents.clear();
      this._drawLines(textFloor, textDiff, textSat);
      return;
    }
    this._lastKey = key;

    // ---- 自動サイズ調整 ----
    this.contents.clear();
    const w1 = this.textWidth(textFloor);
    const w2 = this.textWidth(textDiff);
    const w3 = this.textWidth(textSat);
    const maxW = Math.max(w1, w2, w3);

    const padX = 16; // 左右余白
    const padY = 12; // 上下余白
    const needH = padY*2 + this.lineHeight()*3; // 3行固定
    const needW = Math.max(180, padX*2 + maxW);

    const remainW = Graphics.boxWidth  - this.x;
    const remainH = Graphics.boxHeight - this.y;
    const finalW = Math.min(needW, Math.max(120, remainW));
    const finalH = Math.min(needH, Math.max(48,  remainH));

    if (this.width !== finalW || this.height !== finalH){
      this.move(this.x, this.y, finalW, finalH);
      this.createContents();
      this.contents.fontSize = clamp(st().hudFontSize, 8, 48);
    } else {
      this.contents.clear();
    }

    this._drawLines(textFloor, textDiff, textSat);
  };

  RogueHud.prototype._drawLines = function(t1, t2, t3){
    this.changeTextColor(this._hudColor);
    this.drawText(t1, 0, 0, this.contents.width, 'left');
    this.drawText(t2, 0, this.lineHeight(), this.contents.width, 'left');
    this.drawText(t3, 0, this.lineHeight()*2, this.contents.width, 'left');
    this.resetTextColor();
  };

  // -------------------------------------------------
  // Scene_Map hookup
  // -------------------------------------------------
  const _createAllWindows = Scene_Map.prototype.createAllWindows;
  Scene_Map.prototype.createAllWindows = function(){
    _createAllWindows.call(this);
    this._rogueHud = new RogueHud();
    this.addWindow(this._rogueHud);
  };

  // -------------------------------------------------
  // Plugin Commands
  // -------------------------------------------------
  PluginManager.registerCommand(PN,'NextFloor', function(){
    window.RogueLite.nextFloor();
  });

  PluginManager.registerCommand(PN,'AddSatiety', function(args){
    window.RogueLite.addSatiety(Number(args.value||0));
  });

  // 統一：SubSatietyもAPI経由
  PluginManager.registerCommand(PN,'SubSatiety', function(args){
    window.RogueLite.addSatiety(-Number(args.value||0));
  });

  PluginManager.registerCommand(PN,'SetSatiety', function(args){
    window.RogueLite.setSatiety(Number(args.value||0));
  });

  PluginManager.registerCommand(PN,'SetDifficulty', function(args){
    window.RogueLite.setDifficulty(Number(args.value||1));
    if (SceneManager._scene && SceneManager._scene._rogueHud){
      SceneManager._scene._rogueHud.flashHud(); // 設定変更も視覚強調
    }
  });

  PluginManager.registerCommand(PN,'SetFloor', function(args){
    window.RogueLite.setFloor(Number(args.value||1));
  });

  PluginManager.registerCommand(PN,'AddDifficulty', function(args){
    const o = st();
    o.diff = clamp(o.diff + Number(args.value||0), 1, o.maxD);
    o._lastDelta = Number(args.value||0);
    if (SceneManager._scene && SceneManager._scene._rogueHud){
      SceneManager._scene._rogueHud.flashHud();
    }
  });

  PluginManager.registerCommand(PN,'SubDifficulty', function(args){
    const o = st();
    o.diff = clamp(o.diff - Number(args.value||0), 1, o.maxD);
    o._lastDelta = -Number(args.value||0);
    if (SceneManager._scene && SceneManager._scene._rogueHud){
      SceneManager._scene._rogueHud.flashHud();
    }
  });

  // HUDスタイル変更（文字色/背景不透明度）
  PluginManager.registerCommand(PN,'SetHudStyle', function(args){
    const o = st();
    if (args.color && String(args.color).trim() !== ""){
      o.hudColor = sanitizeColor(String(args.color));
    }
    if (args.bgOpacity !== undefined && args.bgOpacity !== ""){
      o.hudBgOpacity = clamp(Number(args.bgOpacity), 0, 255);
    }
    if (SceneManager._scene && SceneManager._scene._rogueHud){
      const w = SceneManager._scene._rogueHud;
      w._hudColor = o.hudColor;
      w.opacity   = o.hudBgOpacity;
      w.refresh(true);
    }
  });

  // 満腹度減少ON/OFF
  PluginManager.registerCommand(PN,'DisableHungerLoss', function(){
    const o = st();
    o.hungerLossEnabled = false;
    if (SceneManager._scene && SceneManager._scene._rogueHud){
      SceneManager._scene._rogueHud.flashHud();
    }
  });

  PluginManager.registerCommand(PN,'EnableHungerLoss', function(){
    const o = st();
    o.hungerLossEnabled = true;
    if (SceneManager._scene && SceneManager._scene._rogueHud){
      SceneManager._scene._rogueHud.flashHud();
    }
  });

})();
