/*:
 * @target MZ
 * @plugindesc v1.1.15 KT_TamaKyoudaReact（A仕様・安定優先）: タマ強打リアクション（トースト/顔/右半分ランダム/折り返し/戦闘後短縮消化/画面外対策/小幅時顔OFF/\\幅補正/ctx安全クリア/addedStates判定/内容フェード）
 * @author ChatGPT
 *
 * @help
 * ■A仕様（安定優先）
 * 「タマ強打付与行動」とみなすのは、スキル/アイテムの効果欄（item.effects）に
 * ある「ステート付与：タマ強打」だけです。
 *  - 通常攻撃の「攻撃時ステート付与（特徴）」や他プラグインによる付与は対象外です。
 *
 * ■表示タイミング（A仕様）
 * 【敵が対象】
 * 1) effectsにタマ強打付与があり、実際に新規付与されたとき（addedStates基準）
 * 2) タマ強打中の敵に、effectsでタマ強打付与を含む行動を取ったとき（付与済みでも表示）
 * 3) タマ強打付与効果があるのに、死亡により付与がブロック/消滅したとき（同一行動内）
 *
 * 【味方（アクター）が対象】
 * A) effectsにタマ強打付与があり、実際に新規付与されたとき（addedStates基準）
 * B) タマ強打中の味方に、effectsでタマ強打付与を含む行動を取ったとき（付与済みでも表示）
 * C) タマ強打付与効果があるのに、死亡により付与がブロック/消滅したとき（同一行動内）
 *
 * ■補足（行動外付与）
 * 行動外（RT.inAction=false）でタマ強打が新規付与された場合は、便宜上トースト表示します。
 *
 * ■ガード
 * - 1行動（BattleManager.startAction）につき、対象ごとに最大1回だけトーストを出します。
 *   （連続回数スキルでも1回に抑制）
 *
 * ■戦闘終了時まとめ表示
 * - FlushToastOnEndBattle ON のとき、戦闘後（結果画面中）もトーストを消化します。
 * - 戦闘後の消化中は表示時間を自動短縮します（固定60f）。
 *
 * ■画面外対策
 * - トースト幅/高さは画面（Graphics.boxWidth/Height）を超えないようにクランプします。
 *
 * ■小幅時の顔自動OFF
 * - 顔＋テキスト領域が確保できない幅の場合、顔表示を自動でOFFにしてテキスト優先にします。
 *
 * ■制御文字について
 * - 本プラグインの「折り返し計算」は簡易です。
 * - \{ \} などによる文字サイズ変化は折り返し計算に反映しません（対応しません）。
 *
 * @param StateId
 * @text タマ強打ステートID
 * @type state
 * @default 2
 *
 * @param DeathStateId
 * @text 戦闘不能ステートID
 * @type state
 * @default 1
 *
 * @param ShowOnMiss
 * @text ミス/回避でも表示
 * @type boolean
 * @default false
 *
 * @param EnemyMessages
 * @text 敵がタマ強打になった時メッセージ(共通)
 * @type string[]
 * @default ["{target}はタマ強打になった！"]
 *
 * @param EnemyActorMessages
 * @text 敵がタマ強打になった時メッセージ(付与者アクター別)
 * @type struct<ActorMessages>[]
 * @default []
 *
 * @param ActorDefaultMessages
 * @text 味方がタマ強打になった時メッセージ(共通)
 * @type string[]
 * @default ["{target}はタマ強打になった！"]
 *
 * @param ActorMessages
 * @text 味方がタマ強打になった時メッセージ(アクター別)
 * @type struct<ActorMessages>[]
 * @default []
 *
 * @param ActorFaceOverrides
 * @text 顔画像上書き(アクター別)
 * @type struct<ActorFace>[]
 * @default []
 *
 * @param ToastWidth
 * @text トースト幅
 * @type number
 * @min 120
 * @default 420
 *
 * @param ToastMinHeight
 * @text トースト最小高さ
 * @type number
 * @min 72
 * @default 96
 *
 * @param ToastMaxHeight
 * @text トースト最大高さ
 * @type number
 * @min 72
 * @default 260
 *
 * @param AutoFitHeight
 * @text 自動高さ（可能な範囲で）
 * @type boolean
 * @default true
 *
 * @param ToastFrames
 * @text 表示フレーム
 * @type number
 * @min 1
 * @default 180
 *
 * @param ToastOpenCloseSpeed
 * @text 開閉速度
 * @type number
 * @min 1
 * @default 16
 *
 * @param RightHalfDefault
 * @text 右半分にランダム配置(未指定時)
 * @type boolean
 * @default true
 *
 * @param ToastXMin
 * @text トーストX最小(任意)
 * @type number
 * @default -1
 *
 * @param ToastXMax
 * @text トーストX最大(任意)
 * @type number
 * @default -1
 *
 * @param ToastYMin
 * @text トーストY最小(任意)
 * @type number
 * @default -1
 *
 * @param ToastYMax
 * @text トーストY最大(任意)
 * @type number
 * @default -1
 *
 * @param FlushToastOnEndBattle
 * @text 戦闘終了時に未表示分をまとめて表示（結果画面中に消化）
 * @type boolean
 * @default true
 *
 * @param PreloadActorFaces
 * @text 顔画像を先読み（パーティメンバーのみ）
 * @type boolean
 * @default true
 *
 * @param DebugLog
 * @text デバッグログ
 * @type boolean
 * @default false
 */
/*~struct~ActorMessages:
 * @param ActorId
 * @text アクターID
 * @type actor
 * @default 0
 *
 * @param Messages
 * @text メッセージ配列
 * @type string[]
 * @default []
 */
/*~struct~ActorFace:
 * @param ActorId
 * @text アクターID
 * @type actor
 * @default 0
 *
 * @param FaceFile
 * @text 顔ファイル
 * @type file
 * @dir img/faces
 * @default
 *
 * @param FaceIndex
 * @text 顔番号
 * @type number
 * @min 0
 * @max 7
 * @default 0
 */

(() => {
  "use strict";

  const POST_BATTLE_TOAST_FRAMES = 60;
  const MIN_TEXT_W = 48;
  const DEFAULT_BACK_OPACITY = 192;

  // --------------------------
  // plugin parameters (robust)
  // --------------------------
  function currentScriptName() {
    try {
      const src = document.currentScript && document.currentScript.src;
      if (!src) return "";
      const m = src.match(/([^/]+)\.js$/i);
      return m ? m[1] : "";
    } catch {
      return "";
    }
  }
  const PLUGIN_KEY = currentScriptName() || "KT_TamaKyoudaReact";

  function loadParams() {
    const tries = [PLUGIN_KEY, "KT_TamaKyoudaReact", "reaction"];
    for (const k of tries) {
      const p = PluginManager.parameters(k);
      if (p && Object.keys(p).length) return p;
    }
    return {};
  }
  const P = loadParams();

  const U = {
    bool(v, def) { if (v === undefined) return def; return String(v) === "true"; },
    num(v, def) { const n = Number(v); return Number.isFinite(n) ? n : def; },
    clamp(n, a, b) { return Math.max(a, Math.min(b, n)); },
    pickRandom(arr) { return arr[Math.floor(Math.random() * arr.length)]; },
    strArray(v) {
      if (!v) return [];
      if (Array.isArray(v)) return v.map(s => String(s));
      try { const a = JSON.parse(v); if (Array.isArray(a)) return a.map(s => String(s)); } catch {}
      return [];
    },
    inBattle() { return !!($gameParty && $gameParty.inBattle && $gameParty.inBattle()); },
    isActor(b) { return b && b.isActor && b.isActor(); },
    isEnemy(b) { return b && b.isEnemy && b.isEnemy(); },
    actorId(b) { try { return b && b.actorId ? b.actorId() : 0; } catch { return 0; } },
    battlerName(b) { try { return b && b.name ? b.name() : ""; } catch { return ""; } },
    battlerKey(b) {
      try {
        if (!b) return "null";
        if (b.isActor && b.isActor()) return `A${b.actorId()}`;
        if (b.isEnemy && b.isEnemy()) return `E${b.enemyId()}`;
      } catch {}
      return String(b);
    },
  };

  const CFG = {
    stateId: U.num(P.StateId, 2),
    deathStateId: U.num(P.DeathStateId, 1),
    showOnMiss: U.bool(P.ShowOnMiss, false),

    enemyMessages: U.strArray(P.EnemyMessages),
    enemyActorMessageMap: {},

    actorDefaultMessages: U.strArray(P.ActorDefaultMessages),
    actorMessageMap: {},

    actorFaceOverrides: {},

    toastWidth: U.num(P.ToastWidth, 420),
    toastMinHeight: U.num(P.ToastMinHeight, 96),
    toastMaxHeight: U.num(P.ToastMaxHeight, 260),
    autoFitHeight: U.bool(P.AutoFitHeight, true),
    toastFrames: U.num(P.ToastFrames, 180),
    toastOpenCloseSpeed: U.num(P.ToastOpenCloseSpeed, 16),

    rightHalfDefault: U.bool(P.RightHalfDefault, true),
    toastXMin: U.num(P.ToastXMin, -1),
    toastXMax: U.num(P.ToastXMax, -1),
    toastYMin: U.num(P.ToastYMin, -1),
    toastYMax: U.num(P.ToastYMax, -1),

    flushToastOnEndBattle: U.bool(P.FlushToastOnEndBattle, true),
    preloadActorFaces: U.bool(P.PreloadActorFaces, true),

    debugLog: U.bool(P.DebugLog, false),
  };

  function dlog(...args) {
    if (!CFG.debugLog) return;
    try { console.log(`[KT_TamaKyoudaReact]`, ...args); } catch {}
  }

  function parseStructList(rawJson) {
    if (!rawJson) return [];
    let arr = [];
    try { arr = JSON.parse(rawJson); } catch { return []; }
    if (!Array.isArray(arr)) return [];
    const out = [];
    for (const s of arr) {
      try { out.push(JSON.parse(s)); } catch {}
    }
    return out;
  }

  function parseActorMessageList(rawJson, outMap) {
    const list = parseStructList(rawJson);
    for (const o of list) {
      const aid = U.num(o.ActorId, 0);
      const msgs = U.strArray(o.Messages);
      if (aid > 0 && msgs.length > 0) outMap[aid] = msgs;
    }
  }

  function parseActorFaceList(rawJson, outMap) {
    const list = parseStructList(rawJson);
    for (const o of list) {
      const aid = U.num(o.ActorId, 0);
      const faceName = String(o.FaceFile || "").trim();
      const faceIndex = U.clamp(U.num(o.FaceIndex, 0), 0, 7);
      if (aid > 0 && faceName) outMap[aid] = { faceName, faceIndex };
    }
  }

  parseActorMessageList(P.EnemyActorMessages, CFG.enemyActorMessageMap);
  parseActorMessageList(P.ActorMessages, CFG.actorMessageMap);
  parseActorFaceList(P.ActorFaceOverrides, CFG.actorFaceOverrides);

  // --------------------------
  // runtime
  // --------------------------
  const RT = {
    _actionSerial: 0,
    _ctxStack: [],
    _guard: new Map(),
    _pending: [],
    _postBattle: false,

    resetForBattle() {
      this._actionSerial = 0;
      this._ctxStack.length = 0;
      this._guard.clear();
      this._pending.length = 0;
      this._postBattle = false;
    },

    beginAction(subjectAid) {
      const actionId = ++this._actionSerial;
      this._ctxStack.push({ actionId, subjectAid: subjectAid || 0 });
      return actionId;
    },
    endAction() {
      if (this._ctxStack.length > 0) this._ctxStack.pop();
    },
    clearActionStack() {
      this._ctxStack.length = 0;
    },
    ctx() {
      return this._ctxStack.length ? this._ctxStack[this._ctxStack.length - 1] : null;
    },
    inAction() {
      return this._ctxStack.length > 0;
    },

    passGuard(battler, group) {
      const bk = U.battlerKey(battler);
      const key = `${group}:${bk}`;
      const last = this._guard.get(key) || { actionId: -1, frame: -1 };
      const ctx = this.ctx();

      if (ctx) {
        if (last.actionId === ctx.actionId) return false;
        this._guard.set(key, { actionId: ctx.actionId, frame: last.frame });
        return true;
      }

      const frame = Graphics.frameCount || 0;
      if (last.frame === frame) return false;
      this._guard.set(key, { actionId: last.actionId, frame });
      return true;
    },
  };

  // --------------------------
  // message + face
  // --------------------------
  function formatText(text, target) {
    const name = U.battlerName(target);
    return String(text || "").replace(/\{target\}/g, name);
  }

  function pickEnemyMessage(target, inflictorActorId) {
    if (inflictorActorId && CFG.enemyActorMessageMap[inflictorActorId]) {
      return formatText(U.pickRandom(CFG.enemyActorMessageMap[inflictorActorId]), target);
    }
    if (!CFG.enemyMessages.length) return "";
    return formatText(U.pickRandom(CFG.enemyMessages), target);
  }

  function pickActorMessage(target) {
    const aid = U.actorId(target);
    const list = CFG.actorMessageMap[aid] || CFG.actorDefaultMessages;
    if (!list || !list.length) return "";
    return formatText(U.pickRandom(list), target);
  }

  function resolveActorFaceById(actorId) {
    if (actorId && CFG.actorFaceOverrides[actorId]) return CFG.actorFaceOverrides[actorId];
    const a = $gameActors && $gameActors.actor ? $gameActors.actor(actorId) : null;
    if (a && a.faceName && a.faceName()) {
      return { faceName: a.faceName(), faceIndex: a.faceIndex() };
    }
    return { faceName: "", faceIndex: 0 };
  }

  // 先読み：パーティメンバーのみ（＋上書き顔があればそれ）
  function preloadActorFaces() {
    if (!CFG.preloadActorFaces) return;
    if (!$gameParty || !$gameParty.members) return;
    try {
      const members = $gameParty.members();
      for (const a of members) {
        if (!a || !a.actorId) continue;
        const aid = a.actorId();
        const f = resolveActorFaceById(aid);
        if (f.faceName) ImageManager.loadFace(f.faceName);
      }
    } catch {}
  }

  // --------------------------
  // wrap (practical)
  // ※制御文字の幅変化は対応しない（簡易）
  // --------------------------
  function tokenizeForWrap(win, text) {
    const tokens = [];
    const s = String(text ?? "");
    let i = 0;
    const iconW = ImageManager.iconWidth || 32;
    const backslashW = win.textWidth("\\");

    while (i < s.length) {
      const ch = s[i];

      if (ch === "\n") {
        tokens.push({ raw: "\n", w: 0, br: true });
        i++;
        continue;
      }

      if (ch === "\\") {
        const rest = s.slice(i);

        // "\\" (escaped backslash) should count as ONE "\" width
        if (rest.startsWith("\\\\")) {
          tokens.push({ raw: "\\\\", w: backslashW, br: false });
          i += 2;
          continue;
        }

        // \I[n] icon: approximate width (optional)
        let m = rest.match(/^\\I\[(\d+)\]/);
        if (m) {
          tokens.push({ raw: m[0], w: iconW, br: false });
          i += m[0].length;
          continue;
        }

        // general control with [n] or [xxx]
        m = rest.match(/^\\[A-Za-z]+\[[^\]]*\]/);
        if (m) {
          tokens.push({ raw: m[0], w: 0, br: false });
          i += m[0].length;
          continue;
        }

        // one-char control like \{ \} \. \! etc (width not supported)
        m = rest.match(/^\\./);
        if (m) {
          tokens.push({ raw: m[0], w: 0, br: false });
          i += m[0].length;
          continue;
        }
      }

      tokens.push({ raw: ch, w: null, br: false });
      i++;
    }

    return tokens;
  }

  function wrapTextToLines(win, text, maxWidth) {
    const tokens = tokenizeForWrap(win, text);
    const lines = [];
    let line = "";
    let w = 0;

    function flush() {
      lines.push(line);
      line = "";
      w = 0;
    }

    for (const t of tokens) {
      if (t.br) {
        flush();
        continue;
      }
      const addW = (t.w != null) ? t.w : win.textWidth(t.raw);
      if (w > 0 && (w + addW) > maxWidth) flush();
      line += t.raw;
      w += addW;
    }
    if (line.length || !lines.length) flush();
    return lines;
  }

  // --------------------------
  // toast window
  // --------------------------
  class Window_TamaToast extends Window_Base {
    constructor(rect) {
      super(rect);
      this.opacity = 0; // frame off
      this.backOpacity = DEFAULT_BACK_OPACITY;
      this.contentsOpacity = 0;

      this._entries = [];
      this._timer = 0;
      this._phase = "idle"; // idle/open/show/close
      this.openness = 0;

      this.hide();
    }

    updateOpen() {}
    updateClose() {}

    enqueue(text, faceName, faceIndex) {
      this._entries.push({ text: String(text ?? ""), faceName: faceName || "", faceIndex: faceIndex || 0 });
    }

    _effectiveFrames() {
      if (!U.inBattle() && RT._postBattle) {
        return Math.max(1, Math.min(CFG.toastFrames, POST_BATTLE_TOAST_FRAMES));
      }
      return CFG.toastFrames;
    }

    update() {
      super.update();

      // 透明度をopennessに同期（滑らか）
      // ※枠は常にOFF（opacity=0）
      this.contentsOpacity = this.openness;
      this.backOpacity = Math.floor(DEFAULT_BACK_OPACITY * (this.openness / 255));

      const inBattle = U.inBattle();

      // 戦闘外 & ポストバトルでもない：途中でパッと消さず、閉じてから消す
      if (!inBattle && !RT._postBattle) {
        // これ以上表示しない
        this._entries.length = 0;

        if (this._phase === "idle") {
          this.openness = 0;
          this.hide();
          return;
        } else if (this._phase !== "close") {
          this._phase = "close";
        }
        // close処理を継続（returnしない）
      }

      if (this._phase === "idle") {
        if (this._entries.length > 0) {
          const e = this._entries.shift();
          this._setupAndShow(e);
        } else if (!inBattle && RT._postBattle) {
          // 消化完了
          RT._postBattle = false;
          this.hide();
        }
        return;
      }

      if (this._phase === "open") {
        this.openness += CFG.toastOpenCloseSpeed;
        if (this.openness >= 255) {
          this.openness = 255;
          this._phase = "show";
          this._timer = this._effectiveFrames();
        }
        return;
      }

      if (this._phase === "show") {
        if (this._timer > 0) this._timer--;
        if (this._timer <= 0) this._phase = "close";
        return;
      }

      if (this._phase === "close") {
        this.openness -= CFG.toastOpenCloseSpeed;
        if (this.openness <= 0) {
          this.openness = 0;
          this._phase = "idle";

          // 戦闘後消化中で、もうキューがなければ閉じる
          if (!this._entries.length && !inBattle && RT._postBattle) {
            RT._postBattle = false;
            this.hide();
          }
          // 戦闘外でpostBattleでもないなら、次フレームで上の分岐がhideする
        }
      }
    }

    _setupAndShow(entry) {
      this.show();
      this.openness = 0;
      this._phase = "open";

      const ww = Graphics.boxWidth;
      const wh = Graphics.boxHeight;

      const newW = Math.max(1, Math.min(CFG.toastWidth, ww));

      const pad = this.padding;
      const innerW = Math.max(1, newW - pad * 2);

      // face on/off
      let hasFace = !!entry.faceName;
      let faceW = hasFace ? ImageManager.faceWidth : 0;

      let availTextW = innerW - (hasFace ? (faceW + this.itemPadding()) : 0);
      if (hasFace && availTextW < MIN_TEXT_W) {
        hasFace = false;
        faceW = 0;
        availTextW = innerW;
      }

      const textW = Math.max(1, Math.min(availTextW, innerW));

      const rawText = entry.text;
      const lines = CFG.autoFitHeight ? wrapTextToLines(this, rawText, textW) : String(rawText).split("\n");

      const lineH = this.lineHeight();
      const textH = Math.max(1, lines.length) * lineH;
      const faceH = hasFace ? ImageManager.faceHeight : 0;
      const innerHNeed = Math.max(faceH, textH);

      let newH = innerHNeed + pad * 2;
      newH = U.clamp(newH, CFG.toastMinHeight, CFG.toastMaxHeight);
      newH = Math.max(1, Math.min(newH, wh));

      // position ranges
      let xMin = CFG.toastXMin;
      let xMax = CFG.toastXMax;
      let yMin = CFG.toastYMin;
      let yMax = CFG.toastYMax;

      if (xMin < 0 || xMax < 0 || xMax <= xMin) {
        if (CFG.rightHalfDefault) {
          xMin = Math.floor(ww * 0.5);
          xMax = ww - newW;
        } else {
          xMin = 0;
          xMax = ww - newW;
        }
      } else {
        xMax = Math.min(xMax, ww - newW);
      }

      if (yMin < 0 || yMax < 0 || yMax <= yMin) {
        yMin = 0;
        yMax = wh - newH;
      } else {
        yMax = Math.min(yMax, wh - newH);
      }

      xMin = U.clamp(xMin, 0, Math.max(0, ww - newW));
      xMax = U.clamp(xMax, 0, Math.max(0, ww - newW));
      if (xMax < xMin) xMax = xMin;

      yMin = U.clamp(yMin, 0, Math.max(0, wh - newH));
      yMax = U.clamp(yMax, 0, Math.max(0, wh - newH));
      if (yMax < yMin) yMax = yMin;

      const nx = (xMax === xMin) ? xMin : Math.floor(xMin + Math.random() * Math.max(1, (xMax - xMin)));
      const ny = (yMax === yMin) ? yMin : Math.floor(yMin + Math.random() * Math.max(1, (yMax - yMin)));

      this.move(nx, ny, newW, newH);
      this._refresh(entry, lines, textW, hasFace);
    }

    _refresh(entry, lines, textW, hasFace) {
      this.contents.clear();

      let x = 0;
      let y = 0;

      if (hasFace) {
        this.drawFace(entry.faceName, entry.faceIndex, x, y);
        x += ImageManager.faceWidth + this.itemPadding();
      }

      const lineH = this.lineHeight();
      for (let i = 0; i < lines.length; i++) {
        this.drawTextEx(lines[i], x, y + i * lineH, textW);
      }
    }
  }

  // --------------------------
  // scene integration
  // --------------------------
  function getBattleScene() {
    const scene = SceneManager._scene;
    return (scene && scene instanceof Scene_Battle) ? scene : null;
  }

  function ensureToastWindow() {
    const scene = getBattleScene();
    return scene ? scene._tamaToastWindow : null;
  }

  function flushPendingToWindow() {
    const w = ensureToastWindow();
    if (!w) return;
    while (RT._pending.length > 0) {
      const e = RT._pending.shift();
      w.enqueue(e.text, e.faceName, e.faceIndex);
    }
  }

  const _Scene_Battle_createAllWindows = Scene_Battle.prototype.createAllWindows;
  Scene_Battle.prototype.createAllWindows = function() {
    _Scene_Battle_createAllWindows.apply(this, arguments);
    const rect = new Rectangle(0, 0, CFG.toastWidth, CFG.toastMinHeight);
    this._tamaToastWindow = new Window_TamaToast(rect);
    this.addWindow(this._tamaToastWindow);
    flushPendingToWindow();
  };

  const _Scene_Battle_start = Scene_Battle.prototype.start;
  Scene_Battle.prototype.start = function() {
    _Scene_Battle_start.apply(this, arguments);
    preloadActorFaces();
  };

  const _Scene_Battle_terminate = Scene_Battle.prototype.terminate;
  Scene_Battle.prototype.terminate = function() {
    _Scene_Battle_terminate.apply(this, arguments);
    RT._postBattle = false;
    RT._pending.length = 0;
    RT.clearActionStack();
  };

  // --------------------------
  // enqueue + show
  // --------------------------
  function enqueueToast(text, faceName, faceIndex) {
    const w = ensureToastWindow();
    if (w) w.enqueue(text, faceName, faceIndex);
    else RT._pending.push({ text, faceName, faceIndex });
  }

  function showToastFor(target, guardGroup, inflictorActorId = 0) {
    if (!target) return;
    if (!U.inBattle() && !RT._postBattle) return;

    if (!RT.passGuard(target, guardGroup)) {
      dlog("Guarded", { guardGroup, target: U.battlerName(target), ctx: RT.ctx() });
      return;
    }

    let msg = "";
    let face = { faceName: "", faceIndex: 0 };

    if (U.isEnemy(target)) {
      msg = pickEnemyMessage(target, inflictorActorId);
      if (inflictorActorId) face = resolveActorFaceById(inflictorActorId);
    } else if (U.isActor(target)) {
      msg = pickActorMessage(target);
      face = resolveActorFaceById(U.actorId(target));
    } else {
      return;
    }

    if (!msg) {
      dlog("NoMsg", { guardGroup, target: U.battlerName(target) });
      return;
    }

    enqueueToast(msg, face.faceName, face.faceIndex);
    dlog("Shown", { guardGroup, target: U.battlerName(target), inflictorActorId, msg });
  }

  // --------------------------
  // battle lifecycle
  // --------------------------
  const _BattleManager_setup = BattleManager.setup;
  BattleManager.setup = function(troopId, canEscape, canLose) {
    _BattleManager_setup.apply(this, arguments);
    RT.resetForBattle();
  };

  function subjectActorIdFromBattleManager() {
    try {
      const s = BattleManager._subject;
      if (s && s.isActor && s.isActor()) return s.actorId();
    } catch {}
    return 0;
  }

  const _BattleManager_startAction = BattleManager.startAction;
  BattleManager.startAction = function() {
    const subjectAid = subjectActorIdFromBattleManager();
    RT.beginAction(subjectAid);
    _BattleManager_startAction.apply(this, arguments);
  };

  const _BattleManager_endAction = BattleManager.endAction;
  BattleManager.endAction = function() {
    _BattleManager_endAction.apply(this, arguments);
    RT.endAction();
  };

  const _BattleManager_endBattle = BattleManager.endBattle;
  BattleManager.endBattle = function(result) {
    try {
      if (CFG.flushToastOnEndBattle) {
        RT._postBattle = true;
        flushPendingToWindow();
      }
    } catch {}

    const r = _BattleManager_endBattle.apply(this, arguments);

    // safety clear (prevents stuck inAction / bad stack)
    RT.clearActionStack();

    return r;
  };

  // --------------------------
  // addState: action-outside
  // --------------------------
  const _Game_Battler_addState = Game_Battler.prototype.addState;
  Game_Battler.prototype.addState = function(stateId) {
    const had = this.isStateAffected(stateId);
    _Game_Battler_addState.apply(this, arguments);
    if (!U.inBattle()) return;

    if (had) return;
    if (!this.isStateAffected(stateId)) return;

    // During action, show only by apply/effects
    if (RT.inAction() && stateId === CFG.stateId) return;

    if (stateId === CFG.stateId) {
      showToastFor(this, "TAMA_OUTSIDE_ACTION", 0);
    }
  };

  // --------------------------
  // apply hook (A仕様 + addedStates判定)
  // --------------------------
  function actionHasAddStateEffect(action, stateId) {
    try {
      const item = action && action.item ? action.item() : null;
      const effects = item && item.effects ? item.effects : null;
      if (!effects) return false;
      for (let i = 0; i < effects.length; i++) {
        const ef = effects[i];
        if (ef && ef.code === Game_Action.EFFECT_ADD_STATE && ef.dataId === stateId) return true;
      }
    } catch {}
    return false;
  }

  function currentSubjectAidFromCtx() {
    const ctx = RT.ctx();
    return ctx ? (ctx.subjectAid || 0) : 0;
  }

  function isHitResult(result) {
    if (!result) return false;
    if (typeof result.isHit === "function") return result.isHit();
    const used = !!result.used;
    const missed = !!result.missed;
    const evaded = !!result.evaded;
    return used && !missed && !evaded;
  }

  function resultAddedState(result, stateId) {
    if (!result) return false;
    if (typeof result.isStateAdded === "function") return result.isStateAdded(stateId);
    const arr = result.addedStates;
    return Array.isArray(arr) && arr.includes(stateId);
  }

  const _Game_Action_apply = Game_Action.prototype.apply;
  Game_Action.prototype.apply = function(target) {
    const enabled = (U.inBattle() && target);

    let hadTama = false;
    let hadDeath = false;
    let hasTamaEffect = false;

    if (enabled) {
      hadTama = !!target.isStateAffected(CFG.stateId);
      hadDeath = !!target.isStateAffected(CFG.deathStateId);
      hasTamaEffect = actionHasAddStateEffect(this, CFG.stateId);
    }

    _Game_Action_apply.apply(this, arguments);

    if (!enabled) return;
    if (!hasTamaEffect) return;

    const result = (target && target.result) ? target.result() : null;
    const hit = isHitResult(result);
    if (!CFG.showOnMiss && !hit) return;

    const added = resultAddedState(result, CFG.stateId);
    const nowDeath = !!target.isStateAffected(CFG.deathStateId);

    const isEnemy = U.isEnemy(target);
    const isActor = U.isActor(target);
    if (!isEnemy && !isActor) return;

    const subjectAid = currentSubjectAidFromCtx();
    const G = "TAMA_IN_ACTION";

    // (2)(B) already affected => show "付与行動を取った"
    if (hadTama) {
      if (isEnemy) showToastFor(target, G, subjectAid);
      else showToastFor(target, G, 0);
      return;
    }

    // (1)(A) newly applied (addedStates基準：死亡で外れても拾える)
    if (added) {
      if (isEnemy) showToastFor(target, G, subjectAid);
      else showToastFor(target, G, 0);
      return;
    }

    // (3)(C) blocked/vanished by death (same action)
    if (!hadDeath && nowDeath && !added) {
      if (isEnemy) showToastFor(target, G, subjectAid);
      else showToastFor(target, G, 0);
      return;
    }
  };
})();
