/*:
 * @target MZ
 * @plugindesc ショップ購入価格に倍率を適用（スイッチON時）。ショップ個別価格を無視してDB価格参照も可能。 v1.0.0
 * @author ChatGPT
 *
 * @help
 * 【概要】
 * 指定スイッチがONのとき、ショップの購入価格に倍率を掛けます。
 * さらにオプションで「ショップで指定した価格」があっても、DB価格（アイテムの基本価格）を基準にできます。
 *
 * 【例】
 * - DBで薬草=100、倍率=1.5 → 表示/購入価格は150
 * - ショップで薬草を300に指定していても、DB基準ONなら 100*1.5=150
 *
 * 【注意】
 * - 端数は設定した方法で丸めます。
 * - スイッチをショップ表示中に切り替えた場合、再度ウィンドウが更新されるまで表示は変わりません。
 *
 * @param enabledSwitchId
 * @type switch
 * @text 有効化スイッチs
 * @desc このスイッチがONのとき倍率を適用します。0の場合は常に適用します。
 * @default 0
 *
 * @param rate
 * @type number
 * @decimals 4
 * @min 0
 * @text 倍率
 * @desc 例: 1.5 で150%
 * @default 1
 *
 * @param forceDatabasePrice
 * @type boolean
 * @text DB価格を強制参照
 * @desc ON: スイッチON時、ショップ個別価格が指定されていてもDB価格(item.price)を基準にします
 * @default true
 *
 * @param rounding
 * @type select
 * @option 切り捨て
 * @value floor
 * @option 四捨五入
 * @value round
 * @option 切り上げ
 * @value ceil
 * @text 端数処理
 * @default floor
 *
 * @param applyToSell
 * @type boolean
 * @text 売却にも適用
 * @desc ON: スイッチON時、売却価格（通常は半額）にも倍率を適用します
 * @default false
 */

(() => {
  "use strict";

  const PLUGIN_NAME = "shoppricerateMZ";
  const params = PluginManager.parameters(PLUGIN_NAME);

  const enabledSwitchId = Number(params.enabledSwitchId || 0);
  const rate = Number(params.rate || 1);
  const forceDatabasePrice = String(params.forceDatabasePrice || "false") === "true";
  const rounding = String(params.rounding || "floor");
  const applyToSell = String(params.applyToSell || "false") === "true";

  function isSwitchOnSafe() {
    // タイトル画面等で $gameSwitches が未初期化の可能性に備える
    if (enabledSwitchId === 0) return true;
    if (typeof $gameSwitches === "undefined" || !$gameSwitches) return false;
    return !!$gameSwitches.value(enabledSwitchId);
  }

  function roundPrice(value) {
    const v = Number.isFinite(value) ? value : 0;
    if (rounding === "ceil") return Math.ceil(v);
    if (rounding === "round") return Math.round(v);
    return Math.floor(v);
  }

  function applyRateToPrice(basePrice) {
    const v = Number.isFinite(basePrice) ? basePrice : 0;
    const multiplied = v * rate;
    const rounded = roundPrice(multiplied);
    return Math.max(0, rounded);
  }

  // ---- 購入価格（表示/購入処理） ----
  // Window_ShopBuy は makeItemList で _price[] を作るので、そこを書き換えるのが確実。
  const _Window_ShopBuy_makeItemList = Window_ShopBuy.prototype.makeItemList;
  Window_ShopBuy.prototype.makeItemList = function() {
    _Window_ShopBuy_makeItemList.call(this);

    if (!isSwitchOnSafe()) return;

    for (let i = 0; i < this._data.length; i++) {
      const item = this._data[i];
      if (!item) continue;

      // 通常のショップ価格（個別指定も反映済み）
      let base = this._price[i];

      // オプション: DB価格を強制参照（スイッチON時のみ）
      if (forceDatabasePrice) {
        base = item.price; // アイテム/武器/防具 共通で price を持つ
      }

      this._price[i] = applyRateToPrice(base);
    }
  };

  // ---- 売却価格（任意） ----
  if (applyToSell) {
    const _Scene_Shop_sellingPrice = Scene_Shop.prototype.sellingPrice;
    Scene_Shop.prototype.sellingPrice = function() {
      const price = _Scene_Shop_sellingPrice.call(this);
      if (!isSwitchOnSafe()) return price;
      return applyRateToPrice(price);
    };
  }
})();
