//=============================================================================
// SkillFormulaOverrideMZ.js
//=============================================================================
/*:
 * @target MZ
 * @plugindesc v1.0 指定したスキル群のダメージ計算式を一括で上書きします。
 * @author あなたの名前
 *
 * @help
 * 指定したスキルの「ダメージ計算式（damage.formula）」を、
 * データベース上の値とは別にプラグインパラメータで一括指定し、
 * その値でデータベース自体を上書きします。
 *
 * ■仕様
 * ・「スキル計算式グループ一覧」に登録した各グループについて、
 *   グループ内のすべてのスキルの「ダメージ計算式」(database の
 *   skills[n].damage.formula) を指定した文字列に変更します。
 * ・登録していないスキルは、データベースの値がそのまま使われます。
 * ・通常攻撃やアイテムには影響しません（スキルのみ対象）。
 * ・同じスキルIDが複数グループに含まれている場合は、
 *   「後ろに書かれているグループ」の内容が最終的に採用されます。
 *
 * ■注意
 * ・このプラグインは「ゲーム起動時に一度だけ」$dataSkills[...] の
 *   damage.formula を上書きします。他プラグインから見ても、最初から
 *   その値だったかのように扱われます。
 *
 * ■使い方
 * 1. プラグインを導入し、「スキル計算式グループ一覧」に
 *    ・対象スキル群（Skills）
 *    ・適用したい計算式（FormulaText）
 *    を設定してください。
 * 2. 計算式は、データベースの「ダメージ計算式」と同じ書式で記述します。
 *    例）200 + a.mat * 13 - b.mdf * 2
 *
 * このプラグインにはプラグインコマンドはありません。
 *
 * @param SkillFormulaGroupList
 * @text スキル計算式グループ一覧
 * @type struct<SkillFormulaGroup>[]
 * @default ["{\"Skills\":\"[\\\"99\\\",\\\"107\\\",\\\"115\\\",\\\"141\\\",\\\"147\\\"]\",\"FormulaText\":\"200 + a.mat * 13 - b.mdf * 2\"}","{\"Skills\":\"[\\\"100\\\",\\\"108\\\",\\\"116\\\"]\",\"FormulaText\":\"400 + a.mat * 26 - b.mdf * 2\"}","{\"Skills\":\"[\\\"101\\\",\\\"109\\\",\\\"117\\\",\\\"142\\\",\\\"148\\\"]\",\"FormulaText\":\"650 + a.mat * 40 - b.mdf * 2\"}","{\"Skills\":\"[\\\"103\\\",\\\"111\\\",\\\"119\\\",\\\"144\\\",\\\"150\\\",\\\"153\\\"]\",\"FormulaText\":\"100 + a.mat * 8 - b.mdf * 2\"}","{\"Skills\":\"[\\\"104\\\",\\\"112\\\",\\\"116\\\",\\\"120\\\",\\\"145\\\",\\\"151\\\",\\\"154\\\"]\",\"FormulaText\":\"300 + a.mat * 18 - b.mdf * 2\"}","{\"Skills\":\"[\\\"105\\\",\\\"113\\\",\\\"121\\\",\\\"146\\\",\\\"152\\\",\\\"156\\\"]\",\"FormulaText\":\"450 + a.mat * 32 - b.mdf * 2\"}"]
 * @desc 複数スキルに同一のダメージ計算式を与えるグループの一覧。
 */

/*~struct~SkillFormulaGroup:
 * @param Skills
 * @text 対象スキル
 * @type skill[]
 * @desc このグループに含めるスキル。すべて同じ計算式になります。
 *
 * @param FormulaText
 * @text 計算式（テキスト）
 * @desc このグループのスキルに適用するダメージ計算式。
 * 例: 200 + a.mat * 13 - b.mdf * 2
 * @default a.atk * 4 - b.def * 2
 */

(() => {
    'use strict';

    // ★ファイル名と一致させる
    const pluginName = 'SkillFormulaOverrideMZ';

    const parameters = PluginManager.parameters(pluginName);
    const groupListRaw = JSON.parse(parameters['SkillFormulaGroupList'] || '[]');

    /** @type {Record<number, string>} */
    const skillFormulaMap = {};

    // パラメータの解析
    for (const groupEntry of groupListRaw) {
        if (!groupEntry) continue;

        /** @type {{Skills?: string, FormulaText?: string}} */
        const groupObj = JSON.parse(groupEntry);

        const formulaText = (groupObj.FormulaText != null ? groupObj.FormulaText : '').trim();
        if (!formulaText) {
            console.warn('[SkillFormulaOverrideMZ] 空の計算式が指定されています:', groupObj);
            continue;
        }

        const skillsRaw = groupObj.Skills ? JSON.parse(groupObj.Skills) : [];
        if (!Array.isArray(skillsRaw)) continue;

        for (const idStr of skillsRaw) {
            const skillId = Number(idStr || 0);
            if (skillId > 0) {
                // あとから登録されたものが優先（後勝ち）
                skillFormulaMap[skillId] = formulaText;
            }
        }
    }

    console.log('[SkillFormulaOverrideMZ] skillFormulaMap:', skillFormulaMap);

    // データベース読み込み完了後に formula を上書きする
    const _DataManager_isDatabaseLoaded = DataManager.isDatabaseLoaded;
    DataManager.isDatabaseLoaded = function() {
        const loaded = _DataManager_isDatabaseLoaded.call(this);
        if (loaded && !this._skillFormulaOverrideApplied) {
            this._skillFormulaOverrideApplied = true;
            applySkillFormulaOverrides();
        }
        return loaded;
    };

    function applySkillFormulaOverrides() {
        if (!$dataSkills) return;

        for (const idStr in skillFormulaMap) {
            const id = Number(idStr);
            const skill = $dataSkills[id];
            if (skill && skill.damage) {
                skill.damage.formula = skillFormulaMap[id];
                // デバッグ用：
                // console.log(`[SkillFormulaOverrideMZ] skill ${id} (${skill.name}) formula -> ${skill.damage.formula}`);
            }
        }
    }
})();
