/*:
 * @target MZ
 * @plugindesc v1.7.2 スキル威力＆効果詳細ウィンドウ（3行固定・色分け・ヘルプ併用）
 *
 * @param powerMultiplier
 * @text 威力倍率
 * @type number
 * @decimals 2
 * @min 0
 * @default 15
 * @desc ダメージ式の計算結果（四捨五入後の絶対値）に掛ける倍率です。旧仕様の 15 なら従来通りです。
 *
 * @param dummyAtk
 * @text 計算用 攻撃力（a.atk / a.param(2)）
 * @type number
 * @min 0
 * @decimals 0
 * @default 1
 * @desc 威力計算で使用する「使用者aの攻撃力」です。ダメージ式で a.atk や a.param(2) を参照する場合に反映されます。
 * @param dummyADef
 * @text 計算用 防御力（a.def / a.param(3)）
 * @type number
 * @min 0
 * @decimals 0
 * @default 1
 * @desc 威力計算で使用する「使用者aの防御力」です。ダメージ式で a.def や a.param(3) を参照する場合に反映されます。
 *
 * @param dummyMat
 * @text 計算用 魔法攻撃（a.mat / a.param(4)）
 * @type number
 * @min 0
 * @default 1
 * @desc 威力計算で使用する「使用者aの魔法攻撃」です。ダメージ式で a.mat や a.param(4) を参照する場合に反映されます。
 * @param dummyAMdf
 * @text 計算用 魔法防御（a.mdf / a.param(5)）
 * @type number
 * @min 0
 * @decimals 0
 * @default 1
 * @desc 威力計算で使用する「使用者aの魔法防御」です。ダメージ式で a.mdf や a.param(5) を参照する場合に反映されます。
 * @param dummyBAtk
 * @text 計算用 攻撃力（b.atk / b.param(2)）
 * @type number
 * @min 0
 * @decimals 0
 * @default 1
 * @desc 威力計算で使用する「対象bの攻撃力」です。ダメージ式で b.atk や b.param(2) を参照する場合に反映されます。
 *
 * @param dummyDef
 * @text 計算用 防御力（b.def / b.param(3)）
 * @type number
 * @min 0
 * @decimals 0
 * @default 1
 * @desc 威力計算で使用する「対象bの防御力」です。ダメージ式で b.def や b.param(3) を参照する場合に反映されます。
 * @param dummyBMat
 * @text 計算用 魔法攻撃（b.mat / b.param(4)）
 * @type number
 * @min 0
 * @decimals 0
 * @default 1
 * @desc 威力計算で使用する「対象bの魔法攻撃」です。ダメージ式で b.mat や b.param(4) を参照する場合に反映されます。
 *
 * @param dummyMdf
 * @text 計算用 魔法防御（b.mdf / b.param(5)）
 * @type number
 * @min 0
 * @default 1
 * @desc 威力計算で使用する「対象bの魔法防御」です。ダメージ式で b.mdf や b.param(5) を参照する場合に反映されます。
 * @author ChatGPT
 *
 * @help
 * 【概要】
 * ・メニュー画面／バトル画面でスキルにカーソルを合わせたときのみ、
 *   専用フロートウィンドウを表示して
 *   「威力」「属性」「状態付与（確率付き）」「状態回復」
 *   「能力変化」「ダメージ種別／回復効果」を表示します。
 * ・ダメージ式の威力プレビュー計算では、addState / removeState / addBuff などの「副作用系呼び出し」は無視（no-op）します。
 * ・説明用ヘルプウィンドウはそのまま残ります。
 *
 * 【レイアウト】
 * ・メニュー画面:
 *    ヘルプウィンドウの直上に 3 行分のウィンドウを表示
 * ・バトル画面:
 *    ステータスウィンドウの直上に 3 行分のウィンドウを表示
 *    ＋フレーム（枠線）は非表示
 *
 * 【表示タイミング】
 * ・メニュー画面: スキルリストにカーソルがある間だけ表示
 * ・バトル画面: スキルコマンド選択中（スキルウィンドウが開いている間）だけ表示
 *   → スキル決定／キャンセルで自動的に非表示になります。
 *
 * 【色分け】
 * ・ラベル（威力 / 属性 / 状態付与 / 状態回復 / 能力変化 / 効果）は
 *   システムカラーで表示。
 * ・属性名、状態異常名、状態回復名、能力名（攻撃力・防御力など）を
 *   それぞれ専用の色で表示します。
 *
 *   ※色番号は下の定数で編集可能です。
 *     RPGツクールMZの「文章の表示」で \C[n] を試して、
 *     好きな色番号に差し替えてください。
 */

(() => {
  'use strict';

  const PLUGIN_NAME = 'SkillPowerDetailWindowMZ';


  const _params = PluginManager.parameters(PLUGIN_NAME);

  const POWER_MULTIPLIER = (() => {
    const n = Number(_params.powerMultiplier ?? 15);
    return Number.isFinite(n) ? n : 15;
  })();

  const DUMMY_A_ATK = (() => {
    const n = Number(_params.dummyAtk ?? 1);
    return Number.isFinite(n) ? n : 1;
  })();

  const DUMMY_A_DEF = (() => {
    const n = Number(_params.dummyADef ?? 1);
    return Number.isFinite(n) ? n : 1;
  })();

  const DUMMY_A_MAT = (() => {
    const n = Number(_params.dummyMat ?? 1);
    return Number.isFinite(n) ? n : 1;
  })();

  const DUMMY_A_MDF = (() => {
    const n = Number(_params.dummyAMdf ?? 1);
    return Number.isFinite(n) ? n : 1;
  })();

  const DUMMY_B_ATK = (() => {
    const n = Number(_params.dummyBAtk ?? 1);
    return Number.isFinite(n) ? n : 1;
  })();

  const DUMMY_B_DEF = (() => {
    const n = Number(_params.dummyDef ?? 1);
    return Number.isFinite(n) ? n : 1;
  })();

  const DUMMY_B_MAT = (() => {
    const n = Number(_params.dummyBMat ?? 1);
    return Number.isFinite(n) ? n : 1;
  })();

  const DUMMY_B_MDF = (() => {
    const n = Number(_params.dummyMdf ?? 1);
    return Number.isFinite(n) ? n : 1;
  })();
// ----------------------------------------
  // 色番号設定（必要に応じて書き換えてください）
  // ----------------------------------------
  const COLOR_LABEL   = 16; // ラベル用（威力/属性/状態付与など）
  const COLOR_ELEMENT = 0;  // 属性名用
  const COLOR_STATE   = 7;  // 状態名用
  const COLOR_PARAM   = 2;  // 攻撃力・防御力など能力名用
// ----------------------------------------
// 属性・状態名の個別カラー設定
// ----------------------------------------
const COLOR_TAGS = {
  // 属性
  "炎熱": "\\C[2]",
  "氷水": "\\C[1]",
  "雷電": "\\C[17]",
  "光":   "\\C[6]",
  "闇":   "\\C[5]",
  "金的": "\\C[14]",
  "物理": "\\C[3]",

  // 状態異常
  "あちあち状態": "\\C[2]",
  "さむさむ状態": "\\C[1]",
  "ビリビリ状態": "\\C[17]",
  "ピカピカ状態": "\\C[6]",
  "やみやみ状態": "\\C[5]",
  "タマ強打状態": "\\C[14]",
  "ヨロヨロ状態": "\\C[3]",
};

// 対象の文字列を COLOR_TAGS の色で装飾する
function colorizeName(name) {
  if (!name) return name;
  const tag = COLOR_TAGS[name];
  if (tag) {
    return `${tag}${name}\\C[0]`;
  }
  return name;
}

  // ----------------------------------------
  // ダミーバトラー生成（基本は 1 / 攻撃力・防御力は上書き可）
  // ----------------------------------------
  function createDummyBattler(overrides = {}) {
    const dummy = {};
    dummy.level = 1;

    const _ovAtk = Number(overrides.atk);
    const _ovDef = Number(overrides.def);
    const _ovMat = Number(overrides.mat);
    const _ovMdf = Number(overrides.mdf);
    const _atkVal = Number.isFinite(_ovAtk) ? _ovAtk : 1;
    const _defVal = Number.isFinite(_ovDef) ? _ovDef : 1;
    const _matVal = Number.isFinite(_ovMat) ? _ovMat : 1;
    const _mdfVal = Number.isFinite(_ovMdf) ? _ovMdf : 1;


    dummy.hp = 1;
    dummy.mp = 1;
    dummy.tp = 1;

    dummy.mhp = 1;
    dummy.mmp = 1;
    dummy.atk = _atkVal;
    dummy.def = _defVal;
    dummy.mat = _matVal;
    dummy.mdf = _mdfVal;
    dummy.agi = 1;
    dummy.luk = 1;

    dummy._params = [dummy.mhp, dummy.mmp, dummy.atk, dummy.def, dummy.mat, dummy.mdf, dummy.agi, dummy.luk];

    dummy.param = function (paramId) {
      const id = Number(paramId);
      if (Number.isFinite(id) && id >= 0 && id < this._params.length) {
        return this._params[id];
      }
      return 1;
    };
    dummy.xparam = function () { return 1; };
    dummy.sparam = function () { return 1; };

    dummy.isActor = function () { return true; };
    dummy.isEnemy = function () { return false; };
    dummy.currentHp = function () { return 1; };
    dummy.currentMp = function () { return 1; };
    dummy.currentTp = function () { return 1; };

    // ----------------------------------------
    // ダメージ式プレビュー用の no-op（副作用無効化）
    // ※ダメージ式に a.addState(...) などが書かれていても、
    //   ここでは状態変化を起こさず、エラーも出さないためのスタブです。
    // ----------------------------------------
    dummy.addState = function () { return 0; };
    dummy.removeState = function () { return 0; };
    dummy.eraseState = function () { return 0; };
    dummy.clearStates = function () { return 0; };
    dummy.isStateAffected = function () { return false; };
    dummy.stateRate = function () { return 1; };

    dummy.addBuff = function () { return 0; };
    dummy.addDebuff = function () { return 0; };
    dummy.removeBuff = function () { return 0; };
    dummy.removeDebuff = function () { return 0; };
    dummy.isBuffAffected = function () { return false; };
    dummy.isDebuffAffected = function () { return false; };

    dummy.gainHp = function () { return 0; };
    dummy.gainMp = function () { return 0; };
    dummy.gainTp = function () { return 0; };
    dummy.setHp = function () { return 0; };
    dummy.setMp = function () { return 0; };
    dummy.setTp = function () { return 0; };
    dummy.hpRate = function () { return 1; };
    dummy.mpRate = function () { return 1; };
    dummy.tpRate = function () { return 0; };

    return dummy;
  }

  // ----------------------------------------
  // スキル威力の計算
  // ----------------------------------------
  function calcSkillPower(skill) {
    if (!skill || !skill.damage) return null;

    const damage = skill.damage;
    const type = damage.type || 0;
    const formula = (damage.formula || '').trim();

    // ダメージタイプなし or 計算式なし → 威力表示しない
    if (type === 0) return null;
    if (!formula) return null;

    const a = createDummyBattler({ atk: DUMMY_A_ATK, def: DUMMY_A_DEF, mat: DUMMY_A_MAT, mdf: DUMMY_A_MDF });
    const b = createDummyBattler({ atk: DUMMY_B_ATK, def: DUMMY_B_DEF, mat: DUMMY_B_MAT, mdf: DUMMY_B_MDF });
    const v = $gameVariables ? $gameVariables._data : [];

    // ダメージ式に「関数として」書かれた副作用系（例: addState(10)）も
    // ここでは何もしないようにしてエラーを避けます。
    const addState = function () { return 0; };
    const removeState = function () { return 0; };
    const addBuff = function () { return 0; };
    const addDebuff = function () { return 0; };
    const gainHp = function () { return 0; };
    const gainMp = function () { return 0; };
    const gainTp = function () { return 0; };

    let power = null;
    try {
      const result = eval(formula);
      if (typeof result === 'number' && !Number.isNaN(result)) {
        power = Math.abs(Math.round(result));
      }
    } catch (e) {
      console.error(PLUGIN_NAME, 'eval error in formula:', formula, e);
    }
    if (power === null) return null;
    return Math.round(power * POWER_MULTIPLIER);
  }

  // ----------------------------------------
  // ラインごとのテキスト作成
  // ----------------------------------------

  function makeLine1(skill, power) {
    if (!skill) return '';

    const damage = skill.damage || {};
    let elementText = '';
    if (damage.elementId > 0) {
      const name = $dataSystem.elements[damage.elementId];
      if (name) {
        // 属性名を専用色で
        elementText = colorizeName(name);
      }
    }

    let line = '';

    if (power != null) {
      line += `\\C[${COLOR_LABEL}]威力\\C[0]: ${power}`;
    }

    if (elementText) {
      if (line) line += '　'; // 全角スペース
      line += `\\C[${COLOR_LABEL}]属性\\C[0]: ${elementText}`;
    }

    return line;
  }

  function makeLine2(skill) {
    if (!skill) return '';
    const effects = skill.effects || [];

    // 状態付与
    const addEffects = effects.filter(e =>
      e.code === Game_Action.EFFECT_ADD_STATE && e.dataId > 0
    );
    let addText = '';
    if (addEffects.length > 0) {
      const arr = addEffects
        .map(e => {
          const st = $dataStates[e.dataId];
          if (!st || !st.name) return null;
          let prob = e.value1 != null ? Math.round(e.value1 * 100) : 100;
          if (prob < 0) prob = 0;
          if (prob > 100) prob = 100;
          const coloredName = colorizeName(st.name);
          return `${coloredName}（${prob}%）`;
        })
        .filter(t => t);
      if (arr.length > 0) addText = arr.join('、');
    }

    // 状態回復（複数ならまとめ）
    const removeEffects = effects.filter(e =>
      e.code === Game_Action.EFFECT_REMOVE_STATE && e.dataId >= 0
    );
    let removeText = '';
    if (removeEffects.length === 1) {
      const e = removeEffects[0];
      if (e.dataId > 0) {
        const st = $dataStates[e.dataId];
        const name = (st && st.name) ? st.name : '状態';
        removeText = colorizeName(name);
      } else {
        removeText = '\\C[' + COLOR_STATE + ']複数の状態\\C[0]';
      }
    } else if (removeEffects.length > 1) {
      removeText = '\\C[' + COLOR_STATE + ']複数の状態\\C[0]';
    }

    let line = '';

    if (addText) {
      line += `\\C[${COLOR_LABEL}]状態付与\\C[0]: ${addText}`;
    }

    if (removeText) {
      if (line) line += '　';
      line += `\\C[${COLOR_LABEL}]状態回復\\C[0]: ${removeText}`;
    }

    return line;
  }

  function makeLine3(skill) {
    if (!skill) return '';
    const effects = skill.effects || [];

    // 能力変化
    const buffEffects = effects.filter(e =>
      e.code === Game_Action.EFFECT_ADD_BUFF ||
      e.code === Game_Action.EFFECT_ADD_DEBUFF
    );
    let buffText = '';
    if (buffEffects.length > 0) {
      const map = {}; // paramId -> { up, down }
      for (const e of buffEffects) {
        const pid = e.dataId;
        if (!map[pid]) map[pid] = { up: 0, down: 0 };
        if (e.code === Game_Action.EFFECT_ADD_BUFF) {
          map[pid].up += 1;
        } else if (e.code === Game_Action.EFFECT_ADD_DEBUFF) {
          map[pid].down += 1;
        }
      }
      const parts = [];
      for (const key of Object.keys(map)) {
        const pid = Number(key);
        const info = map[pid];
        const baseName = TextManager.param(pid);
        const coloredName = `\\C[${COLOR_PARAM}]${baseName}\\C[0]`;
        if (info.up > 0) parts.push(`${coloredName}+${info.up}段階`);
        if (info.down > 0) parts.push(`${coloredName}-${info.down}段階`);
      }
      if (parts.length > 0) buffText = parts.join('、');
    }

    // ダメージ種別・その他回復
    let effectKindText = '';
    const damage = skill.damage;
    if (damage && damage.type > 0) {
      switch (damage.type) {
        case 1: effectKindText += 'HPダメージ'; break;
        case 2: effectKindText += 'MPダメージ'; break;
        case 3: effectKindText += 'HP回復'; break;
        case 4: effectKindText += 'MP回復'; break;
        case 5: effectKindText += 'HP吸収'; break;
        case 6: effectKindText += 'MP吸収'; break;
      }
    }

    const hasRecover = effects.some(e =>
      e.code === Game_Action.EFFECT_RECOVER_HP ||
      e.code === Game_Action.EFFECT_RECOVER_MP ||
      e.code === Game_Action.EFFECT_GAIN_TP
    );
    if (hasRecover) {
      if (effectKindText) effectKindText += '／';
      effectKindText += '回復効果';
    }

    let line = '';

    if (buffText) {
      line += `\\C[${COLOR_LABEL}]能力変化\\C[0]: ${buffText}`;
    }

    if (effectKindText) {
      if (line) line += '　';
      line += `\\C[${COLOR_LABEL}]効果\\C[0]: ${effectKindText}`;
    }

    return line;
  }

  // ----------------------------------------
  // 専用ウィンドウクラス
  // ----------------------------------------
  class Window_SkillDetail extends Window_Base {
    initialize(rect) {
      super.initialize(rect);
      this._item = null;
      this._power = null;
      this._line1 = '';
      this._line2 = '';
      this._line3 = '';
      this.hide();          // 初期状態では非表示
      this.refresh();
    }

    setItem(skill) {
      if (this._item === skill) return;
      this._item = skill;
      if (skill) {
        this._power = calcSkillPower(skill);
        this._line1 = makeLine1(skill, this._power);
        this._line2 = makeLine2(skill);
        this._line3 = makeLine3(skill);
        this.show();        // スキルがある時だけ表示
      } else {
        this._power = null;
        this._line1 = '';
        this._line2 = '';
        this._line3 = '';
        this.hide();        // スキルが無い時は隠す
      }
      this.refresh();
    }

    clear() {
      this.setItem(null);
    }

    refresh() {
      this.contents.clear();
      if (!this._item) return;

      let y = 0;

      if (this._line1) this.drawTextEx(this._line1, 0, y);
      y += this.lineHeight();

      if (this._line2) this.drawTextEx(this._line2, 0, y);
      y += this.lineHeight();

      if (this._line3) this.drawTextEx(this._line3, 0, y);
    }
  }

  // ----------------------------------------
  // Window_SkillList に専用ウィンドウ参照を持たせる
  // ----------------------------------------
  Window_SkillList.prototype.setDetailWindow = function (detailWindow) {
    this._skillDetailWindow = detailWindow;
    this.callUpdateHelp();
  };

  const _Window_SkillList_updateHelp = Window_SkillList.prototype.updateHelp;
  Window_SkillList.prototype.updateHelp = function () {
    // まず元のヘルプ更新（説明文）
    _Window_SkillList_updateHelp.call(this);

    // その上で専用ウィンドウを更新
    const detail = this._skillDetailWindow;
    if (detail) {
      const item = this.item();
      if (item) detail.setItem(item);
      else detail.clear();
    }
  };

  // ----------------------------------------
  // Scene_Skill に専用ウィンドウを追加（ヘルプの直上）
  // ----------------------------------------
  const _Scene_Skill_create = Scene_Skill.prototype.create;
  Scene_Skill.prototype.create = function () {
    _Scene_Skill_create.call(this);
    this.createSkillDetailWindow();
  };

  Scene_Skill.prototype.skillDetailWindowRect = function () {
    const help = this._helpWindow;
    const ww = help.width;
    const wh = this.calcWindowHeight(3, true); // 3行分固定
    const wx = help.x;
    let wy = help.y - wh;
    if (wy < 0) wy = 0;
    return new Rectangle(wx, wy, ww, wh);
  };

  Scene_Skill.prototype.createSkillDetailWindow = function () {
    const rect = this.skillDetailWindowRect();
    this._skillDetailWindow = new Window_SkillDetail(rect);
    this.addWindow(this._skillDetailWindow);

    if (this._itemWindow) {
      this._itemWindow.setDetailWindow(this._skillDetailWindow);
    }
  };

  // スキル画面でキャンセルしたら非表示にする
  const _Scene_Skill_onItemCancel = Scene_Skill.prototype.onItemCancel;
  Scene_Skill.prototype.onItemCancel = function () {
    if (this._skillDetailWindow) this._skillDetailWindow.hide();
    _Scene_Skill_onItemCancel.call(this);
  };

  // ----------------------------------------
  // Scene_Battle に専用ウィンドウを追加
  // ----------------------------------------
  const _Scene_Battle_createAllWindows = Scene_Battle.prototype.createAllWindows;
  Scene_Battle.prototype.createAllWindows = function () {
    _Scene_Battle_createAllWindows.call(this);
    this.createSkillDetailWindow_Battle();
  };

  Scene_Battle.prototype.skillDetailWindowRect_Battle = function () {
    const status = this._statusWindow;
    const ww = Graphics.boxWidth;
    const wh = this.calcWindowHeight(3, true);
    const wx = 0;
    let wy = status ? status.y - wh : Graphics.boxHeight - wh;
    if (wy < 0) wy = 0;
    return new Rectangle(wx, wy, ww, wh);
  };

  Scene_Battle.prototype.createSkillDetailWindow_Battle = function () {
    const rect = this.skillDetailWindowRect_Battle();
    this._skillDetailWindow = new Window_SkillDetail(rect);

    // バトル画面では枠線を非表示
    this._skillDetailWindow.frameVisible = false;

    this.addWindow(this._skillDetailWindow);

    if (this._skillWindow) {
      this._skillWindow.setDetailWindow(this._skillDetailWindow);
    }
  };

  // スキルコマンドに入ったとき：現在のカーソル位置で即更新
  const _Scene_Battle_commandSkill = Scene_Battle.prototype.commandSkill;
  Scene_Battle.prototype.commandSkill = function () {
    if (this._skillDetailWindow) {
      this._skillDetailWindow.clear();
    }

    _Scene_Battle_commandSkill.call(this);

    if (this._skillWindow && this._skillWindow.active && this._skillWindow.index() >= 0) {
      this._skillWindow.callUpdateHelp();
    }
  };

  // スキル決定／キャンセルで非表示にする
  const _Scene_Battle_onSkillOk = Scene_Battle.prototype.onSkillOk;
  Scene_Battle.prototype.onSkillOk = function () {
    if (this._skillDetailWindow) this._skillDetailWindow.hide();
    _Scene_Battle_onSkillOk.call(this);
  };

  const _Scene_Battle_onSkillCancel = Scene_Battle.prototype.onSkillCancel;
  Scene_Battle.prototype.onSkillCancel = function () {
    if (this._skillDetailWindow) this._skillDetailWindow.hide();
    _Scene_Battle_onSkillCancel.call(this);
  };

})();
