/*:
 * @target MZ
 * @plugindesc v2.0 指定したスキル群の速度補正を一括で上書きします。
 * @author あなたの名前
 *
 * @help
 * 指定したスキルの「速度補正」を、データベース上の値とは別に
 * プラグインパラメータで一括指定し、その値でデータベース自体を上書きします。
 *
 * ■仕様
 * ・「スキル速度グループ一覧」に登録した各グループについて、
 *   グループ内のすべてのスキルの「速度補正」(database の speed) を
 *   指定した値に変更します。
 * ・登録していないスキルは、データベースの値がそのまま使われます。
 * ・通常攻撃やアイテムには影響しません（スキルのみ対象）。
 *
 * ■注意
 * ・このプラグインは「ゲーム起動時に一度だけ」$dataSkills[...] の
 *   speed を上書きします。他プラグインから見ても、最初から
 *   その値だったかのように扱われます。
 *
 * このプラグインにはプラグインコマンドはありません。
 *
 * @param SkillSpeedGroupList
 * @text スキル速度グループ一覧
 * @type struct<SkillSpeedGroup>[]
 * @default []
 * @desc 複数スキルに同一の速度補正を与えるグループの一覧。
 */

/*~struct~SkillSpeedGroup:
 * @param Skills
 * @text 対象スキル
 * @type skill[]
 * @desc このグループに含めるスキル。すべて同じ速度補正になります。
 *
 * @param SpeedText
 * @text 速度補正（テキスト）
 * @desc このグループのスキルに適用する速度補正値。例: 0, 50, -100
 * @default 0
 */

(() => {
    'use strict';

    // ★ファイル名と一致させる
    const pluginName = 'SkillSpeedOverrideMZ';

    const parameters = PluginManager.parameters(pluginName);
    const groupListRaw = JSON.parse(parameters['SkillSpeedGroupList'] || '[]');

    /** @type {Record<number, number>} */
    const skillSpeedMap = {};

    for (const groupEntry of groupListRaw) {
        if (!groupEntry) continue;

        const groupObj = JSON.parse(groupEntry);

        const speedText = (groupObj.SpeedText != null ? groupObj.SpeedText : '0').trim();
        const speed = Number(speedText || '0');
        if (Number.isNaN(speed)) {
            console.warn('[SkillSpeedOverrideMZ] 無効な速度値が指定されています:', groupObj);
            continue;
        }

        const skillsRaw = groupObj.Skills ? JSON.parse(groupObj.Skills) : [];
        if (!Array.isArray(skillsRaw)) continue;

        for (const idStr of skillsRaw) {
            const skillId = Number(idStr || 0);
            if (skillId > 0) {
                skillSpeedMap[skillId] = speed;
            }
        }
    }

    console.log('[SkillSpeedOverrideMZ] skillSpeedMap:', skillSpeedMap);

    // データベース読み込み完了後に speed を上書きする
    const _DataManager_isDatabaseLoaded = DataManager.isDatabaseLoaded;
    DataManager.isDatabaseLoaded = function() {
        const loaded = _DataManager_isDatabaseLoaded.call(this);
        if (loaded && !this._skillSpeedOverrideApplied) {
            this._skillSpeedOverrideApplied = true;
            applySkillSpeedOverrides();
        }
        return loaded;
    };

    function applySkillSpeedOverrides() {
        if (!$dataSkills) return;

        for (const idStr in skillSpeedMap) {
            const id = Number(idStr);
            const skill = $dataSkills[id];
            if (skill) {
                skill.speed = skillSpeedMap[id];
                // デバッグしたい場合はコメントアウトを外す
                // console.log(`[SkillSpeedOverrideMZ] skill ${id} (${skill.name}) speed -> ${skill.speed}`);
            }
        }
    }
})();
