//=============================================================================
// StateBattleBgmMZ.js
//=============================================================================
/*:
 * @target MZ
 * @plugindesc v1.0 指定ステート中だけ戦闘BGM／勝利MEを差し替える
 * @author KumoTen
 * @url 
 * 
 * @help
 * 指定したステートがアクターに付与されている間だけ、
 * 戦闘BGMと勝利MEを差し替えます。
 * ステートが全員から外れるか、戦闘が終了すると元に戻ります。
 * 
 * 特徴:
 *  - 複数ステートを登録可能
 *  - 各ステートごとに戦闘BGM／勝利MEを個別指定可能
 *  - 複数のステートが同時に付与された場合は「優先度」の高いものが採用
 *  - 戦闘中にステートが付与／解除された時もBGMを自動更新
 *  - 勝利MEは勝利時に一度だけ差し替え再生し、その後は自動で元に戻る
 * 
 * 対象:
 *  - パーティアクター（味方側）のステートのみを監視します。
 *    敵ステートは対象外です。
 * 
 * 導入方法:
 *  1. プラグインを js/plugins フォルダに StateBattleBgmMZ.js
 *     として保存してください。
 *  2. プラグイン管理で有効化し、パラメータを設定してください。
 * 
 * 注意:
 *  - 戦闘開始時のデフォルトBGMは $gameSystem.battleBgm() を基準に
 *    保存・復元します。
 *  - 戦闘イベントや他プラグインで BGM を頻繁に変更している場合は、
 *    復元先がそこで指定したものではなく、システム設定の戦闘BGMに
 *    戻ることがあります。
 * 
 * 利用規約:
 *  - RPGツクールMZ向け
 *  - 商用・非商用問わず利用可
 *  - クレジット表記任意（してもらえると嬉しい）
 *
 * @param StateBgmList
 * @text ステート別BGM設定一覧
 * @type struct<StateBgm>[]
 * @default []
 * @desc ステートごとの戦闘BGM／勝利MEを設定します。
 */

/*~struct~StateBgm:
 * @param StateId
 * @text ステートID
 * @type state
 * @default 1
 * @desc 対象となるステートID。
 *
 * @param Priority
 * @text 優先度
 * @type number
 * @min 0
 * @max 9999
 * @default 0
 * @desc 数値が大きいほど優先されます。複数ステートが同時に付与された場合に使用。
 *
 * @param BattleBgm
 * @text 戦闘BGM
 * @type struct<BgmAudio>
 * @default {"name":"","volume":"90","pitch":"100","pan":"0"}
 * @desc このステートが有効中に流す戦闘BGM。未指定（ファイル名空）なら戦闘BGMは変えません。
 *
 * @param VictoryMe
 * @text 勝利ME
 * @type struct<MeAudio>
 * @default {"name":"","volume":"90","pitch":"100","pan":"0"}
 * @desc このステートが有効中に流す勝利ME。未指定（ファイル名空）なら勝利MEは変えません。
 */

/*~struct~BgmAudio:
 * @param name
 * @text ファイル名
 * @type file
 * @dir audio/bgm
 * @default 
 *
 * @param volume
 * @text 音量
 * @type number
 * @min 0
 * @max 100
 * @default 90
 *
 * @param pitch
 * @text ピッチ
 * @type number
 * @min 50
 * @max 150
 * @default 100
 *
 * @param pan
 * @text 位相
 * @type number
 * @min -100
 * @max 100
 * @default 0
 */

/*~struct~MeAudio:
 * @param name
 * @text ファイル名
 * @type file
 * @dir audio/me
 * @default 
 *
 * @param volume
 * @text 音量
 * @type number
 * @min 0
 * @max 100
 * @default 90
 *
 * @param pitch
 * @text ピッチ
 * @type number
 * @min 50
 * @max 150
 * @default 100
 *
 * @param pan
 * @text 位相
 * @type number
 * @min -100
 * @max 100
 * @default 0
 */

(() => {
  "use strict";

  const pluginName = "StateBattleBgmMZ";
  const parameters = PluginManager.parameters(pluginName);

  //-----------------------------------------------------------------------------
  // パラメータ解析
  //-----------------------------------------------------------------------------

  /**
   * @typedef {Object} StateBgmSetting
   * @property {number} stateId
   * @property {number} priority
   * @property {AudioObject|null} battleBgm
   * @property {AudioObject|null} victoryMe
   */

  /**
   * @typedef {Object} AudioObject
   * @property {string} name
   * @property {number} volume
   * @property {number} pitch
   * @property {number} pan
   * @property {number} [pos]
   */

  /**
   * JSON文字列を安全にパース
   * @param {string} str
   * @param {any} [defaultValue]
   * @returns {any}
   */
  function parseJson(str, defaultValue = null) {
    try {
      return JSON.parse(str || "");
    } catch (e) {
      return defaultValue;
    }
  }

  /**
   * BGM/ME用オブジェクトへ整形
   * @param {any} src
   * @param {boolean} isBgm
   * @returns {AudioObject|null}
   */
  function toAudioObject(src, isBgm) {
    if (!src) return null;
    const name = src.name || "";
    if (!name) return null;
    const audio = {
      name: String(name),
      volume: Number(src.volume || 90),
      pitch: Number(src.pitch || 100),
      pan: Number(src.pan || 0)
    };
    if (isBgm) {
      audio.pos = 0;
    }
    return audio;
  }

  /** @type {StateBgmSetting[]} */
  const StateBgmList = parseJson(parameters["StateBgmList"] || "[]", [])
    .map((s) => parseJson(s, null))
    .filter((obj) => obj)
    .map((obj) => {
      const bgmRaw = parseJson(obj.BattleBgm || "null", null);
      const meRaw = parseJson(obj.VictoryMe || "null", null);
      return {
        stateId: Number(obj.StateId || 0),
        priority: Number(obj.Priority || 0),
        battleBgm: toAudioObject(bgmRaw, true),
        victoryMe: toAudioObject(meRaw, false)
      };
    })
    .filter((setting) => setting.stateId > 0);

  //-----------------------------------------------------------------------------
  // 内部状態
  //-----------------------------------------------------------------------------

  // 現在 BGM に反映されている「状態由来ステートID」
  BattleManager._stateBgmCurrentStateId = 0;
  // 戦闘開始時点のデフォルト戦闘BGM
  BattleManager._stateBgmOriginalBgm = null;

  //-----------------------------------------------------------------------------
  // ユーティリティ: 有効なステート設定を取得
  //-----------------------------------------------------------------------------

  /**
   * 現在の戦闘状態から、有効な StateBgmSetting を決定する。
   * - パーティアクターのいずれかがステートを持っていれば候補
   * - 優先度の数値が大きい設定が最優先
   * - 優先度が同じ場合は先に定義されたものが優先
   * @returns {StateBgmSetting|null}
   */
  function findActiveStateBgmSetting() {
    if (!StateBgmList.length) return null;
    if (!$gameParty.inBattle()) return null;

    const members = $gameParty.members();
    if (!members.length) return null;

    /** @type {StateBgmSetting|null} */
    let best = null;

    for (const setting of StateBgmList) {
      const id = setting.stateId;
      if (id <= 0) continue;
      const hasState = members.some((battler) => battler.isStateAffected(id));
      if (!hasState) continue;

      if (!best || setting.priority > best.priority) {
        best = setting;
      }
    }

    return best;
  }

  //-----------------------------------------------------------------------------
  // BGM 更新処理
  //-----------------------------------------------------------------------------

  /**
   * 戦闘中のステート状況に応じて戦闘BGMを更新
   */
  function updateBattleBgmByState() {
    if (!$gameParty.inBattle()) return;

    const newSetting = findActiveStateBgmSetting();
    const currentId = BattleManager._stateBgmCurrentStateId || 0;

    if (newSetting) {
      // 新しく状態由来BGMを適用する場合
      if (!BattleManager._stateBgmOriginalBgm) {
        // 元の戦闘BGMを保存（システム設定の戦闘BGM）
        BattleManager._stateBgmOriginalBgm = JsonEx.makeDeepCopy(
          $gameSystem.battleBgm()
        );
      }

      if (newSetting.stateId !== currentId) {
        BattleManager._stateBgmCurrentStateId = newSetting.stateId;

        if (newSetting.battleBgm) {
          const bgm = JsonEx.makeDeepCopy(newSetting.battleBgm);
          // 現在の再生位置を引き継ぐ場合（好みでコメントアウト可）
          if (AudioManager._currentBgm && AudioManager._currentBgm.name) {
            bgm.pos = AudioManager._currentBgm.pos || 0;
          }
          AudioManager.playBgm(bgm);
        }
      }
    } else {
      // 対象ステートが誰も持っていない場合、元BGMへ戻す
      if (currentId && BattleManager._stateBgmOriginalBgm) {
        BattleManager._stateBgmCurrentStateId = 0;
        const orig = BattleManager._stateBgmOriginalBgm;
        BattleManager._stateBgmOriginalBgm = null;
        if (orig && orig.name) {
          AudioManager.playBgm(orig);
        }
      }
    }
  }

  //-----------------------------------------------------------------------------
  // 戦闘開始時: 元BGMを保存し、開始時点のステートでBGM判定
  //-----------------------------------------------------------------------------

  const _Scene_Battle_start = Scene_Battle.prototype.start;
  Scene_Battle.prototype.start = function () {
    _Scene_Battle_start.call(this);
    BattleManager._stateBgmCurrentStateId = 0;
    BattleManager._stateBgmOriginalBgm = JsonEx.makeDeepCopy(
      $gameSystem.battleBgm()
    );
    updateBattleBgmByState();
  };

  //-----------------------------------------------------------------------------
  // 戦闘終了時: 内部状態をクリア
  //-----------------------------------------------------------------------------

  const _BattleManager_endBattle = BattleManager.endBattle;
  BattleManager.endBattle = function (result) {
    BattleManager._stateBgmCurrentStateId = 0;
    BattleManager._stateBgmOriginalBgm = null;
    _BattleManager_endBattle.call(this, result);
  };

  //-----------------------------------------------------------------------------
  // 勝利ME差し替え
  //-----------------------------------------------------------------------------

  const _BattleManager_playVictoryMe = BattleManager.playVictoryMe;
  BattleManager.playVictoryMe = function () {
    const setting = findActiveStateBgmSetting();
    if (setting && setting.victoryMe && setting.victoryMe.name) {
      AudioManager.playMe(setting.victoryMe);
    } else {
      _BattleManager_playVictoryMe.call(this);
    }
  };

  //-----------------------------------------------------------------------------
  // ステート付与・解除時にBGMを更新
  //-----------------------------------------------------------------------------

  const _Game_Battler_addNewState = Game_Battler.prototype.addNewState;
  Game_Battler.prototype.addNewState = function (stateId) {
    _Game_Battler_addNewState.call(this, stateId);
    if ($gameParty.inBattle()) {
      updateBattleBgmByState();
    }
  };

  const _Game_Battler_removeState = Game_Battler.prototype.removeState;
  Game_Battler.prototype.removeState = function (stateId) {
    _Game_Battler_removeState.call(this, stateId);
    if ($gameParty.inBattle()) {
      updateBattleBgmByState();
    }
  };

  const _Game_Battler_eraseState = Game_Battler.prototype.eraseState;
  Game_Battler.prototype.eraseState = function (stateId) {
    _Game_Battler_eraseState.call(this, stateId);
    if ($gameParty.inBattle()) {
      updateBattleBgmByState();
    }
  };

  const _Game_Battler_clearStates = Game_Battler.prototype.clearStates;
  Game_Battler.prototype.clearStates = function () {
    _Game_Battler_clearStates.call(this);
    if ($gameParty.inBattle()) {
      updateBattleBgmByState();
    }
  };
})();
