/*:
 * @target MZ
 * @plugindesc v1.0 タイトル画面のコンティニューでロードした直後に、指定変数へ定数または他変数の値を代入します。
 * @author YourName
 *
 * @help
 * タイトル画面から「コンティニュー」でセーブデータを読み込んだ直後に、
 * そのセーブデータ内の「指定した変数」に、
 * ①プラグインパラメータで指定した定数値
 * または
 * ②別の「指定した変数」の値
 * を代入します。
 *
 * ■動作タイミング
 * ・タイトル画面 → コンティニュー → セーブデータ選択 → ロード成功後
 *   のタイミングでのみ代入を行います。
 * ・ゲーム中のメニューからの「ロード」では動作しません。
 *
 * ■注意
 * ・対象変数IDが 0 または未設定の場合、このプラグインは何もしません。
 *
 * ------------------------------------------------------------
 * @param TargetVariableId
 * @text 対象変数ID
 * @type variable
 * @default 0
 * @desc ロード後に値を代入する変数ID。0だと何もしません。
 *
 * @param AssignMode
 * @text 代入モード
 * @type select
 * @option 定数を代入
 * @value constant
 * @option 他の変数の値を代入
 * @value variable
 * @default constant
 * @desc 対象変数に代入する内容を指定します。
 *
 * @param ConstantValue
 * @text 定数値
 * @type number
 * @min -999999999
 * @default 0
 * @desc 代入モードが「定数を代入」のときに使われる値。
 *
 * @param SourceVariableId
 * @text 参照元変数ID
 * @type variable
 * @default 0
 * @desc 代入モードが「他の変数の値を代入」のときに参照する変数ID。
 */

(() => {
    "use strict";

    const pluginName = document.currentScript.src.match(/([^/\\]+)\.js$/)[1];
    const params = PluginManager.parameters(pluginName);

    const TARGET_VAR_ID   = Number(params["TargetVariableId"] || 0);
    const ASSIGN_MODE     = String(params["AssignMode"] || "constant");
    const CONSTANT_VALUE  = Number(params["ConstantValue"] || 0);
    const SOURCE_VAR_ID   = Number(params["SourceVariableId"] || 0);

    /**
     * ロード成功時の拡張
     */
    const _Scene_Load_onLoadSuccess = Scene_Load.prototype.onLoadSuccess;
    Scene_Load.prototype.onLoadSuccess = function() {
        _Scene_Load_onLoadSuccess.call(this);

        // 直前がタイトル画面のときのみ（＝タイトルからのコンティニューと判断）
        if (SceneManager.isPreviousScene(Scene_Title)) {
            applyContinueVariableAssignment();
        }
    };

    /**
     * タイトル画面からコンティニューでロードした直後に実行される処理
     */
    function applyContinueVariableAssignment() {
        if (!TARGET_VAR_ID) return;
        if (!$gameVariables) return;

        let value;

        if (ASSIGN_MODE === "variable") {
            // 他の変数の値を代入
            if (!SOURCE_VAR_ID) return;
            value = $gameVariables.value(SOURCE_VAR_ID);
        } else {
            // 定数を代入（デフォルト）
            value = CONSTANT_VALUE;
        }

        $gameVariables.setValue(TARGET_VAR_ID, value);
    }

})();
