/*:
 * @target MZ
 * @plugindesc v1.1 変数の値に応じた文字列を常時表示するウィンドウを追加します（マップ移動中＆バトル中のみ）。
 * @author YourName
 *
 * @help
 * 指定した「変数ID」の値に応じて、あらかじめ指定した文字列を
 * 画面端に常時表示するプラグインです。
 *
 * ■表示タイミング
 *   ・プレイヤーが操作可能なマップ画面（イベント実行中・メッセージ表示中は非表示）
 *   ・バトル画面
 *
 * ■表示されない場面
 *   ・メニュー、ショップ、タイトル、ゲームオーバーなど
 *
 * ------------------------------------------------------------
 * @param VariableId
 * @text 監視する変数ID
 * @type variable
 * @default 0
 * @desc この変数の値に応じてテキストを表示します。0 だと機能しません。
 *
 * @param ValueTextList
 * @text 値と表示テキストの対応表
 * @type struct<ValueText>[]
 * @default []
 * @desc 変数の値ごとの表示内容を設定します。
 *
 * @param DefaultText
 * @text その他用テキスト
 * @type string
 * @default 
 * @desc 対応表に存在しない値のときに表示するテキスト。
 *
 * @param WindowX
 * @text ウィンドウX座標
 * @type number
 * @default 0
 * @desc ウィンドウ左上のX座標（ピクセル）
 *
 * @param WindowY
 * @text ウィンドウY座標
 * @type number
 * @default 0
 * @desc ウィンドウ左上のY座標（ピクセル）
 *
 * @param WindowWidth
 * @text ウィンドウ幅
 * @type number
 * @default 200
 * @desc ウィンドウ幅（ピクセル）
 *
 * @param Lines
 * @text 行数
 * @type number
 * @min 1
 * @max 4
 * @default 1
 * @desc 高さ計算に使う行数
 *
 * @param FontSize
 * @text フォントサイズ
 * @type number
 * @default 28
 * @desc 表示テキストのフォントサイズ
 */
/*~struct~ValueText:
 * @param value
 * @text 変数の値
 * @type number
 * @default 0
 *
 * @param text
 * @text 表示テキスト
 * @type string
 * @default 
 */

(() => {
    "use strict";

    const pluginName = document.currentScript.src.match(/([^/\\]+)\.js$/)[1];
    const params = PluginManager.parameters(pluginName);

    const VAR_ID       = Number(params["VariableId"] || 0);
    const DEFAULT_TEXT = String(params["DefaultText"] || "");
    const WIN_X        = Number(params["WindowX"] || 0);
    const WIN_Y        = Number(params["WindowY"] || 0);
    const WIN_W        = Number(params["WindowWidth"] || 200);
    const LINES        = Math.max(1, Number(params["Lines"] || 1));
    const FONT_SIZE    = Number(params["FontSize"] || 28);

    const valueTextListRaw = JSON.parse(params["ValueTextList"] || "[]");
    const valueTextList = valueTextListRaw.map(str => {
        const obj = JSON.parse(str);
        return {
            value: Number(obj.value || 0),
            text: String(obj.text || "")
        };
    });

    function getTextForValue(value) {
        for (const entry of valueTextList) {
            if (entry.value === value) return entry.text;
        }
        return DEFAULT_TEXT;
    }

    // このプラグイン専用の高さ計算（他のウィンドウには影響しない）
    function variableTextWindowHeight() {
        const lineHeight = FONT_SIZE + 4; // ほぼ行高
        const padding = 18;               // Window_Base の既定パディング想定
        return lineHeight * LINES + padding * 2;
    }

    class Window_VariableTextDisplay extends Window_Base {
        initialize(rect) {
            super.initialize(rect);
            this._lastValue = null;
            this._lastText = "";
            this.opacity = 255;
            this.refresh();
        }

        lineHeight() {
            return FONT_SIZE + 4;
        }

        standardFontSize() {
            return FONT_SIZE;
        }

        update() {
            super.update();
            this.updateVisibility();
            this.updateText();
        }

        updateVisibility() {
            if (!VAR_ID) {
                this.visible = false;
                return;
            }

            const scene = SceneManager._scene;

            if (scene instanceof Scene_Map) {
                const show =
                    !$gameMap.isEventRunning() &&
                    !$gameMessage.isBusy();
                this.visible = show;
            } else if (scene instanceof Scene_Battle) {
                this.visible = true;
            } else {
                this.visible = false;
            }
        }

        updateText() {
            if (!this.visible) return;
            const value = $gameVariables ? $gameVariables.value(VAR_ID) : null;
            if (this._lastValue !== value) {
                this._lastValue = value;
                this.refresh();
            }
        }

        refresh() {
            this.contents.clear();
            if (!VAR_ID) return;
            const value = $gameVariables.value(VAR_ID);
            const text = getTextForValue(value);
            this._lastText = text;
            this.drawText(text, 0, 0, this.contentsWidth(), "center");
        }
    }

    // -------- Scene_Map --------
    const _Scene_Map_createAllWindows = Scene_Map.prototype.createAllWindows;
    Scene_Map.prototype.createAllWindows = function() {
        _Scene_Map_createAllWindows.call(this);
        this.createVariableTextDisplayWindow();
    };

    Scene_Map.prototype.createVariableTextDisplayWindow = function() {
        if (!VAR_ID) return;
        const rect = this.variableTextDisplayWindowRect();
        this._variableTextDisplayWindow = new Window_VariableTextDisplay(rect);
        this.addWindow(this._variableTextDisplayWindow);
    };

    Scene_Map.prototype.variableTextDisplayWindowRect = function() {
        const width  = WIN_W;
        const height = variableTextWindowHeight();
        const x = WIN_X;
        const y = WIN_Y;
        return new Rectangle(x, y, width, height);
    };

    // -------- Scene_Battle --------
    const _Scene_Battle_createAllWindows = Scene_Battle.prototype.createAllWindows;
    Scene_Battle.prototype.createAllWindows = function() {
        _Scene_Battle_createAllWindows.call(this);
        this.createVariableTextDisplayWindow();
    };

    Scene_Battle.prototype.createVariableTextDisplayWindow = function() {
        if (!VAR_ID) return;
        const rect = this.variableTextDisplayWindowRect();
        this._variableTextDisplayWindow = new Window_VariableTextDisplay(rect);
        this.addWindow(this._variableTextDisplayWindow);
    };

    Scene_Battle.prototype.variableTextDisplayWindowRect = function() {
        const width  = WIN_W;
        const height = variableTextWindowHeight();
        const x = WIN_X;
        const y = WIN_Y;
        return new Rectangle(x, y, width, height);
    };

})();
