/*:
 * @target MZ
 * @plugindesc v1.1.6 KT_TamaKyoudaReact: タマ強打ステート付与メッセージ（トースト/顔画像/右半分ランダム配置/全文見える自動高さ/戦闘終了直前に未表示分をまとめてトースト表示・フリーズ対策/敵への「付与行動」でも表示）
 * @author ChatGPT
 *
 * @help
 * ■v1.1.6 メッセージ表示タイミング（今回の要望）
 * - 敵が対象の場合、以下の2条件で表示します。
 *   1) タマ強打を付与したとき（新規付与）
 *   2) タマ強打中の敵に「タマ強打を付与する行動」を取ったとき
 *      （= 行動の効果にタマ強打付与が含まれていれば、既に付与済みでも表示）
 *
 * - 2) は「行動を取った」タイミング重視のため、付与が抵抗/ミス等で無効になっても表示します。
 *   （無効時は表示しない方が良い場合は、その旨を言ってください。isHit等で絞れます）
 *
 * ■戦闘終了直前の未表示分まとめ表示（重要）
 * - 未表示分を「Window側の専用キュー」に移してから順番表示します。
 * - これにより、戦闘終了処理で “全部流す” をしても、RT側キューに戻って循環することがなくなり、フリーズしません。
 * - 旧方式の $gameMessage 退避は使用しません（リザルト窓に会話形式で出るのを避けるため）。
 *
 * ■置換
 * {target} : ステートになった対象名
 *
 * @param StateId
 * @text タマ強打ステートID
 * @type state
 * @default 2
 *
 * @param DeathStateId
 * @text 戦闘不能ステートID
 * @type state
 * @default 1
 *
 * @param EnableDeathSameFrame
 * @text 同フレーム戦闘不能でも表示
 * @type boolean
 * @default true
 *
 * @param DebugLog
 * @text デバッグログ
 * @type boolean
 * @default false
 *
 * @param EnemyMessages
 * @text 敵がタマ強打のときのメッセージ（複数）
 * @type string[]
 * @default ["{target}は悶絶している！","{target}に強烈な一撃！"]
 *
 * @param EnemyActorMessages
 * @text 敵がタマ強打（付与した味方アクター別）
 * @type struct<ActorMsg>[]
 * @default []
 *
 * @param ActorDefaultMessages
 * @text 味方がタマ強打（共通）のメッセージ（複数）
 * @type string[]
 * @default ["{target}！大丈夫！？"]
 *
 * @param ActorMessages
 * @text 味方がタマ強打（アクター別）
 * @type struct<ActorMsg>[]
 * @default []
 *
 * @param ActorFaceOverrides
 * @text アクター顔指定（複数）
 * @type struct<ActorFaceEntry>[]
 * @desc アクターIDごとに顔（img/faces）を指定。未指定ならDB設定の顔を使用。
 * @default []
 *
 * @param ToastWidth
 * @text トースト幅
 * @type number
 * @min 240
 * @default 560
 *
 * @param ToastMinHeight
 * @text トースト最小高さ
 * @type number
 * @min 96
 * @default 160
 *
 * @param ToastMaxHeight
 * @text トースト最大高さ
 * @type number
 * @min 96
 * @default 360
 *
 * @param AutoFitHeight
 * @text 全文が見えるように自動高さ
 * @type boolean
 * @default true
 *
 * @param ToastFrames
 * @text 表示フレーム数
 * @type number
 * @min 1
 * @default 120
 *
 * @param ToastOpenCloseSpeed
 * @text 開閉速度（1-64）
 * @type number
 * @min 1
 * @max 64
 * @default 16
 *
 * @param RightHalfDefault
 * @text 右半分を基本にする
 * @type boolean
 * @desc trueなら、位置範囲未指定時に「画面右半分」内でランダム表示
 * @default true
 *
 * @param ToastXMin
 * @text X最小（範囲指定）
 * @type number
 * @min -99999
 * @max 99999
 * @default 0
 *
 * @param ToastXMax
 * @text X最大（範囲指定）
 * @type number
 * @min -99999
 * @max 99999
 * @default 0
 *
 * @param ToastYMin
 * @text Y最小（範囲指定）
 * @type number
 * @min -99999
 * @max 99999
 * @default 0
 *
 * @param ToastYMax
 * @text Y最大（範囲指定）
 * @type number
 * @min -99999
 * @max 99999
 * @default 0
 *
 * @param FlushToastOnEndBattle
 * @text 戦闘終了直前：未表示分をまとめてトーストに移す
 * @type boolean
 * @default true
 *
 * @param PreloadActorFaces
 * @text 戦闘開始時：顔を先読み
 * @type boolean
 * @default true
 */

/*~struct~ActorMsg:
 * @param ActorId
 * @text アクターID
 * @type actor
 * @default 1
 *
 * @param Messages
 * @text このアクター用メッセージ（複数）
 * @type string[]
 * @default ["{target}！？い、いまのは…！"]
 */

/*~struct~ActorFaceEntry:
 * @param ActorId
 * @text アクターID
 * @type actor
 * @default 1
 *
 * @param FaceFile
 * @text 顔ファイル（img/faces）
 * @type file
 * @dir img/faces
 * @default
 *
 * @param FaceIndex
 * @text 顔インデックス（0-7）
 * @type number
 * @min 0
 * @max 7
 * @default 0
 */

(() => {
  "use strict";

  function detectPluginName() {
    try {
      const cs = document.currentScript;
      if (cs && cs.src) {
        const m = cs.src.match(/([^/\\]+)\.js(?:\?.*)?$/);
        if (m) return m[1];
      }
    } catch {}
    const KEY = "KT_TamaKyoudaReact";
    try {
      if (Array.isArray(window.$plugins)) {
        const hit = window.$plugins.find(p => p && typeof p.description === "string" && p.description.includes(KEY));
        if (hit && hit.name) return String(hit.name);
      }
    } catch {}
    return "KT_TamaKyoudaReact";
  }

  const PLUGIN_NAME = detectPluginName();
  const P = PluginManager.parameters(PLUGIN_NAME) || {};

  const U = {
    bool(v, def) { if (v === undefined || v === null || v === "") return !!def; return String(v) === "true"; },
    num(v, def) { const n = Number(v); return Number.isFinite(n) ? n : def; },
    clamp(n, a, b) { n = Number(n); if (!Number.isFinite(n)) return a; return Math.max(a, Math.min(b, n)); },
    strArray(v) {
      if (!v) return [];
      try {
        const arr = JSON.parse(v);
        if (!Array.isArray(arr)) return [];
        return arr.map(s => String(s)).map(s => s.trim()).filter(s => s.length > 0);
      } catch { return []; }
    },
    pickRandom(arr) {
      if (!arr || arr.length === 0) return "";
      const i = (Math.randomInt ? Math.randomInt(arr.length) : Math.floor(Math.random() * arr.length));
      return arr[i];
    },
    inBattle() { try { return $gameParty && $gameParty.inBattle(); } catch { return false; } },
    battlerName(b) { try { return b && b.name ? b.name() : ""; } catch { return ""; } },
    isActor(b) { try { return !!(b && b.isActor && b.isActor()); } catch { return false; } },
    isEnemy(b) { try { return !!(b && b.isEnemy && b.isEnemy()); } catch { return false; } },
    actorId(b) { try { return b && b.actorId ? b.actorId() : 0; } catch { return 0; } },
    enemyKey(b) {
      try {
        const eid = b && b.enemyId ? b.enemyId() : 0;
        const idx = b && b.index ? b.index() : 0;
        return "E" + eid + "_I" + idx;
      } catch {
        return "E0_I0";
      }
    },
    frameCount() {
      try { return (typeof Graphics !== "undefined" && Number.isFinite(Graphics.frameCount)) ? Graphics.frameCount : 0; }
      catch { return 0; }
    },
    randInt(min, max) {
      min = Math.floor(Number(min)); max = Math.floor(Number(max));
      if (!Number.isFinite(min) || !Number.isFinite(max)) return 0;
      if (min > max) [min, max] = [max, min];
      return min + Math.floor(Math.random() * (max - min + 1));
    }
  };

  const CFG = {
    stateId: U.num(P.StateId, 2),
    deathStateId: U.num(P.DeathStateId, 1),
    enableDeathSameFrame: U.bool(P.EnableDeathSameFrame, true),
    debug: U.bool(P.DebugLog, false),

    enemyMessages: U.strArray(P.EnemyMessages),
    enemyActorMessageMap: Object.create(null),

    actorDefaultMessages: U.strArray(P.ActorDefaultMessages),
    actorMessageMap: Object.create(null),

    actorFaceOverrideMap: Object.create(null),

    toastWidth: Math.max(240, U.num(P.ToastWidth, 560)),
    toastMinHeight: Math.max(96, U.num(P.ToastMinHeight, 160)),
    toastMaxHeight: Math.max(96, U.num(P.ToastMaxHeight, 360)),
    autoFitHeight: U.bool(P.AutoFitHeight, true),
    toastFrames: Math.max(1, U.num(P.ToastFrames, 120)),
    toastSpeed: U.clamp(U.num(P.ToastOpenCloseSpeed, 16), 1, 64),

    rightHalfDefault: U.bool(P.RightHalfDefault, true),
    xMin: U.num(P.ToastXMin, 0),
    xMax: U.num(P.ToastXMax, 0),
    yMin: U.num(P.ToastYMin, 0),
    yMax: U.num(P.ToastYMax, 0),

    flushToastOnEndBattle: U.bool(P.FlushToastOnEndBattle, true),
    preloadActorFaces: U.bool(P.PreloadActorFaces, true),
  };

  function dlog(label, payload) {
    if (!CFG.debug) return;
    try { console.log("[" + PLUGIN_NAME + "] " + label, payload); } catch {}
  }

  function parseStructList(rawJson) {
    const raw = rawJson || "[]";
    let list = [];
    try { list = JSON.parse(raw); if (!Array.isArray(list)) list = []; } catch { list = []; }
    return list.map(s => { try { return JSON.parse(s); } catch { return {}; } });
  }

  function parseActorMsgList(rawJson, outMap) {
    const list = parseStructList(rawJson);
    for (const o of list) {
      const aid = U.num(o.ActorId, 0);
      const msgs = U.strArray(o.Messages);
      if (aid > 0 && msgs.length > 0) outMap[aid] = msgs;
    }
  }

  function parseActorFaceList(rawJson, outMap) {
    const list = parseStructList(rawJson);
    for (const o of list) {
      const aid = U.num(o.ActorId, 0);
      const faceName = String(o.FaceFile || "").trim();
      const faceIndex = U.clamp(U.num(o.FaceIndex, 0), 0, 7);
      if (aid > 0 && faceName) outMap[aid] = { faceName, faceIndex };
    }
  }

  parseActorMsgList(P.ActorMessages, CFG.actorMessageMap);
  parseActorMsgList(P.EnemyActorMessages, CFG.enemyActorMessageMap);
  parseActorFaceList(P.ActorFaceOverrides, CFG.actorFaceOverrideMap);

  const RT = {
    reset() {
      if (!$gameTemp) return;
      $gameTemp._kt_tk_guard = Object.create(null);
      $gameTemp._kt_tk_pendingToast = [];
      $gameTemp._kt_tk_tamaFrame = Object.create(null);
      $gameTemp._kt_tk_actionStack = [];
    },
    guardMap() {
      if (!$gameTemp) return Object.create(null);
      if (!$gameTemp._kt_tk_guard) $gameTemp._kt_tk_guard = Object.create(null);
      return $gameTemp._kt_tk_guard;
    },
    targetKey(target) {
      if (!target) return "";
      if (U.isActor(target)) return "A" + U.actorId(target);
      if (U.isEnemy(target)) return U.enemyKey(target);
      return "";
    },
    passGuardEvent(target, eventKey) {
      const key = this.targetKey(target);
      if (!key) return true;
      const f = U.frameCount();
      const k = key + ":" + eventKey + ":F" + f;
      const m = this.guardMap();
      if (m[k]) return false;
      m[k] = true;
      return true;
    },
    pendingToast() {
      if (!$gameTemp) return [];
      if (!$gameTemp._kt_tk_pendingToast) $gameTemp._kt_tk_pendingToast = [];
      return $gameTemp._kt_tk_pendingToast;
    },
    pushToast(entry) {
      if (!entry || !entry.text) return;
      this.pendingToast().push(entry);
    },
    popToast() {
      const q = this.pendingToast();
      return q.length > 0 ? q.shift() : null;
    },
    drainAllToArray() {
      const q = this.pendingToast();
      if (!q || q.length === 0) return [];
      const out = q.slice(0);
      q.length = 0;
      return out;
    },

    tamaFrameMap() {
      if (!$gameTemp) return Object.create(null);
      if (!$gameTemp._kt_tk_tamaFrame) $gameTemp._kt_tk_tamaFrame = Object.create(null);
      return $gameTemp._kt_tk_tamaFrame;
    },
    markTamaThisFrame(target) {
      const k = this.targetKey(target);
      if (!k) return;
      this.tamaFrameMap()[k] = U.frameCount();
    },
    wasTamaThisFrame(target) {
      const k = this.targetKey(target);
      if (!k) return false;
      return this.tamaFrameMap()[k] === U.frameCount();
    },

    beginActionContext(subjectActorId) {
      if (!$gameTemp) return;
      if (!$gameTemp._kt_tk_actionStack) $gameTemp._kt_tk_actionStack = [];
      $gameTemp._kt_tk_actionStack.push(subjectActorId || 0);
    },
    endActionContext() {
      if (!$gameTemp || !$gameTemp._kt_tk_actionStack) return;
      $gameTemp._kt_tk_actionStack.pop();
    },
    inActionContext() {
      return !!($gameTemp && $gameTemp._kt_tk_actionStack && $gameTemp._kt_tk_actionStack.length > 0);
    },
    currentSubjectActorId() {
      const st = $gameTemp && $gameTemp._kt_tk_actionStack;
      return (st && st.length) ? (st[st.length - 1] || 0) : 0;
    }
  };

  // ------------------------------------------------------------
  // Toast Window（Window専用キューを持つ）
  // ------------------------------------------------------------
  class Window_KT_TamaToast extends Window_Base {
    initialize() {
      const rect = new Rectangle(0, 0, CFG.toastWidth, CFG.toastMinHeight);
      super.initialize(rect);
      this.openness = 0;

      this._speed = CFG.toastSpeed;
      this._remain = 0;
      this._entry = null;
      this._ktNeedClearOnClosed = false;

      this._queue = [];
    }

    setSpeed(n) { this._speed = U.clamp(n, 1, 64); }

    // Window標準の開閉更新に乗せつつ、速度だけ上書き
    updateOpen() {
      if (this._opening) {
        this.openness = Math.min(255, this.openness + this._speed);
        if (this.openness >= 255) this._opening = false;
      }
    }
    updateClose() {
      if (this._closing) {
        this.openness = Math.max(0, this.openness - this._speed);
        if (this.openness <= 0) this._closing = false;
      }
    }

    isBusy() {
      return !!(this._entry || this._remain > 0 || this.openness > 0 || this._opening || this._closing);
    }

    enqueue(entry) {
      if (!entry || !entry.text) return;
      this._queue.push(entry);
    }

    enqueueMany(list) {
      if (!Array.isArray(list) || list.length === 0) return;
      for (const e of list) if (e && e.text) this._queue.push(e);
    }

    _dequeue() {
      return this._queue.length > 0 ? this._queue.shift() : null;
    }

    open() { this._opening = true; this._closing = false; }
    close() { this._closing = true; this._opening = false; this._ktNeedClearOnClosed = true; }

    update() {
      super.update();

      if (this._ktNeedClearOnClosed && this.openness === 0 && !this._closing && !this._opening) {
        this._ktNeedClearOnClosed = false;
        this._entry = null;
        this._remain = 0;
        try { this.contents.clear(); } catch {}
      }

      if (this._entry && !this._closing) {
        this._remain--;
        if (this._remain <= 0) this.close();
      }

      if (!this._entry && this.openness === 0 && !this._opening && !this._closing) {
        let next = this._dequeue();
        if (!next) next = RT.popToast();
        if (next) this._startEntry(next);
      }
    }

    showEntry(entry) {
      if (!entry || !entry.text) return;
      if (this.isBusy()) this.enqueue(entry);
      else this._startEntry(entry);
    }

    _startEntry(entry) {
      this._entry = entry;
      this._remain = Math.max(1, entry.frames || CFG.toastFrames);

      this._resizeForText(entry);
      this._placeRandom();

      this.refresh();
      this.open();
    }

    _resolveRange() {
      const bw = Graphics.boxWidth;
      const bh = Graphics.boxHeight;

      const userXSpecified = !(CFG.xMin === 0 && CFG.xMax === 0);
      const userYSpecified = !(CFG.yMin === 0 && CFG.yMax === 0);

      let xMin, xMax, yMin, yMax;

      if (userXSpecified) {
        xMin = CFG.xMin;
        xMax = CFG.xMax;
      } else if (CFG.rightHalfDefault) {
        xMin = Math.floor(bw / 2);
        xMax = Math.max(xMin, bw - this.width);
      } else {
        xMin = 0;
        xMax = Math.max(0, bw - this.width);
      }

      if (userYSpecified) {
        yMin = CFG.yMin;
        yMax = CFG.yMax;
      } else {
        yMin = 0;
        yMax = Math.max(0, bh - this.height);
      }

      return { xMin, xMax, yMin, yMax, bw, bh };
    }

    _placeRandom() {
      const r = this._resolveRange();
      let x = U.randInt(r.xMin, r.xMax);
      let y = U.randInt(r.yMin, r.yMax);

      x = U.clamp(x, 0, Math.max(0, r.bw - this.width));
      y = U.clamp(y, 0, Math.max(0, r.bh - this.height));

      this.move(x, y, this.width, this.height);
    }

    _resizeForText(entry) {
      if (!CFG.autoFitHeight) {
        if (this.width !== CFG.toastWidth || this.height !== CFG.toastMinHeight) {
          this.move(this.x, this.y, CFG.toastWidth, CFG.toastMinHeight);
          this.createContents();
        }
        return;
      }

      const faceName = entry.faceName || "";
      const text = this.convertEscapeCharacters(String(entry.text));

      const newW = CFG.toastWidth;
      const pad = this.padding;
      const innerW = newW - pad * 2;

      const faceW = (faceName ? ImageManager.faceWidth : 0);
      const textW = Math.max(48, innerW - (faceName ? (faceW + this.itemPadding()) : 0));

      const lines = this._wrapLinesForMeasure(text, textW);
      const lineH = this.lineHeight();
      const textH = Math.max(1, lines.length) * lineH;

      const faceH = (faceName ? ImageManager.faceHeight : 0);
      const innerHNeed = Math.max(faceH, textH);

      let newH = innerHNeed + pad * 2;
      newH = U.clamp(newH, CFG.toastMinHeight, CFG.toastMaxHeight);

      if (this.width !== newW || this.height !== newH) {
        this.move(this.x, this.y, newW, newH);
        this.createContents();
      }
    }

    refresh() {
      this.contents.clear();
      if (!this._entry) return;

      const text = this.convertEscapeCharacters(String(this._entry.text));
      const faceName = this._entry.faceName || "";
      const faceIndex = U.clamp(U.num(this._entry.faceIndex, 0), 0, 7);

      const pad = this.itemPadding();
      const innerW = this.innerWidth;
      const innerH = this.innerHeight;

      let textX = 0;
      if (faceName) {
        try { ImageManager.loadFace(faceName); } catch {}
        this.drawFace(faceName, faceIndex, 0, 0, ImageManager.faceWidth, ImageManager.faceHeight);
        textX = ImageManager.faceWidth + pad;
      }

      const textW = innerW - textX;
      this._drawWrappedText(text, textX, 0, textW, innerH);
    }

    _drawWrappedText(text, x, y, w, h) {
      const lineH = this.lineHeight();
      const maxLines = Math.max(1, Math.floor(h / lineH));
      const lines = this._wrapLinesForDraw(text, w, maxLines);
      for (let i = 0; i < lines.length; i++) this.drawTextEx(lines[i], x, y + i * lineH, w);
    }

    _wrapLinesForMeasure(text, w) {
      const tokens = this._tokenizeForWrap(String(text));
      const lines = [];
      let cur = "";
      for (const token of tokens) {
        const next = cur ? (cur + token) : token;
        if (this._measureWidth(next) <= w) cur = next;
        else { if (cur) lines.push(cur); cur = token; }
      }
      if (cur) lines.push(cur);
      return lines.length > 0 ? lines : [String(text)];
    }

    _wrapLinesForDraw(text, w, maxLines) {
      const tokens = this._tokenizeForWrap(String(text));
      const lines = [];
      let cur = "";
      for (const t of tokens) {
        const next = cur ? (cur + t) : t;
        if (this._measureWidth(next) <= w) cur = next;
        else {
          if (cur) lines.push(cur);
          cur = t;
          if (lines.length >= maxLines) break;
        }
      }
      if (lines.length < maxLines && cur) lines.push(cur);
      return lines;
    }

    _measureWidth(s) {
      const str = String(s);
      let w = 0;
      const icons = str.match(/\x1bI\[\d+\]/gi);
      if (icons && icons.length) w += icons.length * (ImageManager.iconWidth || 32);
      w += this.textWidth(this._stripEscape(str));
      return w;
    }

    _stripEscape(s) {
      return String(s)
        .replace(/\x1bC\[\d+\]/gi, "")
        .replace(/\x1bI\[\d+\]/gi, "")
        .replace(/\x1bV\[\d+\]/gi, "")
        .replace(/\x1bN\[\d+\]/gi, "")
        .replace(/\x1bG/gi, "")
        .replace(/\x1b\{/g, "")
        .replace(/\x1b\}/g, "")
        .replace(/\x1b/g, "");
    }

    _tokenizeForWrap(text) {
      const s = String(text);
      const re = /\x1b[A-Z](?:\[\d+\])?|\x1bG|\x1b\{|\x1b\}|\s+|./g;
      return s.match(re) || [];
    }
  }

  // ------------------------------------------------------------
  // Scene & Battle hooks（reset timing / endBattle flush）
  // ------------------------------------------------------------
  const _BattleManager_startBattle = BattleManager.startBattle;
  BattleManager.startBattle = function() {
    RT.reset();
    _BattleManager_startBattle.apply(this, arguments);

    if (CFG.preloadActorFaces) {
      try {
        const ms = $gameParty && $gameParty.members ? $gameParty.members() : [];
        for (const a of ms) if (a && a.faceName && a.faceName()) ImageManager.loadFace(a.faceName());
      } catch {}
      try {
        for (const k of Object.keys(CFG.actorFaceOverrideMap)) {
          const ov = CFG.actorFaceOverrideMap[k];
          if (ov && ov.faceName) ImageManager.loadFace(ov.faceName);
        }
      } catch {}
    }
  };

  const _BattleManager_endBattle = BattleManager.endBattle;
  BattleManager.endBattle = function(result) {
    if (CFG.flushToastOnEndBattle) {
      try {
        const sc = SceneManager._scene;
        const toastW = sc && sc._ktTamaToastWindow;
        if (toastW) {
          const all = RT.drainAllToArray();
          if (all.length > 0) toastW.enqueueMany(all);
          if (!toastW.isBusy()) {
            const e = toastW._dequeue ? toastW._dequeue() : null;
            if (e) toastW.showEntry(e);
          }
        }
      } catch (e) {
        dlog("endBattle flush error", e);
      }
    }
    return _BattleManager_endBattle.apply(this, arguments);
  };

  if (typeof Scene_Battle !== "undefined") {
    const _Scene_Battle_createAllWindows = Scene_Battle.prototype.createAllWindows;
    Scene_Battle.prototype.createAllWindows = function() {
      _Scene_Battle_createAllWindows.apply(this, arguments);

      this._ktTamaToastWindow = new Window_KT_TamaToast();
      this._ktTamaToastWindow.setSpeed(CFG.toastSpeed);
      this.addWindow(this._ktTamaToastWindow);

      try {
        const all = RT.drainAllToArray();
        if (all.length > 0) this._ktTamaToastWindow.enqueueMany(all);
      } catch {}

      const first = this._ktTamaToastWindow._dequeue();
      if (first) this._ktTamaToastWindow.showEntry(first);
    };

    const _Scene_Battle_terminate = Scene_Battle.prototype.terminate;
    Scene_Battle.prototype.terminate = function() {
      _Scene_Battle_terminate.apply(this, arguments);
      try { RT.reset(); } catch {}
    };
  }

  // ------------------------------------------------------------
  // Output
  // ------------------------------------------------------------
  function formatText(raw, target) {
    const tname = U.battlerName(target);
    return String(raw || "").replace(/{target}/g, tname);
  }

  function resolveActorFaceById(actorId) {
    const aid = Number(actorId || 0);
    if (aid <= 0) return { faceName: "", faceIndex: 0 };

    const ov = CFG.actorFaceOverrideMap[aid];
    if (ov && ov.faceName) return { faceName: ov.faceName, faceIndex: U.clamp(ov.faceIndex, 0, 7) };

    try {
      const a = $gameActors && $gameActors.actor(aid);
      if (a && a.faceName && a.faceName()) {
        return { faceName: a.faceName(), faceIndex: U.clamp(a.faceIndex(), 0, 7) };
      }
    } catch {}
    return { faceName: "", faceIndex: 0 };
  }

  function enqueueToast(entry) {
    try {
      const sc = SceneManager._scene;
      if (sc && sc._ktTamaToastWindow) sc._ktTamaToastWindow.showEntry(entry);
      else RT.pushToast(entry);
    } catch {
      RT.pushToast(entry);
    }
  }

  function pushToast(text, face) {
    if (!text) return;
    const entry = {
      text: String(text),
      faceName: face && face.faceName ? String(face.faceName) : "",
      faceIndex: face && Number.isFinite(face.faceIndex) ? face.faceIndex : 0,
      frames: CFG.toastFrames,
    };
    enqueueToast(entry);
  }

  // ------------------------------------------------------------
  // Message selection + face selection
  // ------------------------------------------------------------
  function pickEnemyMessage(target, inflictorActorId) {
    if (inflictorActorId && CFG.enemyActorMessageMap[inflictorActorId]) {
      return formatText(U.pickRandom(CFG.enemyActorMessageMap[inflictorActorId]), target);
    }
    if (CFG.enemyMessages.length === 0) return "";
    return formatText(U.pickRandom(CFG.enemyMessages), target);
  }

  function pickActorMessage(target) {
    const aid = U.actorId(target);
    const list = CFG.actorMessageMap[aid] || CFG.actorDefaultMessages;
    if (!list || list.length === 0) return "";
    return formatText(U.pickRandom(list), target);
  }

  function showTamaToast(target, reason, inflictorActorId) {
    if (!U.inBattle()) return;
    if (!target) return;

    if (!RT.passGuardEvent(target, "TAMA_TOAST")) {
      dlog("Guarded", { reason, target: U.battlerName(target), inflictorActorId: inflictorActorId || 0 });
      return;
    }

    let msg = "";
    let face = { faceName: "", faceIndex: 0 };

    if (U.isEnemy(target)) {
      msg = pickEnemyMessage(target, inflictorActorId || 0);
      if (inflictorActorId) face = resolveActorFaceById(inflictorActorId);
    } else if (U.isActor(target)) {
      msg = pickActorMessage(target);
      face = resolveActorFaceById(U.actorId(target));
    }

    if (msg) pushToast(msg, face);
    dlog("Shown", { reason, target: U.battlerName(target), inflictorActorId: inflictorActorId || 0, msg });
  }

  // ------------------------------------------------------------
  // Hooks
  // ------------------------------------------------------------
  const _Game_BattlerBase_addNewState = Game_BattlerBase.prototype.addNewState;
  Game_BattlerBase.prototype.addNewState = function(stateId) {
    _Game_BattlerBase_addNewState.apply(this, arguments);

    if (!U.inBattle()) return;

    if (stateId === CFG.stateId) {
      if (CFG.enableDeathSameFrame) RT.markTamaThisFrame(this);

      // 敵は apply 側で「付与者(味方アクター)」を付けて出すので、行動中はここで出さない
      if (U.isEnemy(this) && RT.inActionContext()) return;

      showTamaToast(this, "tamaApplied(addNewState)", 0);
    }

    if (CFG.enableDeathSameFrame && stateId === CFG.deathStateId) {
      if (RT.wasTamaThisFrame(this)) {
        if (U.isEnemy(this) && RT.inActionContext()) return;
        showTamaToast(this, "deathSameFrame(addNewState)", 0);
      }
    }
  };

  const _Game_Battler_addState = Game_Battler.prototype.addState;
  Game_Battler.prototype.addState = function(stateId) {
    const had = this.isStateAffected(stateId);
    _Game_Battler_addState.apply(this, arguments);

    if (!U.inBattle()) return;

    // 新規付与のみ（かけ直しの表示は enemy は apply 側で行う）
    if (had) return;
    if (!this.isStateAffected(stateId)) return;

    if (stateId === CFG.stateId) {
      if (CFG.enableDeathSameFrame) RT.markTamaThisFrame(this);
      if (U.isEnemy(this) && RT.inActionContext()) return;
      showTamaToast(this, "tamaApplied(addState)", 0);
    }

    if (CFG.enableDeathSameFrame && stateId === CFG.deathStateId) {
      if (RT.wasTamaThisFrame(this)) {
        if (U.isEnemy(this) && RT.inActionContext()) return;
        showTamaToast(this, "deathSameFrame(addState)", 0);
      }
    }
  };

  function actionHasAddStateEffect(action, stateId) {
    try {
      const item = action && action.item ? action.item() : null;
      const effects = item && item.effects ? item.effects : null;
      if (!effects) return false;
      for (let i = 0; i < effects.length; i++) {
        const ef = effects[i];
        if (ef && ef.code === Game_Action.EFFECT_ADD_STATE && ef.dataId === stateId) return true;
      }
    } catch {}
    return false;
  }

  function subjectActorIdFromAction(action) {
    try {
      const s = action && action.subject ? action.subject() : null;
      if (s && s.isActor && s.isActor()) return s.actorId();
    } catch {}
    return 0;
  }

  const _Game_Action_apply = Game_Action.prototype.apply;
  Game_Action.prototype.apply = function(target) {
    const enabled = U.inBattle() && target;

    let hadDeath = false;
    let hadTama = false;
    let hasTamaEffect = false;
    let subjectAid = 0;

    if (enabled) {
      try {
        hadDeath = target.isStateAffected(CFG.deathStateId);
        hadTama = target.isStateAffected(CFG.stateId);
        hasTamaEffect = actionHasAddStateEffect(this, CFG.stateId);
        subjectAid = subjectActorIdFromAction(this);
      } catch {}
    }

    if (enabled) RT.beginActionContext(subjectAid);
    try {
      _Game_Action_apply.apply(this, arguments);
    } finally {
      if (enabled) RT.endActionContext();
    }

    if (!enabled) return;

    let nowDeath = false;
    let nowTama = false;
    try {
      nowDeath = target.isStateAffected(CFG.deathStateId);
      nowTama = target.isStateAffected(CFG.stateId);
    } catch {}

    if (!U.isEnemy(target)) return;

    // (2) タマ強打中の敵に、タマ強打付与行動を取ったとき（付与済みでも表示）
    if (hadTama && hasTamaEffect) {
      showTamaToast(target, "enemyTamaReapplyAction(apply)", subjectAid);
      return;
    }

    // (1) 新規付与：付与したとき
    if (!hadTama && nowTama) {
      if (CFG.enableDeathSameFrame) RT.markTamaThisFrame(target);
      showTamaToast(target, "enemyTamaApplied(apply)", subjectAid);
      return;
    }

    // 付与が死亡でブロックされた（付与効果があるのに付かず、そのまま死亡した）
    if (CFG.enableDeathSameFrame && hasTamaEffect && !hadDeath && nowDeath && !hadTama && !nowTama) {
      showTamaToast(target, "enemyTamaBlockedByDeath(apply)", subjectAid);
      return;
    }
  };

  RT.reset();
})();
