// AWY_TransferEx.js Ver.1.0.0
// MIT License (C) 2025 あわやまたな
// http://opensource.org/licenses/mit-license.php

/*:
* @target MZ
* @plugindesc Extends the player transfer event.
* @author あわやまたな (Awaya_Matana)
* @url https://awaya3ji.seesaa.net/article/510685793.html
* @help Ver.1.0.0
* [Comment]
* Place a comment on the first line of the event contents and enter it in the following format:
* <transferEx> //The location of the transfer changes depending on the event and the player's location.
* <transferEx:exWidth,exHeight> //Extend the collision. The transfer location will change.
* <transferEx:exWidth,exHeight,false> //Extends the collision, but leaves the transfer location unchanged.
*
* exWidth: An integer greater than or equal to 0 that extends the collision to the right.
* exHeight: An integer greater than or equal to 0 that extends the collision downwards.
*
* [How to Use]
* By placing the above comment in the event, only one transfer event is required.
* The location of the event command [Transfer Player] can be changed depending on the player's position.
*
* @param shiftDirection
* @text Shift Direction
* @desc Behavior when using <transferEx:exWidth,exHeight>.
* @type select
* @option Extended Direction Only
* @value 0
* @option Both
* @value 1
* @default 0
*
*/

/*:ja
* @target MZ
* @plugindesc 場所移動イベントを拡張します。
* @author あわやまたな (Awaya_Matana)
* @url https://awaya3ji.seesaa.net/article/510685793.html
* @help 
* [注釈]
* イベントの実行内容一行目に注釈を配置し、以下の形式で入力すると反映されます。
* <transferEx> //場所移動の座標がイベントとプレイヤーの位置関係で変化します。
* <transferEx:延長幅,延長高さ> //衝突判定を延長します。場所移動の座標が変化します。
* <transferEx:延長幅,延長高さ,false> //衝突判定を延長しますが、場所移動の座標はそのままです。
*
* 延長幅：衝突判定を右方向に延長する0以上の整数です。
* 延長高さ：衝突判定を下方向に延長する0以上の整数です。
*
* [使い方]
* イベントに上記の注釈を配置する事で、場所移動イベントが一つで済むようになります。
* イベントコマンド [場所移動] の座標をプレイヤーの位置に応じて変化させることができます。
*
* [更新履歴]
* 2025/02/24：Ver.1.0.0　公開。
*
* @param shiftDirection
* @text シフト方向
* @desc <transferEx:延長幅,延長高さ>を使用した時の挙動。
* @type select
* @option 延長した方向のみ
* @value 0
* @option 両方
* @value 1
* @default 0
*
*/

var AWY = AWY || {};
'use strict';
{
	const pluginName = document.currentScript.src.match(/^.*\/(.*).js$/)[1];
	const parameters = PluginManager.parameters(pluginName);
	const shiftDirection = Number(parameters.shiftDirection || 0);

	//-----------------------------------------------------------------------------
	// Game_Interpreter

	const _Game_Interpreter_command201 = Game_Interpreter.prototype.command201;
	Game_Interpreter.prototype.command201 = function(params) {
		const character = this.character(0);
		if (character && character.transferExShift()) {
			params = params.clone();
			if (params[0] === 1) {
				params[0] = 0;
				params[1] = $gameVariables.value(params[1]);
				params[2] = $gameVariables.value(params[2]);
				params[3] = $gameVariables.value(params[3]);
			}
			const forceShift = shiftDirection > 0 || !character.isTransferExCollision();
			if (forceShift || character.transferCollisionWidth()) {
				params[2] += $gamePlayer.deltaXFrom(character.x);
			}
			if (forceShift || character.transferCollisionHeight()) {
				params[3] += $gamePlayer.deltaYFrom(character.y);
			}
		}
		return _Game_Interpreter_command201.call(this, params);
	};

	//-----------------------------------------------------------------------------
	// Game_Player

	const _Game_Player_performTransfer = Game_Player.prototype.performTransfer;
	Game_Player.prototype.performTransfer = function() {
		if (this.isTransferring()) {
			this._newX = $gameMap.roundX(this._newX);
			this._newY = $gameMap.roundY(this._newY);
		}
		_Game_Player_performTransfer.call(this);
	};

	//-----------------------------------------------------------------------------
	// Game_Event

	if (Game_CharacterBase.prototype.pos === Game_Event.prototype.pos) {
		Game_Event.prototype.pos = function(x, y) {
			return Game_CharacterBase.prototype.pos.apply(this, arguments);
		};
	}

	const _Game_Event_pos = Game_Event.prototype.pos;
	Game_Event.prototype.pos = function(x, y) {
		return this.isTransferExCollision() ? this.posTransferEx(x, y) : _Game_Event_pos.apply(this, arguments);
	};

	Game_Event.prototype.posTransferEx = function(x, y) {
		const endX = this._x + this._transferExWidth;
		const endY = this._y + this._transferExHeight;
		let posX = this._x <= x && x <= endX;
		let posY = this._y <= y && y <= endY;
		if ($gameMap.isLoopHorizontal() && !posX && $gameMap.width() <= endX) {
			posX = x <= $gameMap.roundX(endX);
		}
		if ($gameMap.isLoopVertical() && !posY && $gameMap.height() <= endY) {
			posY = y <= $gameMap.roundX(endY);
		}
		return posX && posY;
	};

	Game_Event.prototype.isTransferExCollision = function() {
		return !!(this._transferExWidth || this._transferExHeight);
	};

	Game_Event.prototype.transferExShift = function() {
		return this._transferExShift;
	};

	Game_Event.prototype.transferCollisionWidth = function() {
		return this._transferExWidth || 0;
	};

	Game_Event.prototype.transferCollisionHeight = function() {
		return this._transferExHeight || 0;
	};

	const _Game_Event_clearPageSettings = Game_Event.prototype.clearPageSettings;
	Game_Event.prototype.clearPageSettings = function() {
		_Game_Event_clearPageSettings.call(this);
		this._transferExWidth = 0;
		this._transferExHeight = 0;
		this._transferExShift = false;
	};

	const _Game_Event_setupPageSettings = Game_Event.prototype.setupPageSettings;
	Game_Event.prototype.setupPageSettings = function() {
		_Game_Event_setupPageSettings.call(this);
		const transferEx = this.page().transferEx || {};
		this._transferExWidth = transferEx.width || 0;
		this._transferExHeight = transferEx.height || 0;
		this._transferExShift = transferEx.shift || false;
	};

	//-----------------------------------------------------------------------------
	// DataManager
	//マップデータロード時の処理に追加

	if (!AWY.addPageSettings) {
		AWY._pageSettingHandlers = [];
		const _DataManager_onLoad = DataManager.onLoad;
		DataManager.onLoad = function(object) {
			_DataManager_onLoad.apply(this, arguments);
			AWY.addPageSettings(object);
		};

		AWY.addPageSettings = function(object) {
			if (DataManager.isMapObject(object) && Array.isArray(object.events)) {
				for (const event of object.events) {
					if (event && event.pages){
						extractMetadata(event);
					}
				}
			}
		};

		function extractMetadata(data) {
			for (const page of data.pages) {
				const comment = findComment(page);
				const commentData = {"note": comment};
				DataManager.extractMetadata(commentData);
				AWY._pageSettingHandlers.forEach(handler => handler(page, commentData.meta));
			}
		}

		function findComment(page) {
			const list = page.list;
			if (!list[0] && list[0].code !== 108) {
				return "";
			}
			let comment = list[0].parameters[0];
			for (let i = 1; list[i] && list[i].code === 408; i++) {
				comment += list[i].parameters[0];
			}
			return comment;
		}
	}

	AWY._pageSettingHandlers.push(
		function(page, meta) {
			if (!meta.transferEx) return;
			let w, h, ex;
			if (meta.transferEx !== true && meta.transferEx) {
				const data = (meta.transferEx || "").split(",");
				[w, h, ex] = data;
			}
			const obj = page.transferEx = {};
			obj.width = Number(w || 0);
			obj.height = Number(h || 0);
			obj.shift = ex !== "false";
		}
	);

}