//=============================================================================
// CoinExchangeScene.js
//=============================================================================

/*:ja
 * @plugindesc ver1.01 ギルドコイン交換シーンとそのアイテムの設定
 * @author まっつＵＰ
 * 
 * @param CoinCategory
 * @desc コインの階級分けです。
 * ウインドウに表示される他、ノートタグの値に使います。
 * @type string[]
 * @default ["銅","銀","金","白金"]
 * 
 * @param CoinIds
 * @desc 交換対象のコインのそれぞれのId
 * （データベースの「アイテム」のIdです。）
 * @type item[]
 * @min 1
 * @max 10000
 * @default ["50","51","52","53"]
 * 
 * @param defaultExchangenumber
 * @desc ノートタグで交換数が指定されない場合の
 * デフォルトの交換数
 * @type number
 * @min 1
 * @max 1000
 * @default 5
 * 
 * @param coinWincurbackcolor
 * @desc 現在対象のコイン文字列の背景の矩形の色
 * 0~31(img/system/Window.png参照)
 * @type number
 * @min 0
 * @max 31
 * @default 19
 * 
 * @param yamerutext
 * @desc ギルドコイン交換シーンのウインドウに表示される
 * キャンセル項目（やめる）の文字列（省略はデフォルト用語通り）
 * @default
 * 
 * @param Coinhelptext
 * @desc コインの階級を選ぶときに説明文に表示します
 * \CESで改行しています。
 * @default \C[1]交換したいコインの種類と対象のアイテムを交換して\CESコインを手に入れましょう。\CES\C[0]交換に必要な個数はアイテムごとに異なります。
 * 
 * @param coinWintitle
 * @desc コインウインドウの一行目
 * @default 現在の各コイン枚数
 * 
 *
 * @help
 * 
 * このヘルプとパラメータの説明をよくお読みになってからお使いください。
 * 
 * ---導入---
 * 
 * CoinExchangeScene.jsは
 * プラグイン管理中の任意の位置に導入してください。
 * 
 * 
 * ---ノートタグ---
 * 
 * アイテム・武器・防具のノートタグ
 * (:の前後にはスペース等を入れないでください。)
 * 
 * <CESCoinCategory:x>
 * 例:<CESCoinCategory:銅>
 * 例:<CESCoinCategory:金>
 * 
 * xにはプラグインパラメータで設定した
 * コインの階級を入れてください。
 * これはそのコインと交換できるアイテムにする必須の設定です。
 * 
 * 
 * <CESExchangenumber:x>
 * 例:<CESExchangenumber:5>
 * 
 * xには自然数を入れてください。
 * 指定のコイン１つと交換するのに必要なこのアイテムの個数です。
 * 指定されない場合は代わりに
 * プラグインパラメータで設定したデフォルト値を利用します。
 * 
 * 
 * ---プラグインコマンド---
 * 
 * ModeCoinExchange true
 * 
 * これ以降のイベントコマンドの「ショップの処理」のリストを
 * 交換可能なアイテムのリストとして扱います。
 * そのイベントコマンドの「ショップの処理」を実行次第シーン遷移します。
 * （ただし、このコマンドの設定はセーブデータには保存しません。）
 * ショップシーンに遷移するような挙動を見せますが
 * 実際には別の、専用のシーンに遷移します。
 * 
 * 
 * ModeCoinExchange false
 * 
 * コイン交換モードフラグをfalseにします。
 * 即ち、普通の状態に戻します。
 * 
 * 
 * ---交換の仕様---
 * 
 * シーン中のアイテムウインドウ中では
 * 対象のアイテムが交換可能な状態のみ選択することができます。
 * アイテムが足りない場合（ノートタグ設定の個数に満たない場合）の他、
 * コインが既に最大個数の場合などは選択できません。
 * 
 *  
 * 免責事項：
 * このプラグインを利用したことによるいかなる損害も制作者は一切の責任を負いません。
 * 
 */

var Imported = Imported || {};
Imported.CoinExchangeScene = true;

(function() {

const CESisZeroOrValid = function(value) {
    return !!value || value === 0;
};


//プラグインパラメータ等で使う関数はこれより上に
const parameters = PluginManager.parameters('CoinExchangeScene');

//配列系
const CESCoinCategory = JSON.parse(parameters['CoinCategory']);
const CESCoinIds = JSON.parse(parameters['CoinIds']).map(function(CEStonum) {
    return Number(CEStonum || 0);
});

//数値系
const CESdefaultExchangenumber = Number(parameters['defaultExchangenumber'] || 1);
const CEScoinWincurbackcolor = Number(parameters['coinWincurbackcolor'] || 0);

//任意文字列系
const CESyamerutext = String(parameters['yamerutext'] || '');
const CEScoinWintitle = String(parameters['coinWintitle'] || '');
const CESCoinhelptext = String(parameters['Coinhelptext'] || '');


const _Game_Interpreter_pluginCommand = Game_Interpreter.prototype.pluginCommand;
Game_Interpreter.prototype.pluginCommand = function(command, args) {
    _Game_Interpreter_pluginCommand.call(this, command, args);
    switch(command){

        case 'ModeCoinExchange':
            $gameTemp.CESsetmode(args[0] === 'true');
        break;

    }
};

//イベントコマンドのショップの処理
const _Game_Interpreter_command302 = Game_Interpreter.prototype.command302;
Game_Interpreter.prototype.command302 = function() {
    if(!$gameTemp.CESmode()){
        return _Game_Interpreter_command302.call(this);
    }
    if (!$gameParty.inBattle()) {
        var goods = [this._params];
        while (this.nextEventCode() === 605) {
            this._index++;
            goods.push(this.currentCommand().parameters);
        }
        SceneManager.push(Scene_CESexchange);
        SceneManager.prepareNextScene(goods, this._params[4]);
    }
    return true;
};

DataManager.CESnote = function(data, text) {
    if(!data || !text) return null;
    return data.meta[text];
};

DataManager.CESiscurrentCategory = function(item) {
    const note = this.CESnote(item, 'CESCoinCategory');
    return !!note && note === this.CEScointagname();
};

DataManager.CESExchangenumber = function(item) {
    const note = Number(this.CESnote(item, 'CESExchangenumber') || 0);
    if(note > 0){
        return note;
    }else{
        return CESdefaultExchangenumber;
    }
};

DataManager.CESyamerutext = function() {
    if(CESyamerutext){
        return CESyamerutext;
    }else{
        return TextManager.cancel;
    }
};

DataManager.CEScointagname = function(index) {
    if(!CESisZeroOrValid(index)){
        index = SceneManager.CEScoinindex();
    }
    return CESCoinCategory[index];
};

DataManager.CEScoindataItem = function(index) {
    if(!CESisZeroOrValid(index)){
        index = SceneManager.CEScoinindex();
    }
    const id = CESCoinIds[index];
    return $dataItems[id];
};

//シーンを初めて、最初に選択したコインで処理する。
SceneManager.CEScoinindex = function() {
    if(this._scene._commandWindow){
        return this._scene._commandWindow.index();
    }
    return -1;
};

const _Game_Temp_initialize = Game_Temp.prototype.initialize;
Game_Temp.prototype.initialize = function() {
    _Game_Temp_initialize.call(this);
    this._CESmode = false;
};

Game_Temp.prototype.CESmode = function() {
    return this._CESmode;
};

Game_Temp.prototype.CESsetmode = function(enable) {
    this._CESmode = !!enable;
};


//新規シーン

function Scene_CESexchange() {
    this.initialize.apply(this, arguments);
}

Scene_CESexchange.prototype = Object.create(Scene_MenuBase.prototype);
Scene_CESexchange.prototype.constructor = Scene_CESexchange;

Scene_CESexchange.prototype.initialize = function() {
    Scene_MenuBase.prototype.initialize.call(this);
};

//ショップの処理のリスト反映
Scene_CESexchange.prototype.prepare = function(goods, purchaseOnly) {
    this._goods = goods;
    this._item = null;
};

Scene_CESexchange.prototype.create = function() {
    Scene_MenuBase.prototype.create.call(this);
    this.createHelpWindow();
    this.createCategoryWindow();
    this.createItemWindow();
    this.createGoldWindow();
    this.createNumberWindow();
};

Scene_CESexchange.prototype.createHelpWindow = function() {
    this._helpWindow = new Window_Help(3);
    this.addWindow(this._helpWindow);
};

//xはウインドウ側で調整する。
Scene_CESexchange.prototype.createCategoryWindow = function() {
    this._commandWindow = new Window_CESCoinCategoryH();
    this._commandWindow.y = this._helpWindow.height;
    CESCoinCategory.forEach(function(str) {
        this._commandWindow.setHandler(str, this.onCategoryOk.bind(this));
    }, this);
    this._commandWindow.setHandler('cancel', this.popScene.bind(this));
    this._commandWindow.setHelpWindow(this._helpWindow);
    this.addWindow(this._commandWindow);
};

Scene_CESexchange.prototype.createItemWindow = function() {
    const comline = this._commandWindow.y + this._commandWindow.height;
    const wh = Graphics.boxHeight - comline;
    this._itemWindow = new Window_CESItemlist(wh, this._goods);
    this._itemWindow.y = comline;
    this._itemWindow.setHelpWindow(this._helpWindow);
    this._itemWindow.setHandler('ok',     this.onItemOk.bind(this));
    this._itemWindow.setHandler('cancel', this.onItemCancel.bind(this));
    this.addWindow(this._itemWindow);
    this._commandWindow.setItemWindow(this._itemWindow);
};

//常にempwidth > this._goldWindow.widthであること。
Scene_CESexchange.prototype.createGoldWindow = function() {
    this._goldWindow = new Window_CESCurrentCoin();
    this.addWindow(this._goldWindow);
    const xline = this._itemWindow.x + this._itemWindow.width;
    const empwidth = Graphics.boxWidth - xline;
    this._goldWindow.x = xline + (empwidth - this._goldWindow.width) / 2;
    this._goldWindow.y = this._commandWindow.y + this._commandWindow.height + this._goldWindow.lineHeight();
    this._commandWindow.setGoldWindow(this._goldWindow);
};

Scene_CESexchange.prototype.createNumberWindow = function() {
    const wx = this._itemWindow.x;
    const wy = this._itemWindow.y;
    const ww = this._itemWindow.width;
    const wh = this._itemWindow.height;
    this._numberWindow = new Window_CESInputNumber(wx, wy, ww, wh);
    this._numberWindow.setHandler('ok',     this.onNumberOk.bind(this));
    this._numberWindow.setHandler('cancel', this.onNumberCancel.bind(this));
    this.addWindow(this._numberWindow);
};

Scene_CESexchange.prototype.onCategoryOk = function() {
    this._commandWindow.deactivate();
    this._itemWindow.activate();
    this._itemWindow.select(0);
};

Scene_CESexchange.prototype.onItemOk = function() {
    this._item = this._itemWindow.item();
    this._numberWindow.setup(this._item);
    this._itemWindow.hide();
    this._itemWindow.deactivate();
    this._numberWindow.show();
    this._numberWindow.activate();
};

Scene_CESexchange.prototype.onItemCancel = function() {
    this._itemWindow.deactivate();
    this._itemWindow.deselect();
    this._commandWindow.activate();
    this._commandWindow.reselect();
};

Scene_CESexchange.prototype.endInputNumber = function() {
    this._numberWindow.hide();
    this._numberWindow.deactivate();
    this._goldWindow.refresh();
    this._itemWindow.refresh();
    this._itemWindow.show();
    this._itemWindow.activate();
};

Scene_CESexchange.prototype.onNumberOk = function() {
    SoundManager.playShop();
    this.doExchange(this._numberWindow._number, this._numberWindow.calcTotal());
    this.endInputNumber();
};

Scene_CESexchange.prototype.doExchange = function(losenumber, gainnumber) {
    const loseitem = this._item;
    const gainitem = DataManager.CEScoindataItem();
    $gameParty.loseItem(loseitem, losenumber);
    $gameParty.gainItem(gainitem, gainnumber);
};

Scene_CESexchange.prototype.onNumberCancel = function() {
    SoundManager.playCancel();
    this.endInputNumber();
};


//マスク無効
WindowLayer.prototype._maskWindow = function(window, shift) {
    
};

//drawtextexで書く文字列の幅を返す。
Window_Base.prototype.textWidthForEx = function(text) {
    return this.drawTextEx(text, 0, this.contents.height + this.lineHeight());
};

Window_Base.prototype.textWidthmaxItem = function(item) {
    var num = '00';
    if(item){
        num = $gameParty.maxItems(item);
    }
    return this.textWidth(String(num));
};

//新しい改行コード
const _Window_Base_processEscapeCharacter = Window_Base.prototype.processEscapeCharacter;
Window_Base.prototype.processEscapeCharacter = function(code, textState) {
    switch (code) {
        case 'CES':
            textState.x = textState.left;
            textState.y += textState.height;
            textState.height = this.calcTextHeight(textState, false);
        break;
    
        default:
            _Window_Base_processEscapeCharacter.call(this, code, textState);
        break;
    }
};

//新規ウインドウ

function Window_CESCoinCategoryH() {
    this.initialize.apply(this, arguments);
}

Window_CESCoinCategoryH.prototype = Object.create(Window_HorzCommand.prototype);
Window_CESCoinCategoryH.prototype.constructor = Window_CESCoinCategoryH;

Window_CESCoinCategoryH.prototype.initialize = function() {
    Window_HorzCommand.prototype.initialize.call(this, 0, 0);
    this.x = (Graphics.boxWidth - this.width) / 2;
};

Window_CESCoinCategoryH.prototype.windowWidth = function() {
    return Graphics.boxWidth;
};

Window_CESCoinCategoryH.prototype.maxCols = function() {
    return CESCoinCategory.length + 1;
};

Window_CESCoinCategoryH.prototype.makeCommandList = function() {
    CESCoinCategory.forEach(function(str) {
        this.addCommand(str, str);
    }, this);
    this.addCommand(DataManager.CESyamerutext(), 'cancel');
};

Window_CESCoinCategoryH.prototype.setItemWindow = function(win) {
    this._itemWindow = win;
    this.callUpdateHelp();
};

Window_CESCoinCategoryH.prototype.setGoldWindow = function(win) {
    this._goldWindow = win;
    this.callUpdateHelp();
};

Window_CESCoinCategoryH.prototype.updateHelp = function() {
    this._helpWindow.setText(CESCoinhelptext);
    if (this._itemWindow) {
        this._itemWindow.refresh();
    }
    if (this._goldWindow) {
        this._goldWindow.refresh();
    }
};


function Window_CESItemlist() {
    this.initialize.apply(this, arguments);
}

Window_CESItemlist.prototype = Object.create(Window_Selectable.prototype);
Window_CESItemlist.prototype.constructor = Window_CESItemlist;

Window_CESItemlist.prototype.initialize = function(height, shopGoods) {
    var width = this.windowWidth();
    Window_Selectable.prototype.initialize.call(this, 0, 0, width, height);
    this._shopGoods = shopGoods;
    this.refresh();
    this.select(0);
};

Window_CESItemlist.prototype.windowWidth = function() {
    return 540;
};

Window_CESItemlist.prototype.maxItems = function() {
    return this._data ? this._data.length : 1;
};

Window_CESItemlist.prototype.item = function() {
    return this._data[this.index()];
};

Window_CESItemlist.prototype.Exchangenumber = function(item) {
    return DataManager.CESExchangenumber(item);
};

Window_CESItemlist.prototype.isCurrentItemEnabled = function() {
    return this.isEnabled(this.item());
};

Window_CESItemlist.prototype.isEnabled = function(item) {
    const coin = DataManager.CEScoindataItem();
    if(!(item && coin)) return false;
    const numitem = $gameParty.numItems(item);
    const numex = this.Exchangenumber(item);
    const ismax = ($gameParty.maxItems(coin) <= $gameParty.numItems(coin));
    return numex > 0 && numitem >= numex && !ismax;
};

Window_CESItemlist.prototype.refresh = function() {
    this.makeItemList();
    this.createContents();
    this.drawAllItems();
};

Window_CESItemlist.prototype.makeItemList = function() {
    this._data = [];
    this._shopGoods.forEach(function(goods) {
        var item = null;
        switch (goods[0]) {
            case 0:
                item = $dataItems[goods[1]];
            break;
            case 1:
                item = $dataWeapons[goods[1]];
            break;
            case 2:
                item = $dataArmors[goods[1]];
            break;
        }
        if (DataManager.CESiscurrentCategory(item)) {
            this._data.push(item);
        }
    }, this);
};

Window_CESItemlist.prototype.drawItem = function(index) {
    const item = this._data[index];
    if(!item) return;
    const pad = this.textPadding();
    const rect = this.itemRect(index);
    rect.width -= pad;
    const miw = this.textWidthmaxItem(item);
    const slw = this.textWidth(' / ');
    this.changePaintOpacity(this.isEnabled(item));
    this.drawItemName(item, rect.x + pad, rect.y, rect.width - 168);
    this.drawText($gameParty.numItems(item), rect.x, rect.y, rect.width - miw - slw, 'right');
    this.drawText(' / ', rect.x, rect.y, rect.width - miw, 'right');
    this.drawText(this.Exchangenumber(item), rect.x, rect.y, rect.width, 'right');
    this.changePaintOpacity(true);
};

Window_CESItemlist.prototype.updateHelp = function() {
    this.setHelpWindowItem(this.item());
};


function Window_CESCurrentCoin() {
    this.initialize.apply(this, arguments);
}

Window_CESCurrentCoin.prototype = Object.create(Window_Base.prototype);
Window_CESCurrentCoin.prototype.constructor = Window_CESCurrentCoin;

Window_CESCurrentCoin.prototype.initialize = function() {
    var width = this.windowWidth();
    var height = this.windowHeight();
    Window_Base.prototype.initialize.call(this, 0, 0, width, height);
    this.refresh();
};

Window_CESCurrentCoin.prototype.windowWidth = function() {
    return 400;
};

Window_CESCurrentCoin.prototype.windowHeight = function() {
    return this.fittingHeight(CESCoinCategory.length + 1);
};

Window_CESCurrentCoin.prototype.translucentOpacity = function() {
    return 192;
};

Window_CESCurrentCoin.prototype.refresh = function() {
    const x = this.textPadding();
    var y = 0;
    const line = this.lineHeight();
    const width = this.contentsWidth() - x * 2;
    const tw = this.textWidthForEx(CEScoinWintitle);
    const wx = x + (width - tw) / 2;
    this.contents.clear();

    this.drawTextEx(CEScoinWintitle, wx, y, width);
    this.changePaintOpacity(false);
    this.contents.fillRect(0, line - 2, this.contentsWidth(), 2, this.normalColor());
    this.changePaintOpacity(true);

    CESCoinCategory.forEach(function(str, index) {
        y += line;
        if(index === SceneManager.CEScoinindex()){
            this.changePaintOpacity(false);
            this.contents.fillRect(1, y + 1, this.contentsWidth() - 2, line - 2, this.textColor(CEScoinWincurbackcolor));
            this.changePaintOpacity(true);
        }
        var item = DataManager.CEScoindataItem(index);
        var miw = this.textWidthmaxItem(item);
        this.drawItemName(item, x, y, width - miw - 28);
        this.drawText(':', x, y, width - miw, 'right');
        this.drawText($gameParty.numItems(item), x, y, width, 'right');
    }, this);

};


function Window_CESInputNumber() {
    this.initialize.apply(this, arguments);
}

Window_CESInputNumber.prototype = Object.create(Window_Selectable.prototype);
Window_CESInputNumber.prototype.constructor = Window_CESInputNumber;

Window_CESInputNumber.prototype.initialize = function(x, y, width, height) {
    Window_Selectable.prototype.initialize.call(this, x, y, width, height);
    this._item = null;
    this._CESexnum = 0;
    this._max = 1;
    this._number = 1;
    this.createButtons();
    this.hide();
    this.deactivate();
};

Window_CESInputNumber.prototype.setup = function(item) {
    this._item = item;
    this._CESexnum = DataManager.CESExchangenumber(this._item);
    const coin = DataManager.CEScoindataItem();
    const empcoin = Math.max($gameParty.maxItems(coin) - $gameParty.numItems(coin), 0);
    var maxcoin = Math.floor($gameParty.numItems(this._item) / this._CESexnum);
    maxcoin = Math.min(maxcoin, empcoin);
    this._max = maxcoin * this._CESexnum;
    this._number = this._CESexnum;
    this.placeButtons();
    this.refresh();
};

Window_CESInputNumber.prototype.createButtons = function() {
    const bitmap = ImageManager.loadSystem('ButtonSet');
    const buttonWidth = 48;
    const buttonHeight = 48;
    this._buttons = [];
    for (var i = 0; i < 5; i++) {
        var button = new Sprite_Button();
        var x = buttonWidth * i;
        var w = buttonWidth * (i === 4 ? 2 : 1);
        button.bitmap = bitmap;
        button.setColdFrame(x, 0, w, buttonHeight);
        button.setHotFrame(x, buttonHeight, w, buttonHeight);
        button.visible = true;
        this._buttons.push(button);
        this.addChild(button);
    }
    this._buttons[0].setClickHandler(this.onButtonDown2.bind(this));
    this._buttons[1].setClickHandler(this.onButtonDown.bind(this));
    this._buttons[2].setClickHandler(this.onButtonUp.bind(this));
    this._buttons[3].setClickHandler(this.onButtonUp2.bind(this));
    this._buttons[4].setClickHandler(this.onButtonOk.bind(this));
};

Window_CESInputNumber.prototype.placeButtons = function() {
    const numButtons = this._buttons.length;
    const spacing = 16;
    var totalWidth = -spacing;
    for (var i = 0; i < numButtons; i++) {
        totalWidth += this._buttons[i].width + spacing;
    }
    var x = (this.width - totalWidth) / 2;
    for (var j = 0; j < numButtons; j++) {
        var button = this._buttons[j];
        button.x = x;
        button.y = this.buttonY();
        x += button.width + spacing;
    }
};

Window_CESInputNumber.prototype.refresh = function() {
    this.contents.clear();
    this.drawItemName(this._item, this.textPadding(), this.itemY(), this.contentsWidth() * 2 / 3);
    this.drawNumber();
    this.drawTotalPrice();
};

Window_CESInputNumber.prototype.drawNumber = function() {
    const item = this._item;
    const x = this.amountX();
    const y = this.itemY();
    const width = this.amountWidth();
    const miw = this.textWidthmaxItem(item);
    const slw = this.textWidth(' / ');
    this.resetTextColor();
    this.drawText($gameParty.numItems(item), x, y, width - miw - slw, 'right');
    this.drawText(' / ', x, y, width - miw, 'right');
    this.drawText(this._number, x, y, width, 'right');
};

Window_CESInputNumber.prototype.calcTotal = function() {
    return Math.floor(this._number / this._CESexnum);
};

Window_CESInputNumber.prototype.drawTotalPrice = function() {
    const total = this.calcTotal();
    const pad = this.textPadding();
    const x = pad;
    const y = this.priceY();
    const width = this.contentsWidth();
    const item = DataManager.CEScoindataItem();
    const amounttext = total + '枚と交換';
    this.drawItemName(item, x, y, width - 140);
    this.drawText(amounttext, x, y, width - pad, 'right');
};

Window_CESInputNumber.prototype.itemY = function() {
    return Math.round(this.contentsHeight() / 2 - this.lineHeight() * 4);
};

Window_CESInputNumber.prototype.priceY = function() {
    return Math.round(this.contentsHeight() / 2 - this.lineHeight() * 2);
};

Window_CESInputNumber.prototype.buttonY = function() {
    return Math.round(this.priceY() + this.lineHeight() * 2.5);
};

Window_CESInputNumber.prototype.amountWidth = function() {
    return 224;
};

Window_CESInputNumber.prototype.amountX = function() {
    return this.contentsWidth() - this.amountWidth() - this.textPadding();
};

Window_CESInputNumber.prototype.changeNumber = function(amount) {
    const lastNumber = this._number;
    this._number = (this._number + amount).clamp(this._CESexnum, this._max);
    if (this._number !== lastNumber) {
        SoundManager.playCursor();
        this.refresh();
    }
};

Window_CESInputNumber.prototype.updateCursor = function() {
    this.setCursorRect(0, this.itemY(), this.contentsWidth(), this.lineHeight());
};

Window_CESInputNumber.prototype.update = function() {
    Window_Selectable.prototype.update.call(this);
    this.processNumberChange();
};

Window_CESInputNumber.prototype.isOkTriggered = function() {
    return Input.isTriggered('ok');
};

Window_CESInputNumber.prototype.playOkSound = function() {
};

Window_CESInputNumber.prototype.processNumberChange = function() {
    if (this.isOpenAndActive()) {
        if (Input.isRepeated('right')) {
            this.changeNumber(this._CESexnum);
        }
        if (Input.isRepeated('left')) {
            this.changeNumber(-this._CESexnum);
        }
        if (Input.isRepeated('up')) {
            this.changeNumber(this._CESexnum * 10);
        }
        if (Input.isRepeated('down')) {
            this.changeNumber(-this._CESexnum * 10);
        }
    }
};

Window_CESInputNumber.prototype.onButtonUp = function() {
    this.changeNumber(this._CESexnum);
};

Window_CESInputNumber.prototype.onButtonUp2 = function() {
    this.changeNumber(this._CESexnum * 10);
};

Window_CESInputNumber.prototype.onButtonDown = function() {
    this.changeNumber(-this._CESexnum);
};

Window_CESInputNumber.prototype.onButtonDown2 = function() {
    this.changeNumber(-this._CESexnum * 10);
};

Window_CESInputNumber.prototype.onButtonOk = function() {
    this.processOk();
};


})();
