//=============================================================================
// TRP_TrpItemGroup.js
//=============================================================================
/*:
 * @plugindesc アイテムグループプラグイン
 * @author Thirop
 *
 * @help
 * 事前にアイテムグループを登録し、
 * その中からランダムにアイテムを取得できるプラグインです。
 *
 * ◆アイテムグループの登録開始
 * まずはアイテムグループの登録開始コマンドを呼び出します。
 * □プラグインコマンド
 * 「アイテムグループ 登録 "グループID"」（日本語）
 * 「ItemGroup register "groupId"」(English) 
 * ▷パラメータ
 * "グループID/groupId"：アイテムグループのID
 *
 * □使用例
 * 「アイテムグループ 登録 ノーマル」
 * ノーマルというグループIDでアイテムグループの登録を開始します。
 *
 *
 * ◆アイテムグループの登録
 * 『上記の登録開始コマンドの直後』に「ショップの処理」のイベントコマンドで
 * アイテム登録を行います。間に注釈行などが入っても駄目なので注意してください。
 * 正しい位置で「ショップの処理」を呼び出すと、ショップ処理は行われずに
 * ショップの処理で指定したアイテムがグループとして登録されます。
 *
 * この際に、アイテムの価格を指定にすることで
 * 「アイテムの選ばれやすさ(重み)」を設定できます。
 *
 * 例えば、
 * 薬草<価格1>
 * 毒消し草<価格2>
 * ダガー<価格3>
 *
 * 上記のようにアイテムと価格を指定すると、アイテム取得時に
 * 薬草、毒消し草、ダガーが1:2:3の割合の確率で取得できます。
 * (つまり高く設定したほうが出やすくなります)
 *
 * ◆アイテムの取得
 * □プラグインコマンド
 * 「アイテムグループ 取得 "グループID"」（日本語）
 * 「ItemGroup gain "groupId"」(English)
 * ▷パラメータ
 * "グループID/groupId"：取得するアイテムグループのID
 *
 * □使用例
 * 「アイテムグループ 取得 ノーマル」
 * グループIDノーマルで登録したアイテムグループからランダムにアイテムを取得します。
 *
 * 
 * ◆取得したアイテム名の表示
 * メッセージの表示において、最後に取得したアイテム名を表示できます。
 * □制御文字
 * 「\LAST_ITEM_NAME」最後に取得したアイテムの名前
 * 「\LAST_ITEM_ICON」最後に取得したアイテムのアイコン
 * 「\LAST_ITEM」最後に取得したアイテムのアイコン+名前
 *
 * □使用例
 * 「宝箱から\LAST_ITEMを取得した！」
 *
 *
 * 【更新履歴】
 * 1.00 2020/1/17 初版。
 *
 * @param resultSwitchId
 * @text 取得結果スイッチID
 * @desc 結果を保存するスイッチID。アイテムが取得できた際にON、所持数制限などで取得出来なかった場合はOFFとなる。
 * @type switch
 * @default 0
 *
 */
//============================================================================= 


(function(){
'use strict';
var parameters = PluginManager.parameters('TRP_ItemGroup');

function supplement(defaultValue,optionArg){
	if(optionArg === undefined){
		return defaultValue;
	}
	return optionArg;
};
function supplementDef(defaultValue, optionArg, otherWords) {
	var value = supplement(defaultValue,optionArg);
	var defTargetWords = otherWords || [];
	defTargetWords.push('default');
	defTargetWords.push('def');
	defTargetWords.push('d');
	for(var i=0; i<defTargetWords.length; i++){
		var target = defTargetWords[i];
		if(value === target){
			value = defaultValue;
			break;
		}
	}
	return value;
};
function supplementDefBool(defaultValue, optionArg, otherWords) {
	var value = supplementDef(defaultValue,optionArg,otherWords);
	if(value==='true' || value==='t'){
		value = true;
	}else if(value==='false' || value==='f'){
		value = false;
	}else if(value){
		value = true;
	}else{
		value = false;
	}
	return value;
};

function itemWithType(type,id){
	if(type === 0){
		return $dataItems[id]||null;
	}else if(type === 1){
		return $dataWeapons[id]||null;
	}else{
		return $dataArmors[id]||null;
	}
};


//=============================================================================
// GameInterpreter
//=============================================================================
var _Game_Interpreter_pluginCommand = Game_Interpreter.prototype.pluginCommand;
Game_Interpreter.prototype.pluginCommand = function(command, args) {
	_Game_Interpreter_pluginCommand.call(this, command, args);

	if (command.toLowerCase()==='itemgroup' || command==='アイテムグループ'||command==='アイテムセット'){
		var subCommand = args.shift().toLowerCase();
		if(subCommand==='register'||subCommand==='登録'){
			this.commandRegisterTrpItemGroup(args);
		}else if(subCommand==='gain'||subCommand==='取得'){
			this.commandGainTrpItemGroupItem(args);
		}else if(subCommand==='shop'||subCommand==='ショップ'){
			this.commandShopWithTrpItemGroup(args);
		}else if(subCommand==='clear'||subCommand==='クリア'){
			$gameParty.clearTrpItemGroup(Number(args[0]));
		}
		return;
	}
};

Game_Interpreter.prototype.commandRegisterTrpItemGroup = function(args){
	var id = args[0];
	var goods = this.trpItemGroupWithNextShopCommand();
	if(!goods){
		throw new Error('アイテムグループ登録コマンドの直後にショップの処理でグループに登録するアイテムを指定してください。');
	}

	$gameParty.registerTrpItemGroup(id,goods);
};

Game_Interpreter.prototype.trpItemGroupWithNextShopCommand = function(){
	var nextCode = this.nextEventCode();
	if(nextCode !== 302){
		return null;
	}

	this._index += 1;
	var goods = [this.currentCommand().parameters];
    while (this.nextEventCode() === 605) {
        this._index++;
        var params = this.currentCommand().parameters;
        goods.push(params.concat());
    }
    return goods;
};


Game_Interpreter.prototype.commandGainTrpItemGroupItem = function(args){
	var item = null;
	if(args.length===0){
		var goods = this.trpItemGroupWithNextShopCommand();
		if(!goods){
			throw new Error('アイテム取得コマンドでグループIDを指定しない場合は直後にショップの処理でアイテムを指定してください。');
		}
		item = $gameParty.randomItemWithShopGoods(goods);
	}else{
		var id = args[0];
		item = $gameParty.randomItemWithTrpItemGroupId(id);
	}

	var result = false;
	if(item){
		var lastNum = $gameParty.numItems(item);
		$gameParty.gainItem(item,1);
		result = ($gameParty.numItems(item)>lastNum);
	}

	var switchId = Number(parameters.resultSwitchId);
	if(switchId > 0){
		$gameSwitches.setValue(switchId,result);
	}
};

Game_Interpreter.prototype.commandShopWithTrpItemGroup = function(args){
	var id = args[0];
	var purchaseOnly = supplementDefBool(false,args[1]);
	var goods = $gameParty.trpItemGroup(id);
	SceneManager.push(Scene_Shop);
	SceneManager.prepareNextScene(goods, purchaseOnly);
};


//=============================================================================
// Widnow_Base
//=============================================================================
var _Window_Base_convertEscapeCharacters = Window_Base.prototype.convertEscapeCharacters;
Window_Base.prototype.convertEscapeCharacters = function(text) {
	var text = _Window_Base_convertEscapeCharacters.call(this,text);
	text = text.replace(/\x1bLAST_ITEM_ICON/gi,$gameParty.lastGainItemString(false,true));
	text = text.replace(/\x1bLAST_ITEM_NAME/gi,$gameParty.lastGainItemString(true,false));
	text = text.replace(/\x1bLAST_ITEM/gi,$gameParty.lastGainItemString(true,true));
	return text
};


//=============================================================================
// Game_Party
//=============================================================================
var _Game_Party_initialize = Game_Party.prototype.initialize;
Game_Party.prototype.initialize = function() {
	_Game_Party_initialize.call(this);

	this.safeInitTrpItemGroup();
};
Game_Party.prototype.safeInitTrpItemGroup = function(){
	if(!!this._itemGroup)return;

	this._itemGroup = {};
	this._lastGainItemId = 0;
	this._lastGainItemType = 0;
};

Game_Party.prototype.registerTrpItemGroup = function(id,group){
	this.safeInitTrpItemGroup();
	this._itemGroup[id] = group;
};
Game_Party.prototype.clearTrpItemGroup = function(id){
	this.safeInitTrpItemGroup();
	delete this._itemGroup[id];
};

Game_Party.prototype.trpItemGroup = function(id){
	return this._itemGroup[id];
};

Game_Party.prototype.randomItemWithTrpItemGroupId = function(id){
	this.safeInitTrpItemGroup();

	var group = this._itemGroup[id];
	if(!group)return null;
	return this.randomItemWithShopGoods(group);
};

Game_Party.prototype.randomItemWithShopGoods = function(goods){
	this._lastGainItemType = 0;
	this._lastGainItemId = 0;

	var totalWeight = 0;
	var length = goods.length;
	var i,data;
    for(i = 0; i<length; i=(i+1)|0){
        data = goods[i];
        totalWeight += data[2]||1;
    }

    var randWeight = Math.randomInt(totalWeight)+1;
    var weight = 0;

    for(i = 0; i<length; i=(i+1)|0){
        data = goods[i];
        weight += data[3];
        if(weight>=randWeight){
        	var type = data[0];
    		var id = data[1];
        	var item = itemWithType(type,id);
        	if(item){
        		this._lastGainItemType = type;
				this._lastGainItemId = id;
        	}
        	return item || null;
        }
    }
    return null;
};


var _Game_Party_gainItem = Game_Party.prototype.gainItem;
Game_Party.prototype.gainItem = function(item, amount, includeEquip) {
	if(!item){
		this._lastGainItemId = 0;
		this._lastGainItemType = 0;
	}else{
		this._lastGainItemId = item.id;
		if(DataManager.isItem(item)){
			this._lastGainItemType = 0;
		}else if(DataManager.isWeapon(item)){
			this._lastGainItemType = 1;
		}else{
			this._lastGainItemType = 2;
		}
	}

	_Game_Party_gainItem.call(this,item,amount,includeEquip);
};

Game_Party.prototype.lastGainItemString = function(withName,withIcon){
	this.safeInitTrpItemGroup();

	withName = supplement(true,withName);
	withIcon = supplement(false,withIcon);

	var type = this._lastGainItemType;
	var id = this._lastGainItemId;
	var item = itemWithType(type,id);
	if(!item)return '';

	var text = '';
	if(withIcon){
		text += '\x1bI['+item.iconIndex+']';
	}
	if(withName){
		text += item.name;
	}
	return text;
};



})();