//=============================================================================
// RPGツクールMZ - LL_SprashScreen.js v1.0.1
//-----------------------------------------------------------------------------
// ルルの教会 (Lulu's Church)
// https://nine-yusha.com/
//
// URL below for license details.
// https://nine-yusha.com/plugin/
//=============================================================================

/*:
 * @target MZ
 * @plugindesc スプラッシュスクリーンを表示します。
 * @author ルルの教会
 * @url https://nine-yusha.com/plugin-sprashscreen/
 *
 * @help LL_SprashScreen.js
 *
 * スプラッシュスクリーンを表示します。
 * スプラッシュスクリーンは複数設定することが可能です。
 * 決定キー、クリックでスキップさせることもできます。
 * ジングルMEを再生することも可能です。
 *
 * 利用規約:
 *   ・著作権表記は必要ございません。
 *   ・利用するにあたり報告の必要は特にございません。
 *   ・商用・非商用問いません。
 *   ・R18作品にも使用制限はありません。
 *   ・ゲームに合わせて自由に改変していただいて問題ございません。
 *   ・プラグイン素材としての再配布（改変後含む）は禁止させていただきます。
 *
 * 作者: ルルの教会
 * 作成日: 2021/8/24
 *
 * @param splashImages
 * @text 画像リスト
 * @desc スプラッシュスクリーンとして表示する画像リストです。
 * 上から順番に表示されます。
 * @default []
 * @type struct<splashImages>[]
 *
 * @param skipEnabled
 * @text スキップ許可
 * @desc 決定キー、クリックでスキップを許可するか設定します。
 * @type boolean
 * @default true
 *
 * @param debugDisabled
 * @text テストプレイ時は非表示
 * @desc テストプレイ時は非表示にするか設定します。
 * @type boolean
 * @default false
 *
 * @param jingleSettings
 * @text ジングルの設定
 * @desc ※この項目は使用しません
 *
 * @param meName
 * @text ME
 * @desc 再生する音声ファイルを選択してください。
 * @dir audio/me
 * @type file
 * @require 1
 * @parent jingleSettings
 *
 * @param meVolume
 * @text MEの音量
 * @desc MEの音量(0～100%)です。 (初期値: 90%)
 * @default 90
 * @max 100
 * @min 0
 * @type number
 * @parent jingleSettings
 *
 * @param mePitch
 * @text MEのピッチ
 * @desc MEのピッチ(50～150%)です。 (初期値: 100%)
 * @default 100
 * @max 150
 * @min 50
 * @type number
 * @parent jingleSettings
 *
 * @param mePan
 * @text MEの位相
 * @desc MEの位相(-100～100)です。 (初期値: 0)
 * @default 0
 * @max 100
 * @min -100
 * @type number
 * @parent jingleSettings
 */

/*~struct~splashImages:
 *
 * @param imageName
 * @text 画像ファイル名
 * @desc 表示する画像ファイルを選択してください。
 * @dir img/system
 * @type file
 * @require 1
 *
 * @param fadeTime
 * @text フェード時間
 * @desc フェードイン・フェードアウトの効果時間です。
 * @default 60
 * @min 0
 * @max 999
 * @type number
 *
 * @param waitTime
 * @text 表示時間
 * @desc フェードイン後の表示時間です。
 * @default 60
 * @min 0
 * @max 999
 * @type number
 */


(() => {
    const pluginName = "LL_SprashScreen";

    const parameters = PluginManager.parameters(pluginName);
    const skipEnabled = eval(parameters["skipEnabled"] || "true");
    const debugDisabled = eval(parameters["debugDisabled"] || "false");
    // ME
    const meName = String(parameters["meName"] || "");
    const meVolume = Number(parameters["meVolume"] || 90);
    const mePitch = Number(parameters["mePitch"] || 100);
    const mePan = Number(parameters["mePan"] || 0);
    // 画像リスト
    const splashImages = JSON.parse(parameters["splashImages"] || "null");
	let splashImageLists = [];
	if (splashImages) {
		splashImages.forEach((elm) => {
			splashImageLists.push(JSON.parse(elm));
		});
    }

    const _Scene_Boot_loadSystemImages = Scene_Boot.prototype.loadSystemImages;
    Scene_Boot.prototype.loadSystemImages = function() {
        _Scene_Boot_loadSystemImages.call(this);
        // スプラッシュ画像を読み込み
        splashImageLists.forEach((item) => {
	        ImageManager.loadSystem(item.imageName);
		});
    };

    const _Scene_Boot_startNormalGame = Scene_Boot.prototype.startNormalGame;
    Scene_Boot.prototype.startNormalGame = function() {
        if (!DataManager.isBattleTest() && !DataManager.isEventTest() && splashImageLists.length > 0) {
            if (!Utils.isOptionValid('test') || !debugDisabled) {
                this.checkPlayerLocation();
                DataManager.setupNewGame();
                SceneManager.goto(Scene_SplashScreen);
                return;
            }
        }

        _Scene_Boot_startNormalGame.apply(this, arguments);
    };

    //-----------------------------------------------------------------------------
    // Scene_SplashScreen
    //
    // スプラッシュスクリーンを表示する独自のシーンを追加定義します。

    function Scene_SplashScreen() {
        this.initialize(...arguments);
    }

    Scene_SplashScreen.prototype = Object.create(Scene_Base.prototype);
    Scene_SplashScreen.prototype.constructor = Scene_SplashScreen;

    Scene_SplashScreen.prototype.initialize = function() {
        Scene_Base.prototype.initialize.call(this);
        SceneManager.clearStack();
        this._originalSplash = null;
        this._splashIndex = 0;
        this._fadeTime = 60;
        this._waitTime = 0;
        this._fadeOpacity = 255;
    };

    Scene_SplashScreen.prototype.create = function() {
        Scene_Base.prototype.create.call(this);
        this._originalSplash = new Sprite();
        this.addChild(this._originalSplash);
    };

    Scene_SplashScreen.prototype.start = function() {
        Scene_Base.prototype.start.call(this);
        this.playMeJingle();
    };

    Scene_SplashScreen.prototype.update = function() {
        Scene_Base.prototype.update.call(this);
        // 画像切り替え
        if (this._waitTime == 0) {
            if (this._fadeOpacity == 0) {
                this.startFadeOut(this._fadeTime, false);
            } else if (this._fadeOpacity == 255) {
                this.setSplashImage();
                // 終了判定
                if (!this._originalSplash.bitmap) {
                    this.gotoNextScene();
                    return;
                }
                this.adjustSpriteImage();
                this.startFadeIn(this._fadeTime, false);
            }

        }
        // スキップ判定
        if (skipEnabled) {
            if (Input.isTriggered("ok") || TouchInput.isTriggered()) {
                this._waitTime = 0;
                this.startFadeOut(1, false);
            }
        }
        // ウェイト判定
        if (this._fadeOpacity == 0 && this._waitTime > 0) {
            this._waitTime--;
        }
    };

    Scene_SplashScreen.prototype.setSplashImage = function() {
        const splashImage = splashImageLists[this._splashIndex];
        this._originalSplash.bitmap = null;
        if (splashImage) {
            this._originalSplash.bitmap = ImageManager.loadSystem(String(splashImage.imageName));
            this._fadeTime = Number(splashImage.fadeTime);
            this._waitTime = Number(splashImage.waitTime);
            // インデックスを更新
            this._splashIndex++;
        }
    };

    Scene_SplashScreen.prototype.adjustSpriteImage = function() {
        this.centerSprite(this._originalSplash);
    };

    Scene_SplashScreen.prototype.playMeJingle = function() {
        if (!meName) return;
        AudioManager.playMe({
            "name": meName,
            "volume": meVolume,
            "pitch": mePitch,
            "pan": mePan
        });
    };

    Scene_SplashScreen.prototype.gotoNextScene = function() {
        AudioManager.stopMe();
        SceneManager.goto(Scene_Title);
    };
})();
