//=============================================================================
// SoR_GabWindow_MZ_Right.js
// SoR License inherited from MIT License (C) 2020 蒼竜
// http://dragonflare.blue/dcave/license.php
// ---------------------------------------------------------------------------
// Latest version v1.22 (2021/06/09)
//=============================================================================
/*:ja
 @plugindesc ＜おしゃべりポップアップ Type-R＞ v1.22
 @author 蒼竜
 @target MZ
 @url http://dragonflare.blue/dcave/
 @help 1～2行の簡易的なチャットウィンドウを表示します。
 スクリプトコマンドからの入力に反応して、マップ画面および戦闘画面で随時表示されます。

機能の広範化により、製作者ごとに無駄な処理が氾濫することを抑制するため、
 UIデザインの方向性ごとにスクリプトファイルを分けています。
 好みのスタイルのものを1つだけ選んで導入してください。
 (このスクリプトは、"Type-R"のものです。)
																	   									  
 @param WindowStyle
 @desc ウィンドウ形式 (default: 0)
 @type select
 @option 標準スキンによるウィンドウ
 @value 0
 @option 暗くする
 @value 1
 @option 独自画像を利用
 @value 2
 @default 0

 @param WindowSkinImage 
 @desc WindowStyleで設定したウィンドウ形式が 2.「独自画像を利用」 のときに参照する画像 (default: gab_bg)
 @type file
 @dir img/system
 @default gab_bg
 @param GabWindowYpadd_originalSkin
 @desc WindowStyleで設定したウィンドウ形式が 2.「独自画像を利用」 のときの縦の間隔の補正値(default: 0)
 @default 0
 @type number

@param GabSoundSE
@desc Gab挿入ごとに鳴らす効果音を設定します
@type struct<SEDATA>
@default {"name":"","volume":"100","pitch":"100","pan":"0"}
@dir audio/se

 @param GabWindowY
 @desc 描画位置Y座標(default: 560)
 @default 560
 @type number
 @param GabWindowWidthPadd
 @desc ウィンドウ横幅に対する余白(default: 108)
 @default 108
 @type number
 @param GabWindowHeight
 @desc ウィンドウ縦幅(default: 64)
 @default 64
 @type number
 
 @param GabWindow_baseDuration
 @desc 現在のGabの表示時間(次のGabの挿入を待つ時間)(default: 160)
 @default 160
 @type number
 @param GabWindow_keepDuration
 @desc GabWindow_baseDuration後のGabを消去するまでの待機時間(default: 400)
 @default 400
 @type number
 @param Multiple_GabSpaces
 @desc Gab同士の縦の間隔(default: 62)
 @default 62
 @type number

 @param GabWindowHeightPadd_ForBattle
 @desc 戦闘画面におけるGabWindowのy方向位置補正(default: -196)
 @default -196
 @type number
 @min -9999

 @param FaceXPosition
 @desc 顔画像表示位置、画像中心のx座標(default: 41)
 @default 41
 @type number
 @param FaceYPosition
 @desc 顔画像表示位置、画像中心のy座標(default: 23)
 @default 23
 @type number

 @param scaleX_Face_default
 @desc x方向の顔画像の表示倍率(default: 0.5)
 @default 0.5
 @type number
 @decimals 2
 @param scaleY_Face_default
 @desc y方向の顔画像の表示倍率(default: 0.5)
 @default 0.5
 @type number
 @decimals 2
 @param ActorFaceRect_U
 @desc 顔画像グラフィックの表示領域(Y方向上端)(default: 0)
 @default 0
 @type number
 @param ActorFaceRect_L
 @desc Faceグラフィックの表示領域(Y方向下端)(default: 144)
 @default 144
 @type number

 @param MessageXPosition
 @desc メッセージ表示位置x補正、FaceXPositionを参考に調整のこと(default: 12)
 @default 12
 @type number
 @param GabStringFont
 @desc Gabフォントサイズ(default: 16)
 @default 16
 @type number

 @command PushGab
 @text 呼び出し[おしゃべりポップアップ]
 @desc メッセージの描画キューを登録します。
 @arg arg0
 @type number
 @text 顔グラフィックを表示するアクターID
 @desc データベースの並び順に対応し、0で非表示にします
 @arg arg1
 @text メッセージ文字列
 @desc 1回のポップアップで描画する文字列。 | で区切ると2行に分けることができます

 @command ForceGabClear
 @text 全消去[おしゃべりポップアップ]
 @desc ウィンドウに描画中の内容、および登録中のキューを直ちに全て消去します
*/
/*:
 @plugindesc <GabWindow Type-Right> v1.22
 @author Soryu
 @target MZ
 @url http://dragonflare.blue/dcave/index_e.php
 @help Create a simple gab window to show a few line of text.
 The game accepts script commands via script command
 to show Gab Window on Map and Battle scenes.

 In order to avoid outburst of unnecessary functions for respective developers
 by implementing various features for attractive UI,
 script files are separated by the design.
 Thus, install just ONLY ONE script for your preference. 
 (This file is for "Type-Right".)

 @param WindowStyle
 @desc Style of Gab Window (default: 0)
 @type select
 @option Default window with the skin
 @value 0
 @option Dark
 @value 1
 @option Use original UI images
 @value 2
 @default 0

 @param WindowSkinImage 
 @desc The image used for your original Gab window in case of "Use original UI images". (default: gab_bg)
 @type file
 @dir img/system
 @default gab_bg
 @param GabWindowYpadd_originalSkin
 @desc Additional vertical padding for each gab window in case of "Use original UI images".(default: 0)
 @default 0
 @type number

@param GabSoundSE
@desc SE for inserting every Gab Window (default: none) 
@type struct<SEDATAE>
@default {"name":"","volume":"100","pitch":"100","pan":"0"}
@dir audio/se

 @param GabWindowY
 @desc Y-cooridnate(upper left corner) of Gab Window (default: 560)
 @default 560
 @type number
 @param GabWindowWidthPadd
 @desc Padding after texts inner Gab window (default: 108)
 @default 108
 @type number
 @param GabWindowHeight
 @desc Height of a window for each Gab (default: 64)
 @default 64
 @type number
 
 @param GabWindow_baseDuration
 @desc Interval of processing new Gab window (default: 160)
 @default 160
 @type number
 @param GabWindow_keepDuration
 @desc After time GabWindow_baseDuration elapsed, the duration to keep Gab Window (default: 400)
 @default 400
 @type number
 @param Multiple_GabSpaces
 @desc Vertical space for each Gab (default: 62)
 @default 62
 @type number

 @param GabWindowHeightPadd_ForBattle
 @desc Additional vertical padding for Gab Window in battle scene (default: -196)
 @default -196
 @type number
 @min -9999

 @param FaceXPosition
 @desc X-coordinate (center of the sprite) of Face graphic (default: 41)
 @default 41
 @type number
 @param FaceYPosition
 @desc Y-coordinate (center of the sprite) of Face graphic (default: 23)
 @default 23
 @type number

 @param scaleX_Face_default
 @desc Horizontal scale of face sprite(default: 0.5)
 @default 0.5
 @type number
 @decimals 2
 @param scaleY_Face_default
 @desc Vertical scale of face sprite(default: 0.5)
 @default 0.5
 @type number
 @decimals 2
 @param ActorFaceRect_U
 @desc The range of displaying face sprites (upper)(default: 0)
 @default 0
 @type number
 @param ActorFaceRect_L
 @desc The range of displaying face sprites (lower)(default: 144)
 @default 144
 @type number

 @param MessageXPosition
 @desc Padding for Gab text in a window (default: 80)
 @default 80
 @type number
 @param GabStringFont
 @desc Font size for Gab text(default: 16)
 @default 16
 @type number

 @command PushGab
 @text Message queue[SoR_GabWindow]
 @desc Register a queue with a messange and face graphic
 @arg arg0
 @type number
 @text Actor ID for Face 
 @desc ID corresponds to that in database, 0 to disable 
 @arg arg1
 @text Message
 @desc A message in one window. Split by | to align the message with 2 lines.

 @command ForceGabClear
 @text Queue All Clear [SoR_GabWindow]
 @desc Delete all contents on the window including queues which are to be displayed 

*/
/*~struct~SEDATA:
@type string
@param name
@dir audio/se/
@type file
@desc 効果音
@param volume
@desc 音量 [0...100]
@type number
@default 100
@min 0
@max 100
@param pitch
@desc ピッチ [50...150]
@type number
@default 100
@min 50
@max 150
@param pan
@desc パン(位相) [-50...50]
@type number
@default 0
@min -50
@max 50
*/
/*~struct~SEDATAE:
@type string
@param name
@dir audio/se/
@type file
@desc SE File
@param volume
@desc Voulme [0...100]
@type number
@default 100
@min 0
@max 100
@param pitch
@desc Pitch [50...150]
@type number
@default 100
@min 50
@max 150
@param pan
@desc Pan [-50...50]
@type number
@default 0
@min -50
@max 50
*/

var Imported = Imported || {};
Imported.SoR_GabWindows = true;

var SoR = SoR || {};

(function() {
const pluginName = "SoR_GabWindow_MZ_Right";

if(PluginManager._scripts.includes("SoR_GabWindow_MZ_Left")) throw new Error("[SoR_GabWindows] Do NOT import more than 2 types of <SoR_GabWindow> series.");
if(PluginManager._scripts.includes("SoR_GabWindow_MZ_Left2") || PluginManager._scripts.includes("SoR_GabWindow_MZ_Right2")) throw new Error("[SoR_GabWindows] Do NOT import with <SoR_GabWindow> Type-L2 and R2.");


const GabParam = PluginManager.parameters(pluginName);
const WindowLayoutSide = Number(GabParam['WindowLayoutSide'] || 0);
										
const WindowStyle = Number(GabParam['WindowStyle'] || 0);
WindowSkinImage = String(GabParam['WindowSkinImage'] || 'gab_bg');

GabSoundSE = convertJsonSE(GabParam['GabSoundSE']) || '';
GabWindowY = Number(GabParam['GabWindowY'] || 560);
GabWindowWidthPadd = Number(GabParam['GabWindowWidthPadd'] || 108);
GabWindowHeight = Number(GabParam['GabWindowHeight'] || 64);
GabWindow_baseDuration = Number(GabParam['GabWindow_baseDuration'] || 160);
GabWindow_keepDuration = Number(GabParam['GabWindow_keepDuration'] || 400);
Multiple_GabSpaces = Number(GabParam['Multiple_GabSpaces'] || 62);
GabWindowYpadd_originalSkin = Number(GabParam['GabWindowYpadd_originalSkin'] || 0);

GabWindowHeightPadd_ForBattle = Number(GabParam['GabWindowHeightPadd_ForBattle'] || -196);
FaceXPosition = Number(GabParam['FaceXPosition'] || 41);
FaceYPosition = Number(GabParam['FaceYPosition'] || 23);
scaleX_Face_default = Number(GabParam['scaleX_Face_default'] || 0.5);
scaleY_Face_default = Number(GabParam['scaleY_Face_default'] || 0.5);
ActorFaceRect_U = Number(GabParam['ActorFaceRect_U'] || 0);
ActorFaceRect_L = Number(GabParam['ActorFaceRect_L'] || 144);
MessageXPosition = Number(GabParam['MessageXPosition'] || 80);
GabStringFont = Number(GabParam['GabStringFont'] || 16);

function convertJsonSE(param) {
    const obj = JSON.parse(param);
    obj.volume = Number(obj.volume);
    obj.pan = Number(obj.pan);
    obj.pitch = Number(obj.pitch);
    return obj;
}

PluginManager.registerCommand(pluginName, "PushGab", args => {
	const argc = args.length;
	if(!Number.isFinite(Number(args.arg0))) return;
    const id = args.arg0;
		
	if(!args.arg1) return;	
	let str_arr = ['', ''];

	const strs = args.arg1.split('|');
	str_arr[0] = strs[0];
	if(strs.length == 2)str_arr[1] = strs[1];					   

    const obj = new SoR_GabWindow(str_arr[0],str_arr[1],id);
	$gameTemp.SoR_GabPush(obj);
});
	
PluginManager.registerCommand(pluginName, "ForceGabClear", args => {
	$gameTemp.clearGabCommand();
});

////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////

const SoR_GabW_GT_initialize = Game_Temp.prototype.initialize;
Game_Temp.prototype.initialize = function() {
	SoR_GabW_GT_initialize.call(this);
    this.gab_ready_queue = [];
   // this.gab_log_queue = [];
	this.gab_duration = 0;
	this.isForceClear = false;
	this.ForceTransferClear = false;
};

//force clear gab
Game_Temp.prototype.clearGabCommand = function() {
	this.gab_ready_queue.length = 0;
	this.gab_duration = 0;
	this.isForceClear = true;
}
Game_Temp.prototype.IsForceClearGab = function() {
	return this.isForceClear;
}



Game_Temp.prototype.SoR_GabPush = function(obj) {
	this.gab_ready_queue.push(obj);
}

Game_Temp.prototype.SoR_GabPullHead = function() {
	let obj = null;
	if(this.SoR_GabQueueCount() > 0){
		obj = this.gab_ready_queue[0];
	}
	
	return obj;
}
Game_Temp.prototype.SoR_GabPop = function() {
	this.gab_ready_queue.shift();
}



Game_Temp.prototype.SoR_GabQueueCount = function(){
	return this.gab_ready_queue.length;
}

////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////
const SoR_GabW_SB_initialize = Scene_Base.prototype.initialize;
Scene_Base.prototype.initialize = function() {
    SoR_GabW_SB_initialize.call(this);
		
	this.gab_duration = 0;
	this.gab_shown = [];

};

const SoR_GabW_SM_push = SceneManager.push;
SceneManager.push = function(next_scene) {
    if (this._scene instanceof Scene_Map) {
		for(let i=0; i<this._scene.gab_shown.length; i++){
			const obj = this._scene.gab_shown[i];
			obj.TempCloseGab();
			this._scene.SoR_GabField.removeChild(obj);
			if(WindowStyle == 2) this._scene.SoR_GabField.removeChild(obj.bg_img);
			if(obj.face)this._scene.SoR_GabField.removeChild(obj.face);
		}
		this._scene.gab_shown.length = 0;
    }
	
    SoR_GabW_SM_push.call(this, next_scene);
};

const SoR_GabW_GP_reserveTransfer = Game_Player.prototype.reserveTransfer;
Game_Player.prototype.reserveTransfer = function(mapId, x, y, d, fadeType) {
    SoR_GabW_GP_reserveTransfer.call(this, mapId, x, y, d, fadeType);
	$gameTemp.ForceTransferClear = true;
};

///////////////////////////////////////////////////////////////

Scene_Base.prototype.SoR_createGabWindow = function(scene) {
	this._scene = scene;
	this.SoR_GabField = new Sprite();
    this.addChild(this.SoR_GabField);
	if(WindowStyle == 2) ImageManager.loadSystem(WindowSkinImage); //cache
};

//
//update 
//
Scene_Base.prototype.GabWindowManager = function() {
	//reset for transition
	if($gameTemp.ForceTransferClear || $gameTemp.IsForceClearGab()){
		for(let i=0; i<this.gab_shown.length; i++){
			const obj = this.gab_shown[i];
			this.SoR_GabField.removeChild(obj);
			if(WindowStyle == 2) this.SoR_GabField.removeChild(obj.bg_img);
			if(obj.face)this.SoR_GabField.removeChild(obj.face);
		}
		this.gab_shown.length = 0;
		if($gameTemp.ForceTransferClear) $gameTemp.ForceTransferClear = false;
		if($gameTemp.IsForceClearGab()) $gameTemp.isForceClear = false;
		return;
	}

	//create
	if($gameTemp.SoR_GabQueueCount()>0 && this.gab_duration == 0){
		const obj = $gameTemp.SoR_GabPullHead();
		if(obj!=null) this.SoR_GabSetup(obj, this._scene);
	}
	if(this.gab_duration>0) this.gab_duration--;
	
	//main update
	for(let i=0; i<this.gab_shown.length; i++){
		const obj = this.gab_shown[i];
		if(i==this.gab_shown.length-1){
			if(this.gab_duration==0){
				if($gameTemp.SoR_GabPullHead() == obj)	$gameTemp.SoR_GabPop();
				obj.keepDuration--;
			}
		}
		else obj.keepDuration--;
		
		this.gab_shown[i].shiftY(this.gab_shown.length-i-1);
	}

    //expire
	for(let i=0; i<this.gab_shown.length; i++){
		const obj = this.gab_shown[i];
		if(obj.keepDuration==0){
			this.SoR_GabField.removeChild(obj);
			if(WindowStyle == 2) this.SoR_GabField.removeChild(obj.bg_img);
			if(obj.face)this.SoR_GabField.removeChild(obj.face);
			this.gab_shown.splice(i, 1);
			i--;
		}
	}
	
}

Scene_Base.prototype.SoR_GabSetup = function(obj, scene){	
		this.gab_shown.push(obj);
		this.gab_duration = obj.drawDuration;
		
		obj.setupGab(scene);
		if(WindowStyle == 2){
			if( obj.bg_img.bitmap._baseTexture == null || obj.bg_img.transform == null) this.SoR_GabField.reloadImg();
			this.SoR_GabField.addChild(obj.bg_img);
	   }

		this.SoR_GabField.addChild(obj);
		
		if(obj.faceID!=0 && ( obj.face.bitmap._baseTexture == null || obj.face.transform == null) )  obj.face = obj.GetFaceSprite(obj.faceID);
		if(obj.face) this.SoR_GabField.addChild(obj.face);
		if(obj.gab_se)AudioManager.playSe(obj.gab_se);
}

////////////////////////////////////////////////////////////
// For Scene_Map
////////////////////////////////////////////////////////////
const SoR_GabW_SM_createAllWindows = Scene_Map.prototype.createAllWindows;
Scene_Map.prototype.createAllWindows = function() {
    SoR_GabW_SM_createAllWindows.call(this);
    this.SoR_createGabWindow('Scene_Map');
};

const SoR_GabW_SM_update = Scene_Map.prototype.update;
Scene_Map.prototype.update = function() {
	SoR_GabW_SM_update.call(this);
	this.GabWindowManager();
}

////////////////////////////////////////////////////////////
// For Scene_Battle
////////////////////////////////////////////////////////////

const SoR_GabW_SB_createAllWindows = Scene_Battle.prototype.createAllWindows;
Scene_Battle.prototype.createAllWindows = function() {
    SoR_GabW_SB_createAllWindows.call(this);
    this.SoR_createGabWindow('Scene_Battle');
};

const SoR_GabW_SB_update = Scene_Battle.prototype.update;
Scene_Battle.prototype.update = function() {
	SoR_GabW_SB_update.call(this);
	this.GabWindowManager();
}




////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////
function SoR_GabWindow() {
    this.initialize.apply(this, arguments);
}

SoR_GabWindow.prototype = Object.create(Window_Base.prototype);
SoR_GabWindow.prototype.constructor = SoR_GabWindow;

SoR_GabWindow.prototype.initialize = function(text1, text2, ID){
	Window_Base.prototype.initialize.call(this,new Rectangle( 0, 0, 800, GabWindowHeight+16));
	this.faceID = ID;
    this._text1 = text1;
    this._text2 = text2;
	this.contents.fontSize = GabStringFont;
	this.drawLength = this.CalcLength();
	this.drawDuration = this.CalcDuration();
	this.keepDuration = GabWindow_keepDuration;
	this.enter_effect = 512;
	this.x = Graphics.boxWidth- this.drawLength +this.enter_effect;
	this.Basey = GabWindowY; 
	this.y = this.Basey;
	this.diffY = 0;
	this.bg_img = null;
	this.gab_se = null;
	
	
	if(Imported.SoR_MessageExtension){
		this.isCenterizeLine = [false, false, false, false];
		this.isSoR_MesRuby = false;
		this.SoR_MesRubyarr = [];
		this.CalcLineDrawWidth = [0, 0, 0, 0, 0];
		this.Message_CurrentPageline = 0; 
	}			

	this.gab_se = GabSoundSE;
	
	if(ID>=1) this.face = this.GetFaceSprite(ID);
	else this.face = null;
	
	this.width = this.drawLength + GabWindowWidthPadd;
	this.height = GabWindowHeight;	
    this.openness = 0;
	
	if(WindowStyle == 0) this.setBackgroundType(0);	
	else this.setBackgroundType(2);	//no default window
	
	if(WindowStyle == 2){
		this.bg_img = new Sprite(ImageManager.loadSystem(WindowSkinImage));
	}
}

Object.defineProperty(SoR_GabWindow.prototype, "innerWidth", {
    get: function() {
        return Math.max(0, this._width);
    },
    configurable: true
});
Object.defineProperty(SoR_GabWindow.prototype, "innerHeight", {
    get: function() {
        return Math.max(0, this._height);
    },
    configurable: true
});


SoR_GabWindow.prototype.destroy = function(options) {}


SoR_GabWindow.prototype.setupGab = function(scene){
	
	this.scene_ypadd = 0;
	if(scene == 'Scene_Battle') this.scene_ypadd = GabWindowHeightPadd_ForBattle;
	

	
	this.contents.clear();
	if(WindowStyle != 0) this.DrawBackground(255);
	this.DrawMessages();	 

	this.width = this.drawLength + GabWindowWidthPadd;
	this.x = Graphics.boxWidth- this.drawLength + this.enter_effect;
	this.y = this.Basey + this.scene_ypadd;
	
	this.openness = 255;
	if(this.face) this.face.visible = true; 
}

SoR_GabWindow.prototype.TempCloseGab = function(){
	this.openness = 0;
	if(this.face) this.face.visible = false;
}


SoR_GabWindow.prototype.DrawMessages = function(){
	this.contents.fontSize = GabStringFont;
	let numl = 0;
	if(this._text2 == '') numl=1;

    let xpos = MessageXPosition;
	if(!this.face) xpos = FaceXPosition;

	this.SoR_GabRubyarr = []; 
	
	let l1=0, l2=0;
	l1 = this.DrawTextsXX(this._text1, xpos, numl*10 -8, this.width,0, numl);
	if(numl==0) l2 = this.DrawTextsXX(this._text2, xpos, 10, this.width,1, numl);
	this.drawLength = l1>l2? l1: l2;															 
}

SoR_GabWindow.prototype.shiftY = function(tmp_y){
	if(this.y != this.Basey + this.scene_ypadd - tmp_y*Multiple_GabSpaces){
		this.diffY = (this.Basey + this.scene_ypadd - tmp_y*Multiple_GabSpaces) - this.y;
	}
	else this.diffY = 0;
	this.changeOpacity(tmp_y);
}

SoR_GabWindow.prototype.changeOpacity = function(opaLev){
	const Eraser = this.keepDuration<=45 ? (45-this.keepDuration)*5 : 0;
	const currentO = this.opacity;							   
	
	if(WindowStyle == 0) this.opacity = 255 - opaLev*55 - Eraser;
	if(this.opacity < 0) this.opacity = 0;
	this.contents.paintOpacity = 255 - opaLev*55 - Eraser;
	if(this.contents.paintOpacity < 0) this.contents.paintOpacity = 0;
	if(this.face)this.face.opacity = this.contents.paintOpacity;

	if(WindowStyle == 0 && currentO == this.opacity) return; 
	
	this.contents.clear();
	if(WindowStyle != 0) this.DrawBackground(this.contents.paintOpacity);	
	this.DrawMessages(); 
	
	const op = this.contents.paintOpacity/512.0;
	const rgba = 'rgba(0, 0, 0, ' + op.toFixed(2) + ')' ;
	this.contents.outlineColor =  rgba;
}


SoR_GabWindow.prototype.update = function(){
	Window_Base.prototype.update.call(this);
	
	if(this.enter_effect>0) this.enter_effect = Math.floor(this.enter_effect/1.25);
	else this.enter_effect = 0;
		
	this.x = Graphics.boxWidth- this.drawLength-GabWindowWidthPadd +this.enter_effect;
	this.y += Math.floor(this.diffY/2.25);
	if(this.face)this.face.x = Graphics.boxWidth-FaceXPosition+this.enter_effect;
	if(this.face)this.face.y = this.y+FaceYPosition + Math.floor(this.diffY/2.25);
}


SoR_GabWindow.prototype.DrawBackground = function(opa) {	
	if(WindowStyle == 1){
		const color1 = ColorManager.dimColor1();
		const color2 = ColorManager.dimColor2(); 
		this.contents.fillRect(0, 0, this.width / 2, this.height, color1);
		this.contents.gradientFillRect(this.width / 2, 0, this.width / 2, this.height, color1, color2);
	}
	else{
		this.bg_img.x = this.x;
		this.bg_img.y = this.y+GabWindowYpadd_originalSkin;
		this.bg_img.opacity = opa;
	}
}

SoR_GabWindow.prototype.CalcLength = function(){
	const x1 = this.textWidth(this._text1);
	const x2 = this._text2 == '' ? x1 : this.textWidth(this._text2);
	return x1 > x2 ? x1 : x2;
}
SoR_GabWindow.prototype.CalcDuration = function(){
	const x1 = this.textWidth(this._text1);
	const x2 = this.textWidth(this._text2);
	const base = this.textWidth('００００００００００００００００００００');
	const time = (x1+x2)/base;
	return time > 1.0 ? Math.floor(GabWindow_baseDuration*time) : GabWindow_baseDuration; 
}

SoR_GabWindow.prototype.GetFaceSprite = function(id){
	const fname = $dataActors[id].faceName;
	const fidx  = $dataActors[id].faceIndex;
	const img = ImageManager.loadFace(fname);
	let Face_spr = new Sprite(img);	

    const pw = ImageManager.faceWidth;
    const ph = ImageManager.faceHeight;
	
	const upp = ActorFaceRect_U;
	const low = ActorFaceRect_L;
	if(ActorFaceRect_U <= -1) upp = 0;	
	if(ActorFaceRect_L <= -1 || ActorFaceRect_L > Window_Base._faceHeight) low = ph;
    
	const drawarea = Math.abs(upp-low);
	Face_spr.setFrame((fidx%4) * pw, Math.floor(fidx/4)*ph+upp, pw, drawarea);
	Face_spr.scale.x = scaleX_Face_default;
	Face_spr.scale.y = scaleY_Face_default;
	Face_spr.anchor.x = 0.5;
	Face_spr.anchor.y = 0.5;
	Face_spr.x = this.x+FaceXPosition;
	Face_spr.y = this.y+FaceYPosition;
	
	return Face_spr;
}

//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////

SoR_GabWindow.prototype.DrawTextsXX = function(text, x, y, width, l, numl) {
	if(Imported.SoR_MessageExtension) this.Message_CurrentPageline = l;
	const textState = this.createTextState(text, x, y+6, width);
	if(Imported.SoR_MessageExtension) this.textRect_onMessage(textState.text, l);

	const vl = this.SoR_GabRubyarr[l]!=undefined ? this.SoR_GabRubyarr[l].length : 0;

	const rb_ypadd = numl==1? GabStringFont : GabStringFont/2;
	
	if(vl>0){ textState.y+= rb_ypadd-2}
	if(l==1){
		const vl2 = this.SoR_GabRubyarr[l-1]!=undefined ? this.SoR_GabRubyarr[l-1].length : 0;
		if(vl2>0){ textState.y+= GabStringFont/2+1;}
	}
	this.processAllText(textState);
	
	return textState.outputWidth;
}

SoR_GabWindow.prototype.processDrawIcon = function(iconIndex, textState) {
    if (textState.drawing) this.drawIconObjInd(iconIndex, textState.x + 2, textState.y + 2);
    textState.x += this.contents.fontSize + 4;
}

SoR_GabWindow.prototype.drawIconObjInd = function(iconIndex, x, y) {
    const bitmap = ImageManager.loadSystem("IconSet");
    const pw = ImageManager.iconWidth;
    const ph = ImageManager.iconHeight;
    const sx = (iconIndex % 16) * pw;
	const sy = Math.floor(iconIndex / 16) * ph;
	const drawSize = this.contents.fontSize;
    this.contents.blt(bitmap, sx, sy, pw, ph, x,y+2, drawSize, drawSize);
}

SoR_GabWindow.prototype.flushTextState = function(textState) {
    const text = textState.buffer;
    const rtl = textState.rtl;
    const tw = this.textWidth(text);
    const wdpd = this.width -this.padding*2; //////////
    const width = tw < wdpd ? tw: wdpd;     //////////
    const height = textState.height;

    const x = rtl ? textState.x - width : textState.x;
    const y = textState.y;
    if (textState.drawing) {
        this.contents.drawTextWithAlphaOutline(text, x, y, width, height);
    }
    textState.x += rtl ? -width : width;
    textState.buffer = this.createTextBuffer(rtl);
    const outputWidth = Math.abs(textState.x - textState.startX);
    if (textState.outputWidth < outputWidth) {
        textState.outputWidth = outputWidth;
    }
	textState.outputHeight = y - textState.startY + height;
	
	if(Imported.SoR_MessageExtension) this.flushrubytext(textState)
}

Bitmap.prototype.drawTextWithAlphaOutline = function(text, x, y, maxWidth, lineHeight, align) {
    const context = this.context;
    const alpha = context.globalAlpha;
    maxWidth = maxWidth || 0xffffffff;
    let tx = x;
    let ty = Math.round(y + lineHeight / 2 + this.fontSize * 0.35);
    if (align === "center") {
        tx += maxWidth / 2;
    }
    if (align === "right") {
        tx += maxWidth;
    }
    context.save();
    context.font = this._makeFontNameText();
    context.textAlign = align;
    context.textBaseline = "alphabetic";
    context.globalAlpha = alpha;
    this._drawTextOutline(text, tx, ty, maxWidth);
    this._drawTextBody(text, tx, ty, maxWidth);
    context.restore();
    this._baseTexture.update();
}





//For ruby
SoR_GabWindow.prototype.flushrubytext = function(textState) {
		if(this.SoR_GabRubyarr.length==0) return;
	
		if (textState.drawing) {
			const thisline = this.Message_CurrentPageline;
			const vl = this.SoR_GabRubyarr[thisline]? this.SoR_GabRubyarr[thisline].length : 0;
			
			for(let i = 0; i < vl; i++){
				const ruby = this.SoR_GabRubyarr[thisline][i];
				
					let LinePadd = GabStringFont-10;
					const rby = textState.y - LinePadd ;
					const tmpf = this.contents.fontSize;
					this.contents.fontSize = GabStringFont-7;
					
					const rbw = this.textWidth(ruby.rb);
					let rbx = textState.startX + ruby.xpos - rbw/2;
					this.contents.drawText(ruby.rb, rbx, rby, rbw, GabStringFont-4);
					this.contents.fontSize = tmpf;
			}
		}
}


//process ruby
SoR_GabWindow.prototype.textRect_onMessage = function(str_allmes, l) {
    const orig_fsize = this.contents.fontSize;
    let exp_fsize = orig_fsize;
    let next_fsize = orig_fsize;

    let FontChangeSplitting = false;
    let rubys = [];
 

    if(this.SoR_MesRubyarr.length!=0){ //make ruby
		rubys = this.SoR_MesRubyarr.slice();
        //words rb
    }

	const str_arr = str_allmes;
	
    let maxwidth = 0;
    let totalHeight = 0;
    let lenwidth = 0;
    //for(let row=0; row<str_arr.length;row++){
		let row = l;
            let str = RemoveEscapeCharactersForMeasurement(str_arr);

            let ckruby_line = [];
            for(let nrb=0; nrb<rubys.length;nrb++){//rb check
                let idx = str_arr.indexOf(rubys[nrb].words);
                while(idx!=-1){
                    ckruby_line.push({wd: rubys[nrb].words, rb: rubys[nrb].rb, idx: idx});
                    idx = str_arr.indexOf(rubys[nrb].words,idx+1);
                }
            }
            let ruby1line = [];


            let currentWidth = 0;
            let maxheight = 0;

            let head_idx = 0;
            for(let i=0; i<str.length;i++){
                const c = str[i];
                if (c.charCodeAt(0) < 0x20) { //Is control letter?
                    const code = str[i+1];
                    i++;
                    switch(code){
                        case "{":
                            next_fsize = MkFontBigger(exp_fsize);
                            FontChangeSplitting = true;
                            break;
                        case "}":
                            next_fsize = MkFontSmaller(exp_fsize);
                            FontChangeSplitting = true;
                            break;
                        default:
                        i++;
                        break;
                    }
                }


                //ruby setting
                for(let nrb=0; nrb < ckruby_line.length; nrb++){
                    if(i==ckruby_line[nrb].idx){
                        const idx = ckruby_line[nrb].idx;
                        const wdl = ckruby_line[nrb].wd.length;

                        const prevwd = str.substring(0, idx);
                        const prevarea = this.contents.measureTextRectArea(prevwd);
                        const rbbasewd = str.substring(idx, idx+wdl);
                        const rbbasearea = this.contents.measureTextRectArea(rbbasewd);
                        ruby1line.push({rb: ckruby_line[nrb].rb , xpos: prevarea.x+rbbasearea.x/2, idx: lenwidth+idx , idx2: lenwidth+idx+wdl});
                        break;
                    }
                }
                //font size scaling
                if(FontChangeSplitting === true || i>=str.length-1){
                    let pad_cntchar = -1;
					if(!FontChangeSplitting && i>=str.length-1) pad_cntchar = 0;//////
					
                    const test_arr = str.substring(head_idx, i+pad_cntchar);
                    this.contents.fontSize = exp_fsize;
                    const area = this.contents.measureTextRectArea(test_arr);
                    const rowtxwid = area.x;
                    const rowtxhgt = area.y;
                    if(maxheight < rowtxhgt) maxheight = rowtxhgt;//height

                    currentWidth += rowtxwid;
                    head_idx = i+1;
                    FontChangeSplitting = false;
                    exp_fsize = next_fsize;
                }
                
			}//for one row
 
			this.SoR_GabRubyarr[l] = ruby1line.length==0? undefined:ruby1line;
            if(ruby1line.length!=0) totalHeight+= (GabStringFont-6);//rubyfontsize

            this.CalcLineDrawWidth[row] = currentWidth;
            if(currentWidth > maxwidth) maxwidth = currentWidth;
            totalHeight += maxheight;
            lenwidth += str.length;

       // }//for entire rows

    this.contents.fontSize = orig_fsize;
    this.CalcLineDrawWidth[4] = {width: maxwidth, height: totalHeight, line: str_arr.length};
    return {width: maxwidth, height: totalHeight};
}

function RemoveEscapeCharactersForMeasurement(text){
	//Just remove escape characters to measure the length
		text = text.replace(/\x1bC\[(\d+)\]/gi, "");
		text = text.replace(/\x1bI\[(\d+)\]/gi, "あ");//count as a length for one character
		text = text.replace(/\x1bPX/gi, "");
		text = text.replace(/\x1bPY/gi, "");
		text = text.replace(/\x1bFS/gi, "");
	
		return text;
	}
	
	SoR.Gab = SoR_GabWindow;
}());